# Material System Known Limitations

## Issue: Small Brush Radius Material Artifacts

### Problem Description
When placing materials with a brush radius smaller than ~1.5 voxels, the material may not display correctly on all faces of the placed block. Symptoms include:
- Bottom/side faces showing terrain texture instead of placed material
- Rocky/stone color appearing at block edges
- Flat brown coloring on some faces

### Root Cause
The voxel-based material system has inherent limitations:

1. **Material Storage**: Materials are stored at discrete voxel positions in a 3D texture (33x33x33 per chunk)

2. **Marching Cubes Interpolation**: The terrain mesh is generated by marching cubes, which creates smooth surfaces that interpolate between voxels. Surface vertices exist at fractional positions between voxels.

3. **Boundary Mismatch**: When the shader samples materials for surface pixels, it may sample from voxels that are OUTSIDE the placed block's material coverage, especially at boundaries.

4. **Small Brush = Few Voxels**: With radius 0.5, only ~1-2 voxels get material assigned. The interpolated surface extends beyond this, causing sampling failures.

### Current Workaround
**Minimum brush radius of 1.5** is enforced for material placement. This ensures enough voxels have material to cover all faces the shader samples.

### Technical Details

**Material Radius Extension** (in `modify_density.glsl`):
```glsl
float material_radius = params.brush_pos.w + 0.49;  // Extended slightly beyond density radius
```

**Dual Voxel Sampling** (in `terrain.gdshader`):
```glsl
// Sample both direct voxel and biased (toward solid) voxel
vec3 voxel_pos1 = floor(local_vertex + vec3(0.5));  // Direct
vec3 biased_pos = local_vertex - normal * 0.7;       // Biased toward solid side
vec3 voxel_pos2 = floor(biased_pos + vec3(0.5));

// Use whichever has player material
int mat_id = (mat_id1 >= 100) ? mat_id1 : mat_id2;
```

### Potential Future Solutions

#### 1. **GPU-Side Vertex Material Assignment (Recommended)**

The most robust solution: assign materials during marching cubes mesh generation, not in the fragment shader.

**How it works:**
- In `marching_cubes.glsl`, when creating each triangle vertex:
  - Sample the material buffer at the SOLID voxel (the one with density < 0)
  - For edges between solid/air, always use the solid side's material
  - Encode material ID in vertex color alpha channel (or add custom vertex attribute)
- Fragment shader simply reads the vertex color - no texture sampling needed

**Pros:**
- ✅ Perfect accuracy - material assigned at geometry creation time
- ✅ Works for any brush size down to 1 voxel
- ✅ No coordinate mismatch possible
- ✅ Faster fragment shader (no texture lookups)

**Implementation steps:**
1. Modify `interpolate_vertex()` in `marching_cubes.glsl` to return material ID
2. For each edge, determine which corner is solid (density < 0) and use its material
3. Output material ID encoded in vertex color RGBA (e.g., A = mat_id/255.0)
4. Update `chunk_manager.gd` to pass material buffer to mesh shader (already done)
5. Simplify `terrain.gdshader` to read `COLOR.a` for material detection

---

#### 2. **Per-Triangle Material Storage**

Store one material per triangle instead of per-voxel.

**How it works:**
- During mesh generation, each triangle gets a material from its source voxel cube
- Store material ID per-triangle in a separate buffer or texture
- Fragment shader uses `gl_PrimitiveID` (or flat varying) to look up material

**Pros:**
- ✅ Exact material per face
- ✅ No interpolation issues

**Cons:**
- ❌ Requires geometry shader or primitive ID support
- ❌ More complex mesh format

---

#### 3. **Hybrid Block + Marching Cubes**

Use the existing building system for player placements, marching cubes only for procedural terrain.

**How it works:**
- Player material placement creates BLOCK geometry (like building system)
- These blocks exist as separate meshes overlaid on marching cubes terrain
- Each block has explicit material, no sampling needed

**Pros:**
- ✅ Perfect material accuracy at any size
- ✅ Already have the building system working
- ✅ No changes to marching cubes needed

**Cons:**
- ❌ Visual mismatch between smooth terrain and blocky placements
- ❌ Two rendering systems to maintain

---

#### 4. **Dual Contouring Algorithm**

Replace marching cubes with dual contouring, which has better sharp feature support.

**How it works:**
- Dual contouring generates vertices inside voxel cells, not on edges
- Each vertex inherently belongs to one voxel = one material
- Better at preserving sharp corners and edges

**Pros:**
- ✅ Natural material assignment per vertex
- ✅ Better visual quality for sharp features

**Cons:**
- ❌ Complete algorithm rewrite
- ❌ More complex implementation
- ❌ Different LOD challenges

---

#### 5. **Material Flood Fill During Placement**

Aggressively extend material to ALL solid voxels in contact during placement.

**How it works:**
- When placing, do a flood-fill from brush center
- Mark ALL connected solid voxels (not just within radius) with player material
- Stop at air boundaries

**Pros:**
- ✅ Covers all connected solid areas
- ✅ Works with current architecture

**Cons:**
- ❌ May spread further than intended
- ❌ GPU flood-fill is complex

---

## Recommended Approach for Rebuild

If starting fresh, implement **Solution #1 (GPU-Side Vertex Material Assignment)**:

1. Design material system FIRST before any other features
2. In `marching_cubes.glsl`, always sample material from the SOLID corner of each edge
3. Pass material ID through vertex colors to fragment shader
4. Test with 1x1x1 placements immediately - if they work, the system is sound
5. Only then add procedural terrain, biomes, roads, etc.

**Key principle:** Materials should be resolved at mesh creation time, never sampled in the fragment shader from a separate texture.

---

## Issue: Material Missing at Chunk Boundaries (FIXED)

### Problem Description
When placing materials near chunk boundaries, faces touching the boundary would show wrong texture (procedural terrain instead of placed material). The face geometry was generated correctly, but material sampling failed.

### Root Cause
1. **Chunk bounds calculation** used only the brush radius when determining which chunks need modification
2. The shader samples material from voxels using normal-biased positions
3. When placement was near boundary, the neighboring chunk didn't receive the material modification
4. The boundary face was rendered by the neighboring chunk which had `has_material_map = false`

### Solution (Implemented Dec 2024)
**Expanded chunk bounds calculation** in `chunk_manager.gd`:

```gdscript
func modify_terrain(pos: Vector3, radius: float, value: float, shape: int = 0, layer: int = 0, material_id: int = -1):
    # Add extra margin (1.0) to account for material radius extension and shader sampling
    var extra_margin = 1.0 if material_id >= 0 else 0.0
    var min_pos = pos - Vector3(radius + extra_margin, radius + extra_margin, radius + extra_margin)
    var max_pos = pos + Vector3(radius + extra_margin, radius + extra_margin, radius + extra_margin)
```

This ensures that when placing materials, neighboring chunks at boundaries also receive the modification, so their 3D material textures are properly updated.

### Debug Feature
Press **F9** to toggle chunk boundary visualization (red lines at X/Z boundaries, blue at Y).

### Future Optimization
The shader currently uses **4 texture samples** to detect player materials (direct, 0.8 inward, 1.5 inward, 2.5 inward). This was added during debugging. Now that the chunk bounds fix is in place, this could potentially be **reduced to 2 samples** (direct + one biased) for better performance.
