# -*- coding: utf-8 -*-
"""独立悬浮消息窗口 - Flet 应用（独立进程运行）- 美化版"""

import flet as ft
import json
import sys
import os
import threading
from datetime import datetime
from pathlib import Path

# 处理打包后的导入路径
if getattr(sys, 'frozen', False):
    # PyInstaller 打包后
    if hasattr(sys, '_MEIPASS'):
        sys.path.insert(0, sys._MEIPASS)
    try:
        from gui.flet_app.themes import theme
    except ImportError:
        # 如果导入失败，使用内置主题
        class _FallbackColors:
            bg_base = "#0d1117"
            bg_surface = "#161b22"
            bg_elevated = "#21262d"
            text_primary = "#f0f6fc"
            text_secondary = "#8b949e"
            text_muted = "#6e7681"
            accent = "#58a6ff"
            success = "#3fb950"
            error = "#f85149"
            border_default = "#30363d"
        class _FallbackTheme:
            colors = _FallbackColors()
        theme = _FallbackTheme()
else:
    # 开发环境
    ROOT_DIR = Path(__file__).parent.parent.parent.parent
    sys.path.insert(0, str(ROOT_DIR))
    from gui.flet_app.themes import theme


class FloatWindowApp:
    """独立悬浮窗口应用 - 美化版"""
    
    def __init__(self, page: ft.Page, msg_file: str, cmd_file: str):
        self.page = page
        self.msg_file = msg_file
        self.cmd_file = cmd_file
        self._msg_count = 0
        self._is_logged_in = False
        self._running = True  # 控制轮询线程
        self._setup_page()
        self._build_ui()
        self._start_polling()
        
        # 监听窗口关闭事件
        self.page.on_window_event = self._on_window_event
    
    def _setup_page(self):
        """设置窗口属性"""
        self.page.title = "直播消息"
        self.page.bgcolor = ft.Colors.TRANSPARENT
        self.page.padding = 0
        self.page.spacing = 0
        
        # 兼容新旧版本 flet API - 无边框模式
        if hasattr(self.page, 'window') and hasattr(self.page.window, 'width'):
            self.page.window.width = 340
            self.page.window.height = 450
            self.page.window.always_on_top = True
            self.page.window.frameless = True
            self.page.window.bgcolor = ft.Colors.TRANSPARENT
            self.page.window.left = 100
            self.page.window.top = 100
        else:
            self.page.window_width = 340
            self.page.window_height = 450
            self.page.window_always_on_top = True
            self.page.window_frameless = True
            self.page.window_bgcolor = ft.Colors.TRANSPARENT
            self.page.window_left = 100
            self.page.window_top = 100
    
    def _build_ui(self):
        """构建界面"""
        c = theme.colors
        
        # 标题栏内容 - 精简版（包含清空按钮）
        title_content = ft.Container(
            content=ft.Row([
                ft.Row([
                    ft.Icon(ft.Icons.CHAT_ROUNDED, color=c.accent, size=14),
                    ft.Text("直播消息", size=12, weight=ft.FontWeight.W_600, color=c.text_primary),
                ], spacing=4),
                ft.Row([
                    ft.TextButton(
                        "清空",
                        on_click=self._on_clear,
                        style=ft.ButtonStyle(color=c.text_muted, padding=ft.padding.all(4)),
                    ),
                    ft.IconButton(
                        icon=ft.Icons.REMOVE_ROUNDED,
                        icon_color=c.text_muted,
                        icon_size=14,
                        on_click=self._on_minimize,
                        width=24, height=24,
                    ),
                    ft.IconButton(
                        icon=ft.Icons.CLOSE_ROUNDED,
                        icon_color=c.error,
                        icon_size=14,
                        on_click=self._on_close,
                        width=24, height=24,
                    ),
                ], spacing=0),
            ], alignment=ft.MainAxisAlignment.SPACE_BETWEEN),
            padding=ft.padding.symmetric(horizontal=8, vertical=4),
            bgcolor=c.bg_surface,
        )
        
        # 使用 WindowDragArea 实现拖动
        title_bar = ft.WindowDragArea(
            content=title_content,
            maximizable=False,
        )
        
        # 消息列表 - 紧凑排版
        self.msg_list = ft.ListView(
            expand=True,
            spacing=2,
            auto_scroll=True,
            padding=ft.padding.symmetric(horizontal=8, vertical=4),
        )
        
        # 状态组件（简化）
        self.status_dot = ft.Container(width=6, height=6, bgcolor=c.text_muted, border_radius=3)
        self.status_text = ft.Text("未连接", size=10, color=c.text_muted)
        self.msg_count_text = ft.Text("0", size=10, color=c.text_muted)
        
        self._add_message("system", "系统", "独立窗口已启动，等待连接...")
        
        # 输入区域
        self.input_field = ft.TextField(
            hint_text="输入弹幕内容...",
            hint_style=ft.TextStyle(color=c.text_muted, size=12),
            border=ft.InputBorder.NONE,
            bgcolor=ft.Colors.TRANSPARENT,
            text_size=12,
            color=c.text_primary,  # 输入文字颜色
            cursor_color=c.accent,  # 光标颜色
            expand=True,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            on_submit=self._on_send,
            autofocus=True,
        )
        
        self.send_btn = ft.Container(
            content=ft.Icon(ft.Icons.SEND_ROUNDED, color="white", size=16),
            width=32, height=32,
            bgcolor=c.accent,
            border_radius=8,
            alignment=ft.alignment.center,
            on_click=self._on_send,
            ink=True,
        )
        
        input_row = ft.Container(
            content=ft.Row([
                ft.Container(
                    content=self.input_field,
                    expand=True,
                    bgcolor=c.bg_elevated,
                    border_radius=8,
                    border=ft.border.all(1, c.border_default),
                ),
                self.send_btn,
            ], spacing=8),
            padding=ft.padding.symmetric(horizontal=10, vertical=6),
            bgcolor=c.bg_surface,
        )
        
        
        # 右下角调整大小手柄
        resize_handle = ft.GestureDetector(
            content=ft.Container(
                content=ft.Icon(ft.Icons.DRAG_HANDLE, size=12, color=c.text_muted),
                width=20, height=20,
                alignment=ft.alignment.center,
            ),
            on_pan_update=self._on_resize,
            mouse_cursor=ft.MouseCursor.RESIZE_DOWN_RIGHT,
        )
        
        # 主容器 - 带圆角和阴影
        main_container = ft.Container(
            content=ft.Stack([
                ft.Column([
                    title_bar,
                    ft.Container(content=self.msg_list, expand=True, bgcolor=c.bg_base),
                    input_row,
                ], spacing=0, expand=True),
                # 调整大小手柄放在右下角
                ft.Container(
                    content=resize_handle,
                    right=2, bottom=2,
                ),
            ], expand=True),
            bgcolor=c.bg_base,
            border_radius=10,
            border=ft.border.all(1, c.border_default),
            shadow=ft.BoxShadow(
                spread_radius=0,
                blur_radius=15,
                color=ft.Colors.with_opacity(0.1, "black"),
                offset=ft.Offset(0, 2),
            ),
            clip_behavior=ft.ClipBehavior.ANTI_ALIAS,
        )
        
        self.page.add(
            ft.Container(content=main_container, expand=True, padding=8)
        )
    
    def _on_minimize(self, e=None):
        """最小化窗口"""
        if hasattr(self.page, 'window') and hasattr(self.page.window, 'minimized'):
            self.page.window.minimized = True
        else:
            self.page.window_minimized = True
        self.page.update()
    
    def _on_resize(self, e: ft.DragUpdateEvent):
        """调整窗口大小"""
        if hasattr(self.page, 'window') and hasattr(self.page.window, 'width'):
            new_width = max(280, self.page.window.width + e.delta_x)
            new_height = max(300, self.page.window.height + e.delta_y)
            self.page.window.width = new_width
            self.page.window.height = new_height
        else:
            new_width = max(280, self.page.window_width + e.delta_x)
            new_height = max(300, self.page.window_height + e.delta_y)
            self.page.window_width = new_width
            self.page.window_height = new_height
        self.page.update()
    
    def _add_message(self, msg_type: str, user: str, content: str):
        """添加消息 - 紧凑版无背景"""
        c = theme.colors
        
        type_colors = {
            "entrance": "#22c55e",
            "gift": "#f59e0b",
            "follow": "#ec4899",
            "chat": c.accent,
            "system": c.text_muted,
            "send": "#3b82f6",
        }
        
        color = type_colors.get(msg_type, c.accent)
        time_str = datetime.now().strftime("%H:%M")
        
        # 紧凑单行布局
        msg_item = ft.Container(
            content=ft.Row([
                ft.Text(user, size=11, color=color, weight=ft.FontWeight.W_600),
                ft.Text(content, size=11, color=c.text_secondary, expand=True),
                ft.Text(time_str, size=9, color=c.text_muted),
            ], spacing=6),
            padding=ft.padding.symmetric(horizontal=4, vertical=2),
        )
        
        self.msg_list.controls.append(msg_item)
        self._msg_count += 1
        self.msg_count_text.value = f"{self._msg_count} 条消息"
        self.page.update()
    
    def _on_send(self, e=None):
        """发送消息"""
        text = self.input_field.value.strip()
        if not text:
            return
        
        if not self._is_logged_in:
            self._add_message("system", "提示", "请先在主程序登录后再发送弹幕")
            return
        
        try:
            with open(self.cmd_file, 'w', encoding='utf-8') as f:
                json.dump({"type": "send", "text": text}, f)
            self._add_message("send", "我", text)
            self.input_field.value = ""
            self.page.update()
        except Exception as ex:
            self._add_message("system", "错误", f"发送失败: {ex}")
    
    def _on_clear(self, e=None):
        """清空消息"""
        self.msg_list.controls.clear()
        self._msg_count = 0
        self.msg_count_text.value = "0 条消息"
        self._add_message("system", "系统", "消息已清空")

    def _force_close(self):
        """强制关闭窗口 - 与主窗口相同的方式"""
        print("[FloatWindow] _force_close called", flush=True)
        self._running = False
        
        try:
            with open(self.cmd_file, 'w', encoding='utf-8') as f:
                json.dump({"type": "close"}, f)
        except:
            pass
        
        # 与主窗口相同：使用 window.close()
        try:
            if hasattr(self.page, 'window') and hasattr(self.page.window, 'close'):
                print("[FloatWindow] calling page.window.close()", flush=True)
                self.page.window.close()
            elif hasattr(self.page, 'window_close'):
                print("[FloatWindow] calling page.window_close()", flush=True)
                self.page.window_close()
        except Exception as ex:
            print(f"[FloatWindow] window.close error: {ex}", flush=True)
    
    def _on_close(self, e=None):
        """关闭窗口"""
        self._force_close()
    
    def _on_window_event(self, e):
        """窗口事件处理"""
        print(f"[FloatWindow] window event: {e.data}", flush=True)
        if e.data == "close":
            self._force_close()

    def _start_polling(self):
        """开始轮询消息文件"""
        import time
        
        def poll():
            last_id = 0
            while self._running:
                try:
                    if os.path.exists(self.msg_file):
                        with open(self.msg_file, 'r', encoding='utf-8') as f:
                            content = f.read()
                        if content:
                            data = json.loads(content)
                            msg_id = data.get("_id", 0)
                            # 只处理新消息（基于 _id）
                            if msg_id > last_id:
                                last_id = msg_id
                                if data.get("type") == "message":
                                    self._add_message(
                                        data.get("msg_type", "chat"),
                                        data.get("user", ""),
                                        data.get("content", "")
                                    )
                                elif data.get("type") == "status":
                                    self._update_status(
                                        data.get("status", ""),
                                        data.get("connected", False)
                                    )
                                elif data.get("type") == "send_enabled":
                                    self._set_send_enabled(data.get("enabled", False))
                except Exception as ex:
                    pass
                time.sleep(0.05)  # 更快的轮询
        
        threading.Thread(target=poll, daemon=True).start()
    
    def _update_status(self, status: str, connected: bool):
        """更新状态"""
        c = theme.colors
        self.status_dot.bgcolor = c.success if connected else c.text_muted
        self.status_text.value = status
        self.status_text.color = c.success if connected else c.text_muted
        self.page.update()
    
    def _set_send_enabled(self, enabled: bool):
        """设置登录状态（不禁用输入框，发送时检查）"""
        self._is_logged_in = enabled
        # 更新按钮样式提示状态
        c = theme.colors
        if enabled:
            self.send_btn.bgcolor = c.accent
        else:
            self.send_btn.bgcolor = c.text_muted
        self.page.update()


def main(page: ft.Page):
    # 从命令行参数获取文件路径
    msg_file = sys.argv[1] if len(sys.argv) > 1 else ""
    cmd_file = sys.argv[2] if len(sys.argv) > 2 else ""
    FloatWindowApp(page, msg_file, cmd_file)


if __name__ == "__main__":
    ft.app(target=main)
