# -*- coding: utf-8 -*-
"""标题栏组件"""

import flet as ft
from ..themes import theme

# 导入版本号
try:
    from ...license_manager import APP_VERSION
except ImportError:
    APP_VERSION = "1.0.0"


class TitleBar(ft.Container):
    """自定义标题栏"""
    
    def __init__(self, 
                 title: str = "梧桐抖音运营伴侣",
                 on_theme_toggle=None,
                 on_settings=None,
                 on_minimize=None,
                 on_close=None,
                 on_license_click=None,
                 on_logout=None,
                 on_profile_click=None,
                 show_dev_info: bool = True,
                 show_license: bool = True,
                 show_logout: bool = True,
                 show_profile: bool = True,
                 bgcolor: str = None):  # 自定义背景色
        c = theme.colors
        self._custom_bgcolor = bgcolor
        self._on_license_click = on_license_click
        self._on_logout = on_logout
        self._on_profile_click = on_profile_click
        
        # 主题切换按钮
        self.theme_btn = ft.IconButton(
            icon=ft.Icons.DARK_MODE if theme.is_dark else ft.Icons.LIGHT_MODE,
            icon_color=c.text_muted,
            icon_size=18,
            tooltip="切换主题",
            on_click=on_theme_toggle,
        )
        
        # 授权状态显示（可点击）
        self.license_icon = ft.Text("●", size=10, color=ft.Colors.GREY_500)
        self.license_status = ft.Text("未授权", size=11, color=ft.Colors.GREY_500)
        self.license_expire = ft.Text("", size=10, color=c.text_muted)
        
        self.license_container = ft.Container(
            content=ft.Row([
                self.license_icon,
                self.license_status,
                self.license_expire,
            ], spacing=4),
            bgcolor=c.bg_elevated,
            padding=ft.padding.symmetric(horizontal=10, vertical=4),
            border_radius=4,
            border=ft.border.all(1, c.border_default),
            on_click=self._handle_license_click,
            tooltip="点击管理授权",
            visible=show_license,
        )
        
        # 开发者信息
        dev_info_widgets = []
        if show_dev_info:
            dev_info_widgets = [
                ft.Text("客服QQ: 1622068165", size=11, color=c.text_muted),
            ]
        
        # 构建右侧按钮列表
        right_buttons = []
        if show_license:
            right_buttons.append(self.license_container)
            right_buttons.append(ft.VerticalDivider(width=1, color=c.border_default))
        right_buttons.extend(dev_info_widgets)
        right_buttons.append(self.theme_btn)
        if show_profile:
            right_buttons.append(ft.IconButton(
                icon=ft.Icons.ACCOUNT_CIRCLE_OUTLINED, 
                icon_color=c.text_muted, 
                icon_size=20, 
                tooltip="个人中心",
                on_click=self._handle_profile_click,
            ))
        if show_logout:
            right_buttons.append(ft.IconButton(
                icon=ft.Icons.LOGOUT, 
                icon_color=c.text_muted, 
                icon_size=18, 
                tooltip="退出登录",
                on_click=self._handle_logout,
            ))
        right_buttons.append(ft.IconButton(
            icon=ft.Icons.REMOVE, 
            icon_color=c.text_muted, 
            icon_size=18, 
            tooltip="最小化",
            on_click=on_minimize,
        ))
        right_buttons.append(ft.IconButton(
            icon=ft.Icons.CLOSE, 
            icon_color=c.error, 
            icon_size=18, 
            tooltip="关闭",
            on_click=on_close,
        ))
        
        # 使用自定义背景色或默认的 bg_surface
        bar_bgcolor = bgcolor if bgcolor is not None else c.bg_surface
        
        super().__init__(
            content=ft.Row([
                # 左侧标题
                ft.Row([
                    ft.Icon(ft.Icons.LIVE_TV_ROUNDED, color=c.accent, size=22),
                    ft.Text(title, size=15, weight=ft.FontWeight.W_600, color=c.text_primary),
                    ft.Container(
                        content=ft.Text(f"v{APP_VERSION}", size=10, color=c.accent),
                        bgcolor=c.accent_subtle,
                        padding=ft.padding.symmetric(horizontal=8, vertical=2),
                        border_radius=4,
                    ),
                ], spacing=10),
                
                # 右侧按钮
                ft.Row(right_buttons, spacing=8),
            ], alignment=ft.MainAxisAlignment.SPACE_BETWEEN),
            bgcolor=bar_bgcolor,
            padding=ft.padding.symmetric(horizontal=20, vertical=8),
            border=ft.border.only(bottom=ft.BorderSide(1, c.border_default)) if bgcolor is None else None,
        )
    
    def _handle_license_click(self, e):
        """处理授权点击"""
        if self._on_license_click:
            self._on_license_click(e)
    
    def _handle_logout(self, e):
        """处理退出登录"""
        if self._on_logout:
            self._on_logout(e)
    
    def _handle_profile_click(self, e):
        """处理个人中心点击"""
        if self._on_profile_click:
            self._on_profile_click(e)
    
    def update_license_status(self, is_valid: bool, license_type: str = "", expire_date: str = "", remaining_days: int = 0):
        """更新授权状态显示
        
        Args:
            is_valid: 是否有效授权
            license_type: 授权类型 (TRIAL/STANDARD/PRO)
            expire_date: 过期日期
            remaining_days: 剩余天数
        """
        c = theme.colors
        
        if is_valid:
            # 已授权
            self.license_icon.value = "✓"
            self.license_icon.color = ft.Colors.GREEN_400
            
            # 授权类型显示
            type_map = {"TRIAL": "试用", "STANDARD": "标准", "PRO": "专业"}
            type_text = type_map.get(license_type, license_type or "已授权")
            self.license_status.value = type_text
            self.license_status.color = ft.Colors.GREEN_400
            
            # 过期时间（始终显示）
            if expire_date:
                expire_display = expire_date[:10] if len(expire_date) > 10 else expire_date
                if remaining_days <= 7:
                    self.license_expire.value = f" | {expire_display} (剩余{remaining_days}天)"
                    self.license_expire.color = ft.Colors.ORANGE_400
                else:
                    self.license_expire.value = f" | 到期: {expire_display}"
                    self.license_expire.color = c.text_muted
            else:
                self.license_expire.value = ""
            
            self.license_container.bgcolor = "#1B3D1B" if theme.is_dark else "#E8F5E9"
            self.license_container.border = ft.border.all(1, ft.Colors.GREEN_900 if theme.is_dark else "#C8E6C9")
        else:
            # 未授权
            self.license_icon.value = "○"
            self.license_icon.color = ft.Colors.GREY_500
            self.license_status.value = "未授权"
            self.license_status.color = ft.Colors.GREY_500
            self.license_expire.value = ""
            self.license_container.bgcolor = c.bg_elevated
            self.license_container.border = ft.border.all(1, c.border_default)
        
        if self.page:
            self.update()
    
    def refresh_theme(self):
        """刷新主题"""
        c = theme.colors
        self.bgcolor = c.bg_surface
        self.border = ft.border.only(bottom=ft.BorderSide(1, c.border_default))
        self.theme_btn.icon = ft.Icons.DARK_MODE if theme.is_dark else ft.Icons.LIGHT_MODE
