# -*- coding: utf-8 -*-
"""
API 日志记录器 - 统一记录所有接口请求和响应
根据 config.yaml 中的 debug.log_api 配置决定是否记录
"""

import json
import yaml
import logging
from pathlib import Path
from datetime import datetime
from typing import Any, Dict, Optional

# 配置目录
ROOT_DIR = Path(__file__).parent.parent.parent.parent.absolute()
CONFIG_FILE = ROOT_DIR / "config" / "config.yaml"
USER_CONFIG_DIR = Path.home() / ".douyin_live_assistant"

# 全局配置缓存
_config: Optional[Dict] = None
_logger: Optional[logging.Logger] = None


def _load_config() -> Dict:
    """加载配置"""
    global _config
    if _config is not None:
        return _config
    
    _config = {
        'log_api': False,
        'api_log_file': './logs/api.log'
    }
    
    # 尝试从配置文件加载
    for config_path in [CONFIG_FILE, USER_CONFIG_DIR / "config.yaml"]:
        try:
            if config_path.exists():
                with open(config_path, 'r', encoding='utf-8') as f:
                    config = yaml.safe_load(f)
                    debug_config = config.get('debug', {})
                    _config['log_api'] = debug_config.get('log_api', False)
                    _config['api_log_file'] = debug_config.get('api_log_file', './logs/api.log')
                    break
        except Exception:
            pass
    
    return _config


def _get_logger() -> logging.Logger:
    """获取日志记录器"""
    global _logger
    if _logger is not None:
        return _logger
    
    config = _load_config()
    _logger = logging.getLogger('api_logger')
    _logger.setLevel(logging.DEBUG)
    
    # 清除已有的处理器
    _logger.handlers.clear()
    
    if config['log_api']:
        # 创建日志目录
        log_file = Path(config['api_log_file'])
        if not log_file.is_absolute():
            log_file = ROOT_DIR / log_file
        log_file.parent.mkdir(parents=True, exist_ok=True)
        
        # 文件处理器
        file_handler = logging.FileHandler(log_file, encoding='utf-8')
        file_handler.setLevel(logging.DEBUG)
        formatter = logging.Formatter(
            '%(asctime)s | %(levelname)s | %(message)s',
            datefmt='%Y-%m-%d %H:%M:%S'
        )
        file_handler.setFormatter(formatter)
        _logger.addHandler(file_handler)
    
    return _logger


def is_api_logging_enabled() -> bool:
    """检查是否启用 API 日志记录"""
    return _load_config()['log_api']


def log_request(method: str, url: str, headers: Dict = None, data: Any = None):
    """记录请求日志"""
    if not is_api_logging_enabled():
        return
    
    logger = _get_logger()
    
    # 构建日志内容
    log_parts = [
        f">>> REQUEST",
        f"Method: {method}",
        f"URL: {url}",
    ]
    
    if headers:
        # 隐藏敏感信息
        safe_headers = {k: ('***' if k.lower() in ['token', 'satoken', 'authorization'] else v) 
                        for k, v in headers.items()}
        log_parts.append(f"Headers: {json.dumps(safe_headers, ensure_ascii=False)}")
    
    if data:
        # 隐藏敏感字段
        if isinstance(data, dict):
            safe_data = data.copy()
            for key in ['password', 'token', 'secret', 'apiKey']:
                if key in safe_data:
                    safe_data[key] = '***'
            log_parts.append(f"Body: {json.dumps(safe_data, ensure_ascii=False)}")
        else:
            log_parts.append(f"Body: {data}")
    
    logger.info(" | ".join(log_parts))


def log_response(url: str, status_code: int = None, success: bool = None, 
                 data: Any = None, error: str = None, duration_ms: float = None):
    """记录响应日志"""
    if not is_api_logging_enabled():
        return
    
    logger = _get_logger()
    
    log_parts = [
        f"<<< RESPONSE",
        f"URL: {url}",
    ]
    
    if status_code is not None:
        log_parts.append(f"Status: {status_code}")
    
    if success is not None:
        log_parts.append(f"Success: {success}")
    
    if duration_ms is not None:
        log_parts.append(f"Duration: {duration_ms:.0f}ms")
    
    if error:
        log_parts.append(f"Error: {error}")
    elif data is not None:
        # 限制数据长度
        data_str = json.dumps(data, ensure_ascii=False) if isinstance(data, (dict, list)) else str(data)
        if len(data_str) > 500:
            data_str = data_str[:500] + "...(truncated)"
        log_parts.append(f"Data: {data_str}")
    
    logger.info(" | ".join(log_parts))


def reload_config():
    """重新加载配置（配置变更时调用）"""
    global _config, _logger
    _config = None
    _logger = None

