# -*- coding: utf-8 -*-
"""
认证服务 - 登录注册 API 调用
使用统一的 C 端登录接口（邮箱是唯一用户身份标识，一个用户可使用多个应用）
"""

import json
import base64
import time
import yaml
import sys
from pathlib import Path
from typing import Dict, Any, Optional, Tuple
import urllib.request
import urllib.error
import urllib.parse

from .api_logger import log_request, log_response

# 用户配置目录
USER_CONFIG_DIR = Path.home() / ".douyin_live_assistant"
AUTH_FILE = USER_CONFIG_DIR / "auth.json"

# 项目根目录
ROOT_DIR = Path(__file__).parent.parent.parent.parent.absolute()
if getattr(sys, 'frozen', False):
    ROOT_DIR = Path(sys._MEIPASS)

def _load_api_base_url() -> str:
    """从配置文件加载 API 基础地址（必须配置）"""
    config_paths = [
        ROOT_DIR / "config" / "config.yaml",
        USER_CONFIG_DIR / "config.yaml",
    ]
    
    for config_path in config_paths:
        try:
            if config_path.exists():
                with open(config_path, 'r', encoding='utf-8') as f:
                    config = yaml.safe_load(f)
                    server_config = config.get('server', {})
                    url = server_config.get('api_base_url')
                    if url:
                        return url.rstrip('/')  # 移除末尾斜杠
        except Exception as e:
            print(f"[Auth] 读取配置文件失败 {config_path}: {e}")
    
    raise ValueError(f"未找到 API 配置，请在 config/config.yaml 中配置 server.api_base_url，已检查路径: {config_paths}")


class AuthConfig:
    """认证配置"""
    # API 基础地址 - 从配置文件读取
    API_BASE_URL = _load_api_base_url()
    
    # 应用标识（与后台 BIZ_SOFTWARE_PRODUCT.SOFTWARE_CODE 一致）
    APP_CODE = "DOUYIN_LIVE_ASSISTANT"
    
    # 存储 Key
    STORAGE_KEY = {
        "TOKEN": "douyin_live_assistant_token",
        "USER_INFO": "douyin_live_assistant_user_info"
    }
    
    # 统一 C 端认证接口
    AUTH_ENDPOINTS = {
        "GET_PIC_CAPTCHA": "/auth/c/getPicCaptcha",
        "GET_EMAIL_CODE": "/auth/c/getEmailValidCode",
        "LOGIN_OR_REGISTER": "/auth/c/loginOrRegisterByEmail",
        "GET_LOGIN_USER": "/auth/c/getLoginUser",
        "UPDATE_USER_INFO": "/auth/c/updateUserInfo",
        "LOGOUT": "/auth/c/doLogout",
    }
    
    # 应用绑定接口（可选，用于统计）
    BIND_APP_ENDPOINT = "/wapp/user/bindApp"
    
    @classmethod
    def reload(cls):
        """重新加载配置"""
        cls.API_BASE_URL = _load_api_base_url()


class AuthService:
    """认证服务"""
    
    def __init__(self):
        self.token: Optional[str] = None
        self.user_info: Optional[Dict[str, Any]] = None
        # 确保配置目录存在
        USER_CONFIG_DIR.mkdir(parents=True, exist_ok=True)
        # 加载本地存储的认证信息
        self._load_auth()
    
    def _load_auth(self):
        """加载本地存储的认证信息"""
        try:
            if AUTH_FILE.exists():
                with open(AUTH_FILE, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                    self.token = data.get('token')
                    self.user_info = data.get('user_info')
        except Exception as e:
            print(f"[Auth] 加载认证信息失败: {e}")
    
    def _save_auth(self):
        """保存认证信息到本地"""
        try:
            data = {
                'token': self.token,
                'user_info': self.user_info
            }
            with open(AUTH_FILE, 'w', encoding='utf-8') as f:
                json.dump(data, f, ensure_ascii=False, indent=2)
        except Exception as e:
            print(f"[Auth] 保存认证信息失败: {e}")
    
    def _clear_auth(self):
        """清除本地认证信息"""
        self.token = None
        self.user_info = None
        try:
            if AUTH_FILE.exists():
                AUTH_FILE.unlink()
        except Exception as e:
            print(f"[Auth] 清除认证信息失败: {e}")
    
    def _api_request(self, url: str, method: str = "GET", data: Dict = None) -> Tuple[bool, Any]:
        """API 请求封装
        
        Returns:
            (success, data/error_message)
        """
        full_url = f"{AuthConfig.API_BASE_URL}{url}"
        
        headers = {
            'Content-Type': 'application/json',
        }
        if self.token:
            headers['token'] = self.token
        
        # 记录请求日志
        log_request(method, full_url, headers, data)
        start_time = time.time()
        
        try:
            if method == "GET" and data:
                query = urllib.parse.urlencode(data)
                full_url = f"{full_url}?{query}"
                req = urllib.request.Request(full_url, headers=headers)
            elif method == "POST":
                body = json.dumps(data).encode('utf-8') if data else None
                req = urllib.request.Request(full_url, data=body, headers=headers, method='POST')
            else:
                req = urllib.request.Request(full_url, headers=headers)
            
            with urllib.request.urlopen(req, timeout=30) as response:
                result = json.loads(response.read().decode('utf-8'))
                duration_ms = (time.time() - start_time) * 1000
                
                if result.get('code') == 200:
                    log_response(full_url, status_code=200, success=True, data=result.get('data'), duration_ms=duration_ms)
                    return True, result.get('data')
                else:
                    error_msg = result.get('msg', '请求失败')
                    log_response(full_url, status_code=result.get('code'), success=False, error=error_msg, duration_ms=duration_ms)
                    return False, error_msg
                    
        except urllib.error.HTTPError as e:
            duration_ms = (time.time() - start_time) * 1000
            error_msg = f"HTTP 错误: {e.code}"
            log_response(full_url, status_code=e.code, success=False, error=error_msg, duration_ms=duration_ms)
            return False, error_msg
        except urllib.error.URLError as e:
            duration_ms = (time.time() - start_time) * 1000
            error_msg = f"网络错误: {e.reason}"
            log_response(full_url, success=False, error=error_msg, duration_ms=duration_ms)
            return False, error_msg
        except Exception as e:
            duration_ms = (time.time() - start_time) * 1000
            error_msg = f"请求异常: {str(e)}"
            log_response(full_url, success=False, error=error_msg, duration_ms=duration_ms)
            return False, error_msg
    
    def is_logged_in(self) -> bool:
        """检查是否已登录"""
        return self.token is not None and self.user_info is not None
    
    def get_user_info(self) -> Optional[Dict[str, Any]]:
        """获取当前用户信息"""
        return self.user_info
    
    def get_pic_captcha(self) -> Tuple[bool, Any]:
        """获取图片验证码
        
        Returns:
            (success, {validCodeBase64, validCodeReqNo} / error_message)
        """
        return self._api_request(AuthConfig.AUTH_ENDPOINTS["GET_PIC_CAPTCHA"])
    
    def get_email_valid_code(self, email: str, valid_code: str, valid_code_req_no: str) -> Tuple[bool, Any]:
        """获取邮箱验证码
        
        Args:
            email: 邮箱地址
            valid_code: 图片验证码
            valid_code_req_no: 图片验证码请求号
            
        Returns:
            (success, email_valid_code_req_no / error_message)
        """
        params = {
            'email': email,
            'validCode': valid_code,
            'validCodeReqNo': valid_code_req_no
        }
        return self._api_request(AuthConfig.AUTH_ENDPOINTS["GET_EMAIL_CODE"], method='GET', data=params)
    
    def login_or_register(self, email: str, email_valid_code: str, email_valid_code_req_no: str) -> Tuple[bool, Any]:
        """邮箱验证码登录/注册（自动判断新用户或老用户）
        
        邮箱是唯一用户身份标识，如果用户已在其他应用注册过，可直接登录。
        
        Args:
            email: 邮箱地址
            email_valid_code: 邮箱验证码
            email_valid_code_req_no: 邮箱验证码请求号
            
        Returns:
            (success, user_info / error_message)
        """
        data = {
            'email': email,
            'emailValidCode': email_valid_code,
            'emailValidCodeReqNo': email_valid_code_req_no,
            'device': 'PC'
        }
        
        # 使用统一的 C 端登录/注册接口
        success, result = self._api_request(
            AuthConfig.AUTH_ENDPOINTS["LOGIN_OR_REGISTER"], 
            method='POST', 
            data=data
        )
        
        if success:
            # 统一接口返回的是 token 字符串
            self.token = result
            
            # 获取用户详细信息
            user_success, user_result = self._get_login_user()
            if user_success:
                self.user_info = {
                    'userId': user_result.get('id'),
                    'nickname': user_result.get('nickname') or user_result.get('name') or '',
                    'avatar': user_result.get('avatar') or '',
                    'phone': user_result.get('phone') or '',
                    'email': user_result.get('email') or user_result.get('account') or email,
                    'token': self.token
                }
                self._save_auth()
                
                # 绑定到当前应用（可选，用于统计）
                self._bind_to_app(self.user_info['userId'])
                
                return True, self.user_info
            else:
                return False, user_result
            
        return success, result
    
    def _get_login_user(self) -> Tuple[bool, Any]:
        """获取当前登录用户信息（内部方法）"""
        success, result = self._api_request(AuthConfig.AUTH_ENDPOINTS["GET_LOGIN_USER"])
        if success:
            print(f"[Auth] 用户信息字段: {list(result.keys()) if isinstance(result, dict) else type(result)}")
            print(f"[Auth] 用户信息: {result}")
        return success, result
    
    def _bind_to_app(self, user_id: str):
        """绑定用户到当前应用（可选，失败不影响登录）"""
        try:
            self._api_request(
                f"{AuthConfig.BIND_APP_ENDPOINT}?appCode={AuthConfig.APP_CODE}&userId={user_id}",
                method='POST'
            )
        except Exception as e:
            # 绑定失败不影响登录，可能是已经绑定过
            print(f"[Auth] 应用绑定: {e}")
    
    def logout(self) -> Tuple[bool, Any]:
        """退出登录"""
        # 调用统一的 C 端登出接口
        self._api_request(AuthConfig.AUTH_ENDPOINTS["LOGOUT"])
        # 清除本地认证信息
        self._clear_auth()
        return True, None
    
    def update_user_info(self, nickname: str = None, phone: str = None, avatar: str = None) -> Tuple[bool, Any]:
        """更新用户信息
        
        Args:
            nickname: 昵称
            phone: 手机号
            avatar: 头像URL
            
        Returns:
            (success, updated_user_info / error_message)
        """
        if not self.token:
            return False, "未登录"
        
        # 构建更新数据
        data = {}
        if nickname is not None:
            data['nickname'] = nickname
        if phone is not None:
            data['phone'] = phone
        if avatar is not None:
            data['avatar'] = avatar
        
        if not data:
            return False, "没有要更新的数据"
        
        success, result = self._api_request(
            AuthConfig.AUTH_ENDPOINTS["UPDATE_USER_INFO"],
            method='POST',
            data=data
        )
        
        if success:
            # 更新本地缓存
            if self.user_info:
                if nickname is not None:
                    self.user_info['nickname'] = nickname
                if phone is not None:
                    self.user_info['phone'] = phone
                if avatar is not None:
                    self.user_info['avatar'] = avatar
                self._save_auth()
            return True, self.user_info
        
        return False, result
    
    def check_auth(self) -> Tuple[bool, Any]:
        """检查认证状态（验证 token 是否有效）
        
        Returns:
            (is_valid, user_info / error_message)
        """
        if not self.token:
            return False, "未登录"
        
        # 使用统一的 C 端接口获取用户信息
        success, result = self._api_request(AuthConfig.AUTH_ENDPOINTS["GET_LOGIN_USER"])
        
        if success:
            self.user_info = {
                'userId': result.get('id'),
                'nickname': result.get('nickname') or result.get('name') or '',
                'avatar': result.get('avatar') or '',
                'phone': result.get('phone') or '',
                'email': result.get('email') or result.get('account') or '',
                'token': self.token
            }
            self._save_auth()
            return True, self.user_info
        else:
            # token 无效，清除本地认证
            self._clear_auth()
            return False, result


# 全局认证服务实例
_auth_service: Optional[AuthService] = None


def get_auth_service() -> AuthService:
    """获取认证服务实例（单例）"""
    global _auth_service
    if _auth_service is None:
        _auth_service = AuthService()
    return _auth_service
