# -*- coding: utf-8 -*-
"""推流服务管理"""

import sys
import threading
from pathlib import Path
from typing import Callable, Optional, Dict, Any
from dataclasses import dataclass

ROOT_DIR = Path(__file__).parent.parent.parent.parent
if getattr(sys, 'frozen', False):
    ROOT_DIR = Path(sys._MEIPASS)
sys.path.insert(0, str(ROOT_DIR))

# 尝试导入推流服务
def _log_import_error(msg: str):
    """记录导入错误到日志文件"""
    import sys
    from pathlib import Path
    from datetime import datetime
    try:
        if getattr(sys, 'frozen', False):
            log_dir = Path(sys.executable).parent / "logs"
        else:
            log_dir = Path(__file__).parent.parent.parent.parent / "logs"
        log_dir.mkdir(parents=True, exist_ok=True)
        log_file = log_dir / "import_errors.log"
        with open(log_file, 'a', encoding='utf-8') as f:
            f.write(f"[{datetime.now().strftime('%Y-%m-%d %H:%M:%S')}] {msg}\n")
    except:
        pass

try:
    from src.stream import StreamService, StreamServiceState, StreamServiceConfig
    from src.stream.packet_capture import Platform, NetworkInterface, StreamInfo
    STREAM_SERVICE_AVAILABLE = True
    print("[StreamService] stream module loaded OK")
except ImportError as e:
    STREAM_SERVICE_AVAILABLE = False
    import traceback
    error_msg = f"stream module load FAILED: {e}\n{traceback.format_exc()}"
    print(f"[StreamService] {error_msg}")
    _log_import_error(error_msg)
    StreamService = None
    StreamServiceState = None
    StreamServiceConfig = None
    Platform = None


@dataclass
class StreamStatus:
    """推流状态"""
    state: str = "idle"  # idle, starting, capturing, connected, error
    progress: int = 0
    message: str = ""
    stream_url: str = ""
    stream_key: str = ""
    room_id: str = ""
    web_rid: str = ""


class StreamServiceManager:
    """推流服务管理器"""
    
    def __init__(self):
        self._service: Optional[StreamService] = None
        self._thread: Optional[threading.Thread] = None
        self._running = False
        self._config: Dict[str, Any] = {}
        self.status = StreamStatus()
        self._obs_manager = None

        # 回调
        self._on_state_change: Optional[Callable] = None
        self._on_stream_captured: Optional[Callable] = None
        self._on_room_captured: Optional[Callable] = None
        self._on_error: Optional[Callable] = None
        self._on_progress: Optional[Callable] = None
        self._on_log: Optional[Callable] = None  # 新增：日志回调

    @property
    def is_available(self) -> bool:
        return STREAM_SERVICE_AVAILABLE

    def on_state_change(self, callback: Callable[[StreamStatus], None]):
        self._on_state_change = callback

    def on_stream_captured(self, callback: Callable[[str, str], None]):
        """stream_url, stream_key"""
        self._on_stream_captured = callback

    def on_room_captured(self, callback: Callable[[str, str], None]):
        """room_id, web_rid"""
        self._on_room_captured = callback

    def on_error(self, callback: Callable[[str], None]):
        self._on_error = callback

    def on_progress(self, callback: Callable[[int], None]):
        self._on_progress = callback

    def on_log(self, callback: Callable[[str], None]):
        """注册日志回调"""
        self._on_log = callback
    
    def start(self, config: Dict[str, Any] = None):
        """启动一键开播"""
        if not STREAM_SERVICE_AVAILABLE:
            self._notify_error("推流服务模块未加载")
            return False
        
        if self._running:
            return False
        
        self._config = config or {}
        self._thread = threading.Thread(target=self._run, daemon=True)
        self._thread.start()
        return True
    
    def stop(self):
        """停止服务"""
        self._running = False
        if self._service:
            try:
                self._service.stop()
            except:
                pass
        self._service = None
    
    def _run(self):
        """运行推流服务"""
        self._running = True
        self.status.state = "starting"
        self._notify_state()

        try:
            # 创建服务配置
            service_config = StreamServiceConfig(
                auto_start_companion=self._config.get('auto_start_companion', True),
                auto_capture=True,
                auto_connect=True,
                capture_timeout=self._config.get('capture_timeout', 300),
                network_interface=self._config.get('network_interface'),
                target_platform=Platform.DOUYIN if Platform else None,
                companion_path=self._config.get('companion_path'),
            )

            # 创建服务
            self._service = StreamService(service_config)

            # 注册回调
            self._service.on('state_changed', self._handle_state_changed)
            self._service.on('stream_captured', self._handle_stream_captured)
            self._service.on('room_id_captured', self._handle_room_captured)
            self._service.on('error', self._handle_error)
            self._service.on('progress', self._handle_progress)

            # 新增：注册抓包日志回调
            if hasattr(self._service, '_capture') and self._service._capture:
                self._service._capture._on_log = self._handle_log

            # 启动一键开播
            self._service.start_one_click()

            # 等待完成
            while self._running:
                if self._service:
                    status = self._service.get_status()
                    if status.state in [StreamServiceState.CONNECTED, StreamServiceState.ERROR]:
                        break
                import time
                time.sleep(0.5)

        except Exception as e:
            self._notify_error(f"推流服务异常: {e}")
        finally:
            self._running = False
    
    def _handle_state_changed(self, status):
        """处理状态变化"""
        state_map = {
            StreamServiceState.IDLE: "idle",
            StreamServiceState.STARTING: "starting",
            StreamServiceState.CAPTURING: "capturing",
            StreamServiceState.CONNECTED: "connected",
            StreamServiceState.ERROR: "error",
        }
        self.status.state = state_map.get(status.state, "idle")
        self.status.message = status.message if hasattr(status, 'message') else ""
        self._notify_state()
    
    def _handle_stream_captured(self, stream_info):
        """处理推流信息捕获"""
        # 兼容不同的属性名：server_url/url, stream_key/key
        self.status.stream_url = getattr(stream_info, 'server_url', '') or getattr(stream_info, 'url', '') or ""
        self.status.stream_key = getattr(stream_info, 'stream_key', '') or getattr(stream_info, 'key', '') or ""
        
        print(f"[StreamService] 推流码捕获: url={self.status.stream_url[:50]}, key={self.status.stream_key[:50]}")
        
        # 捕获成功后更新状态为 connected
        if self.status.stream_url and self.status.stream_key:
            self.status.state = "connected"
            self._notify_state()
        
        if self._on_stream_captured:
            self._on_stream_captured(self.status.stream_url, self.status.stream_key)
    
    def _handle_room_captured(self, room_id, web_rid):
        """处理房间 ID 捕获"""
        self.status.room_id = room_id
        self.status.web_rid = web_rid
        if self._on_room_captured:
            self._on_room_captured(room_id, web_rid)
    
    def _handle_error(self, error):
        """处理错误"""
        self.status.state = "error"
        self.status.message = str(error)
        self._notify_error(str(error))
    
    def _handle_progress(self, progress):
        """处理进度"""
        self.status.progress = progress
        if self._on_progress:
            self._on_progress(progress)

    def _handle_log(self, msg: str):
        """处理抓包日志"""
        if self._on_log:
            self._on_log(msg)

    def _notify_state(self):
        if self._on_state_change:
            self._on_state_change(self.status)
    
    def _notify_error(self, msg: str):
        self.status.state = "error"
        self.status.message = msg
        if self._on_error:
            self._on_error(msg)
    
    # ========== OBS 控制 ==========
    
    def push_to_obs(self, stream_url: str = None, stream_key: str = None, 
                    on_progress: Callable[[str], None] = None) -> bool:
        """推送到 OBS - 完全自动化"""
        url = stream_url or self.status.stream_url
        key = stream_key or self.status.stream_key
        
        # 必须有真实推流信息才能推送
        if not url or not key:
            if on_progress:
                on_progress("❌ 请先抓取推流信息")
            return False
        
        try:
            from src.stream import ObsManager
            
            def log(msg):
                print(f"[OBS] {msg}")
                if on_progress:
                    on_progress(msg)
            
            # 创建 OBS 管理器
            obs_manager = ObsManager(logger=log)
            
            # 检查 OBS 是否安装
            if not obs_manager.is_installed:
                self._notify_error("OBS 未安装")
                return False
            
            # 调用完全自动化推流（与原系统一致）
            success, message = obs_manager.full_auto_stream(url, key, on_progress=log)
            
            if success:
                self._obs_manager = obs_manager
                return True
            else:
                self._notify_error(f"推流失败: {message}")
                return False
                
        except ImportError as e:
            self._notify_error(f"OBS 模块未加载: {e}")
            return False
        except Exception as e:
            self._notify_error(f"OBS 推送异常: {e}")
            return False
