# -*- coding: utf-8 -*-
"""AI Tab"""

import flet as ft
from typing import Callable, Optional
from ..themes import theme
from ..components import Card

# AI 服务商配置
AI_PROVIDERS = {
    "deepseek": {"name": "DeepSeek", "models": ["deepseek-chat", "deepseek-coder"]},
    "dashscope": {"name": "阿里百炼", "models": ["qwen-turbo", "qwen-plus", "qwen-max", "qwen-long"]},
    "openai": {"name": "OpenAI", "models": ["gpt-4o-mini", "gpt-4o", "gpt-4-turbo", "gpt-3.5-turbo"]},
    "custom": {"name": "自定义", "models": []},
}

WARM_CATEGORIES = [
    ("random", "随机"),
    ("joke", "讲笑话"),
    ("philosophy", "富有哲理"),
    ("warm", "温暖治愈"),
    ("trivia", "冷知识"),
]


class AITab(ft.Container):
    """AI Tab"""
    
    def __init__(self, on_test: Callable = None, on_config_change: Callable = None, on_save: Callable = None, config_manager = None):
        self._on_test = on_test
        self._on_config_change = on_config_change
        self._on_save = on_save
        self.config_manager = config_manager
        self._original_ai_config = {}
        self._original_reply_config = {}
        self._original_warm_config = {}
        
        c = theme.colors
        
        # ========== AI 启用 ==========
        self.ai_enable_check = ft.Checkbox(
            label="启用 AI 智能对话",
            value=False,
            on_change=self._notify_change,
        )
        
        # ========== 服务商选择 ==========
        self.provider_dropdown = ft.Dropdown(
            options=[ft.dropdown.Option(k, v["name"]) for k, v in AI_PROVIDERS.items()],
            value="deepseek",
            width=180,
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            on_change=self._on_provider_change,
        )
        
        # ========== 模型选择 ==========
        self.model_dropdown = ft.Dropdown(
            options=[ft.dropdown.Option(m) for m in AI_PROVIDERS["deepseek"]["models"]],
            value="deepseek-chat",
            width=180,
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            on_change=self._notify_change,
        )
        
        # ========== API Key ==========
        self.api_key_input = ft.TextField(
            hint_text="输入 API Key",
            password=True,
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            expand=True,
            on_change=self._notify_change,
        )
        
        self.test_btn = ft.ElevatedButton(
            "测试",
            bgcolor=c.accent,
            color="#ffffff",
            on_click=self._handle_test,
            style=ft.ButtonStyle(shape=ft.RoundedRectangleBorder(radius=6)),
        )
        
        self.test_status = ft.Text("", size=12, color=c.text_muted)
        
        # ========== 提示词 ==========
        self.prompt_input = ft.TextField(
            hint_text="你是一个有趣的直播助手...",
            multiline=True,
            min_lines=3,
            max_lines=5,
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            on_change=self._notify_change,
        )
        
        # ========== AI 大脑卡片 ==========
        self.ai_card = Card("AI 大脑", ft.Column([
            self.ai_enable_check,
            ft.Row([
                ft.Text("服务商:", size=13, color=c.text_muted, width=70),
                self.provider_dropdown,
            ], spacing=10),
            ft.Row([
                ft.Text("模型:", size=13, color=c.text_muted, width=70),
                self.model_dropdown,
            ], spacing=10),
            ft.Text("API Key:", size=13, color=c.text_muted),
            ft.Row([
                self.api_key_input,
                self.test_btn,
            ], spacing=10),
            self.test_status,
            ft.Text("人设提示词:", size=13, color=c.text_muted),
            self.prompt_input,
        ], spacing=12), on_save=self._save_ai_config, show_save_button=True)
        
        # ========== 回复模式 ==========
        self.reply_mode_dropdown = ft.Dropdown(
            options=[
                ft.dropdown.Option("tts", "语音播报"),
                ft.dropdown.Option("danmu", "发送弹幕"),
                ft.dropdown.Option("both", "语音+弹幕"),
            ],
            value="tts",
            width=150,
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            on_change=self._notify_change,
        )
        
        self.reply_rate_slider = ft.Slider(
            min=0, max=100, value=100, divisions=10, label="{value}%",
            on_change=self._notify_change,
            width=200,
        )
        
        self.reply_card = Card("回复设置", ft.Column([
            ft.Row([
                ft.Text("回复模式:", size=13, color=c.text_muted, width=80),
                self.reply_mode_dropdown,
            ], spacing=10),
            ft.Row([
                ft.Text("回复率:", size=13, color=c.text_muted, width=80),
                self.reply_rate_slider,
            ], spacing=10),
        ], spacing=12), show_save_button=True, on_save=self._save_ai_config)
        
        # ========== 暖场功能 ==========
        self.warm_check = ft.Checkbox(
            label="启用暖场 (无互动时AI主动说话)",
            value=False,
            on_change=self._notify_change,
        )
        
        self.warm_category_dropdown = ft.Dropdown(
            options=[ft.dropdown.Option(k, v) for k, v in WARM_CATEGORIES],
            value="random",
            width=150,
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            on_change=self._notify_change,
        )
        
        self.warm_interval_input = ft.TextField(
            value="60",
            suffix_text="秒",
            width=100,
            border_color=c.border_default,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            on_change=self._notify_change,
        )
        
        self.warm_card = Card("暖场功能", ft.Column([
            self.warm_check,
            ft.Row([
                ft.Text("暖场类型:", size=13, color=c.text_muted, width=80),
                self.warm_category_dropdown,
            ], spacing=10),
            ft.Row([
                ft.Text("静默间隔:", size=13, color=c.text_muted, width=80),
                self.warm_interval_input,
            ], spacing=10),
        ], spacing=12), show_save_button=True, on_save=self._save_ai_config)
        
        # ========== 主容器 ==========
        super().__init__(
            content=ft.Column([self.ai_card, self.reply_card, self.warm_card], spacing=15, scroll=ft.ScrollMode.AUTO, expand=True),
            padding=15,
            expand=True,
        )
    
    def _on_provider_change(self, e):
        """服务商切换"""
        provider = e.control.value
        models = AI_PROVIDERS.get(provider, {}).get("models", [])
        
        self.model_dropdown.options = [ft.dropdown.Option(m) for m in models]
        if models:
            self.model_dropdown.value = models[0]
        else:
            self.model_dropdown.value = None
        
        self._notify_change(e)
    
    def _handle_test(self, e):
        """测试连接"""
        if self._on_test:
            self.test_btn.disabled = True
            self.test_btn.text = "测试中..."
            self.test_status.value = ""
            self._on_test(
                self.provider_dropdown.value,
                self.model_dropdown.value,
                self.api_key_input.value
            )
    
    def set_test_result(self, success: bool, message: str):
        """设置测试结果"""
        c = theme.colors
        self.test_btn.disabled = False
        self.test_btn.text = "测试"
        self.test_status.value = message
        self.test_status.color = c.success if success else c.error
    
    def _notify_change(self, e=None):
        """通知配置变化并更新待保存状态 - 不立即更新 config_manager"""
        # 分别检测每个卡片的变化
        ai_config = self._get_ai_card_config()
        reply_config = self._get_reply_card_config()
        warm_config = self._get_warm_card_config()
        
        self.ai_card.set_has_changes(ai_config != self._original_ai_config)
        self.reply_card.set_has_changes(reply_config != self._original_reply_config)
        self.warm_card.set_has_changes(warm_config != self._original_warm_config)
        
        # 注意：不再调用 _on_config_change，配置只在保存时才生效
    
    def _get_ai_card_config(self) -> dict:
        """获取 AI 大脑卡片配置（用于变更检测）"""
        return {
            "enabled": self.ai_enable_check.value,
            "provider": self.provider_dropdown.value,
            "model": self.model_dropdown.value,
            "api_key": self.api_key_input.value or "",
            "prompt": self.prompt_input.value or "",
        }
    
    def _get_reply_card_config(self) -> dict:
        """获取回复设置卡片配置（用于变更检测）"""
        return {
            "reply_mode": self.reply_mode_dropdown.value,
            "reply_rate": int(self.reply_rate_slider.value),
        }
    
    def _get_warm_card_config(self) -> dict:
        """获取暖场功能卡片配置（用于变更检测）"""
        return {
            "warm_enabled": self.warm_check.value,
            "warm_category": self.warm_category_dropdown.value,
            "warm_interval": self.warm_interval_input.value or "60",
        }
    
    def _save_ai_config(self, e=None):
        """保存 AI 配置"""
        if self.config_manager:
            config = self.get_config()["ai"]
            self.config_manager.set_ai_enabled(config["enabled"])
            self.config_manager.set_ai_provider(config["provider"])
            self.config_manager.set_ai_model(config["model"])
            self.config_manager.set_ai_api_key(config["api_key"])
            self.config_manager.set_ai_prompt(config["system_prompt"])
            self.config_manager.set_ai_reply_mode(config["reply_mode"])
            self.config_manager.set_ai_reply_rate(config["reply_rate"])
            self.config_manager.set_warm_enabled(config["warm_enabled"])
            self.config_manager.set_warm_category(config["warm_category"])
            self.config_manager.set_warm_interval(config["warm_interval"])
            self.config_manager.save()
        
        self._original_ai_config = self._get_ai_card_config()
        self._original_reply_config = self._get_reply_card_config()
        self._original_warm_config = self._get_warm_card_config()
        self.ai_card.set_has_changes(False)
        self.reply_card.set_has_changes(False)
        self.warm_card.set_has_changes(False)
        
        self._show_save_dialog(True, "AI 配置已保存")
        
        if self._on_save:
            self._on_save(self.get_config())
    
    def _show_save_dialog(self, success: bool, message: str):
        """显示保存结果弹窗"""
        def close_dialog(e):
            dialog.open = False
            if self.page:
                self.page.update()
        
        title = "保存成功" if success else "保存失败"
        btn_color = "#4CAF50" if success else "#E53935"
        
        dialog = ft.AlertDialog(
            modal=True,
            title=ft.Text(title, weight=ft.FontWeight.BOLD),
            content=ft.Text(message, size=14),
            actions=[
                ft.ElevatedButton("确定", bgcolor=btn_color, color="#ffffff", on_click=close_dialog),
            ],
            actions_alignment=ft.MainAxisAlignment.END,
        )
        
        if self.page:
            self.page.overlay.append(dialog)
            dialog.open = True
            self.page.update()
    
    def get_config(self) -> dict:
        """获取当前配置"""
        return {
            "ai": {
                "enabled": self.ai_enable_check.value,
                "provider": self.provider_dropdown.value,
                "model": self.model_dropdown.value,
                "api_key": self.api_key_input.value,
                "system_prompt": self.prompt_input.value,
                "reply_mode": self.reply_mode_dropdown.value,
                "reply_rate": self.reply_rate_slider.value / 100.0,
                "warm_enabled": self.warm_check.value,
                "warm_category": self.warm_category_dropdown.value,
                "warm_interval": int(self.warm_interval_input.value or 60),
            }
        }
    
    def load_config(self, config_manager):
        """从配置加载数据"""
        self.config_manager = config_manager
        
        self.ai_enable_check.value = config_manager.get_ai_enabled()
        self.provider_dropdown.value = config_manager.get_ai_provider()
        
        # 更新模型列表
        provider = self.provider_dropdown.value
        models = AI_PROVIDERS.get(provider, {}).get("models", [])
        self.model_dropdown.options = [ft.dropdown.Option(m) for m in models]
        
        model = config_manager.get_ai_model()
        if model in models:
            self.model_dropdown.value = model
        elif models:
            self.model_dropdown.value = models[0]
        
        self.api_key_input.value = config_manager.get_ai_api_key()
        self.prompt_input.value = config_manager.get_ai_prompt()
        
        mode = config_manager.get_ai_reply_mode()
        self.reply_mode_dropdown.value = mode
        self.reply_rate_slider.value = config_manager.get_ai_reply_rate() * 100
        
        self.warm_check.value = config_manager.get_warm_enabled()
        self.warm_category_dropdown.value = config_manager.get_warm_category()
        self.warm_interval_input.value = str(config_manager.get_warm_interval())
        
        # 保存原始配置用于变更检测
        self._original_ai_config = self._get_ai_card_config()
        self._original_reply_config = self._get_reply_card_config()
        self._original_warm_config = self._get_warm_card_config()
        self.ai_card.set_has_changes(False)
        self.reply_card.set_has_changes(False)
        self.warm_card.set_has_changes(False)
