# -*- coding: utf-8 -*-
"""直播间数据分析 Tab"""

import flet as ft
from typing import Optional, Callable
from ..themes import theme
from ..components import Card
from src.analytics.collector import LiveDataCollector
from src.analytics.stats import LiveStats
from src.analytics.session import SessionManager

# 统一字号定义
FONT_SIZE_TITLE = 13      # 模块标题
FONT_SIZE_VALUE = 18      # 数值显示
FONT_SIZE_LABEL = 12      # 标签文字
FONT_SIZE_ITEM = 12       # 列表项
FONT_SIZE_SMALL = 11      # 小字


class AnalyticsTab(ft.Container):
    """数据分析 Tab"""
    
    # 功能标识，用于授权检查
    FEATURE_KEY = "analytics"
    
    def __init__(self, collector: LiveDataCollector = None, check_feature: Callable = None):
        self._collector = collector or LiveDataCollector()
        self._check_feature = check_feature  # 授权检查回调
        self._is_authorized = True  # 是否已授权
        c = theme.colors
        
        # ========== 汇总卡片 ==========
        self.duration_text = ft.Text("0秒", size=FONT_SIZE_VALUE, weight=ft.FontWeight.BOLD, color=c.text_primary)
        self.online_text = ft.Text("0", size=FONT_SIZE_VALUE, weight=ft.FontWeight.BOLD, color=c.accent)
        self.peak_text = ft.Text("峰值: 0", size=FONT_SIZE_SMALL, color=c.text_muted)
        self.entrance_text = ft.Text("0", size=FONT_SIZE_VALUE, weight=ft.FontWeight.BOLD, color=c.success)
        self.follow_text = ft.Text("0", size=FONT_SIZE_VALUE, weight=ft.FontWeight.BOLD, color=ft.Colors.PINK_400)
        self.chat_text = ft.Text("0", size=FONT_SIZE_VALUE, weight=ft.FontWeight.BOLD, color=ft.Colors.BLUE_400)
        self.gift_text = ft.Text("0", size=FONT_SIZE_VALUE, weight=ft.FontWeight.BOLD, color=ft.Colors.ORANGE_400)
        self.heat_text = ft.Text("😴 冷清", size=FONT_SIZE_LABEL, color=c.text_muted)
        
        # 汇总面板
        summary_panel = ft.Container(
            content=ft.Row([
                self._stat_card("⏱️ 时长", self.duration_text),
                self._stat_card("👥 在线", self.online_text, self.peak_text),
                self._stat_card("🚪 进场", self.entrance_text),
                self._stat_card("➕ 关注", self.follow_text),
                self._stat_card("💬 弹幕", self.chat_text),
                self._stat_card("🎁 礼物", self.gift_text),
            ], alignment=ft.MainAxisAlignment.SPACE_EVENLY),
            padding=ft.padding.symmetric(horizontal=10, vertical=15),
            bgcolor=c.bg_surface,
            border_radius=8,
            border=ft.border.all(1, c.border_default),
        )
        
        # ========== 排行榜区域 ==========
        self.chat_rank_list = ft.Column([], spacing=4, scroll=ft.ScrollMode.AUTO)
        self.gift_rank_list = ft.Column([], spacing=4, scroll=ft.ScrollMode.AUTO)
        
        rank_panel = ft.Row([
            # 发言排行
            ft.Container(
                content=ft.Column([
                    ft.Text("💬 发言排行", size=FONT_SIZE_TITLE, weight=ft.FontWeight.BOLD, color=c.text_primary),
                    ft.Container(
                        content=self.chat_rank_list,
                        height=180,
                        padding=ft.padding.only(top=8),
                    ),
                ], spacing=0),
                bgcolor=c.bg_surface,
                padding=15,
                border_radius=8,
                border=ft.border.all(1, c.border_default),
                expand=True,
            ),
            # 礼物排行
            ft.Container(
                content=ft.Column([
                    ft.Text("🎁 礼物排行", size=FONT_SIZE_TITLE, weight=ft.FontWeight.BOLD, color=c.text_primary),
                    ft.Container(
                        content=self.gift_rank_list,
                        height=180,
                        padding=ft.padding.only(top=8),
                    ),
                ], spacing=0),
                bgcolor=c.bg_surface,
                padding=15,
                border_radius=8,
                border=ft.border.all(1, c.border_default),
                expand=True,
            ),
        ], spacing=15)
        
        # ========== 最近礼物 ==========
        self.recent_gift_list = ft.Column([], spacing=4, scroll=ft.ScrollMode.AUTO)
        
        recent_gift_panel = ft.Container(
            content=ft.Column([
                ft.Text("🎁 最近礼物", size=FONT_SIZE_TITLE, weight=ft.FontWeight.BOLD, color=c.text_primary),
                ft.Container(
                    content=self.recent_gift_list,
                    height=120,
                    padding=ft.padding.only(top=8),
                ),
            ], spacing=0),
            bgcolor=c.bg_surface,
            padding=15,
            border_radius=8,
            border=ft.border.all(1, c.border_default),
            expand=True,
        )
        
        # ========== 热度指示 ==========
        self.heat_bar = ft.ProgressBar(value=0, bgcolor=c.bg_elevated, color=c.accent, height=8)
        
        heat_panel = ft.Container(
            content=ft.Row([
                ft.Text("🔥 直播热度", size=FONT_SIZE_TITLE, weight=ft.FontWeight.BOLD, color=c.text_primary, width=100),
                ft.Container(content=self.heat_bar, expand=True, padding=ft.padding.symmetric(horizontal=10)),
                ft.Container(content=self.heat_text, width=80, alignment=ft.alignment.center_right),
            ], alignment=ft.MainAxisAlignment.START, vertical_alignment=ft.CrossAxisAlignment.CENTER),
            bgcolor=c.bg_surface,
            padding=ft.padding.symmetric(horizontal=15, vertical=12),
            border_radius=8,
            border=ft.border.all(1, c.border_default),
        )
        
        # ========== 历史场次 ==========
        self.session_list = ft.Column([], spacing=4, scroll=ft.ScrollMode.AUTO)
        
        session_panel = ft.Container(
            content=ft.Column([
                ft.Row([
                    ft.Text("📅 历史场次", size=FONT_SIZE_TITLE, weight=ft.FontWeight.BOLD, color=c.text_primary),
                    ft.TextButton("刷新", on_click=self._refresh_sessions, style=ft.ButtonStyle(padding=0)),
                ], alignment=ft.MainAxisAlignment.SPACE_BETWEEN),
                ft.Container(
                    content=self.session_list,
                    height=120,
                    padding=ft.padding.only(top=8),
                ),
            ], spacing=0),
            bgcolor=c.bg_surface,
            padding=15,
            border_radius=8,
            border=ft.border.all(1, c.border_default),
            expand=True,
        )
        
        # ========== 主布局 ==========
        super().__init__(
            content=ft.Column([
                summary_panel,
                heat_panel,
                rank_panel,
                ft.Row([recent_gift_panel, session_panel], spacing=15),
            ], spacing=15, scroll=ft.ScrollMode.AUTO),
            padding=20,
            expand=True,
        )
        
        # 设置数据更新回调
        self._collector.set_update_callback(self._on_data_update)
    
    def _stat_card(self, title: str, value_text: ft.Text, sub_text: ft.Text = None) -> ft.Container:
        """创建统计卡片"""
        c = theme.colors
        
        content_items = [
            ft.Text(title, size=FONT_SIZE_SMALL, color=c.text_muted),
            value_text,
        ]
        if sub_text:
            content_items.append(sub_text)
        
        return ft.Container(
            content=ft.Column(content_items, spacing=4, horizontal_alignment=ft.CrossAxisAlignment.CENTER),
            padding=ft.padding.symmetric(horizontal=8, vertical=10),
            width=95,
        )
    
    def _rank_item(self, rank: int, name: str, value: str, color=None) -> ft.Container:
        """创建排行项"""
        c = theme.colors
        
        # 排名颜色
        rank_colors = {
            1: ft.Colors.AMBER_400,
            2: ft.Colors.GREY_400,
            3: ft.Colors.ORANGE_700,
        }
        rank_color = rank_colors.get(rank, c.text_muted)
        
        return ft.Container(
            content=ft.Row([
                ft.Container(
                    content=ft.Text(str(rank), size=FONT_SIZE_ITEM, color=rank_color, weight=ft.FontWeight.BOLD),
                    width=24,
                ),
                ft.Text(name, size=FONT_SIZE_ITEM, color=c.text_primary, expand=True, max_lines=1, overflow=ft.TextOverflow.ELLIPSIS),
                ft.Text(value, size=FONT_SIZE_ITEM, color=color or c.text_muted, width=60, text_align=ft.TextAlign.RIGHT),
            ], spacing=6, vertical_alignment=ft.CrossAxisAlignment.CENTER),
            padding=ft.padding.symmetric(vertical=4),
        )
    
    def _gift_item(self, name: str, gift_name: str, count: int, value: str) -> ft.Container:
        """创建礼物记录项"""
        c = theme.colors
        
        return ft.Container(
            content=ft.Row([
                ft.Text(name, size=FONT_SIZE_ITEM, color=c.text_primary, width=70, max_lines=1, overflow=ft.TextOverflow.ELLIPSIS),
                ft.Text(f"送出 {gift_name} x{count}", size=FONT_SIZE_ITEM, color=c.text_muted, expand=True),
                ft.Text(value, size=FONT_SIZE_ITEM, color=ft.Colors.ORANGE_400, width=60, text_align=ft.TextAlign.RIGHT),
            ], spacing=6, vertical_alignment=ft.CrossAxisAlignment.CENTER),
            padding=ft.padding.symmetric(vertical=3),
        )
    
    def _on_data_update(self):
        """数据更新回调"""
        if self.page:
            self.page.run_thread(self.refresh_data)
    
    def refresh_data(self):
        """刷新显示数据"""
        if not self._collector:
            return
        
        c = theme.colors
        
        try:
            # 获取汇总数据
            summary = self._collector.get_summary()
            
            self.duration_text.value = summary['duration']
            self.online_text.value = LiveStats.format_number(summary['current_online'])
            self.peak_text.value = f"峰值: {summary['peak_online']}"
            self.entrance_text.value = LiveStats.format_number(summary['total_entrance'])
            self.follow_text.value = str(summary['total_follow'])
            self.chat_text.value = LiveStats.format_number(summary['total_chat'])
            self.gift_text.value = LiveStats.format_gift_value(summary['total_gift_value'])
            
            # 热度计算
            chat_freq = self._collector.get_chat_frequency()
            heat_name, heat_value = LiveStats.calculate_heat_level(chat_freq, summary['current_online'])
            self.heat_text.value = heat_name
            self.heat_bar.value = heat_value / 100
            
            # 发言排行
            chat_rank = self._collector.get_chat_rank(10)
            self.chat_rank_list.controls.clear()
            for i, user in enumerate(chat_rank, 1):
                self.chat_rank_list.controls.append(
                    self._rank_item(i, user.user.nickname, f"{user.chat_count}条", ft.Colors.BLUE_400)
                )
            if not chat_rank:
                self.chat_rank_list.controls.append(
                    ft.Text("暂无数据", size=12, color=c.text_muted)
                )
            
            # 礼物排行
            gift_rank = self._collector.get_gift_rank(10)
            self.gift_rank_list.controls.clear()
            for i, user in enumerate(gift_rank, 1):
                self.gift_rank_list.controls.append(
                    self._rank_item(i, user.user.nickname, LiveStats.format_gift_value(user.gift_total), ft.Colors.ORANGE_400)
                )
            if not gift_rank:
                self.gift_rank_list.controls.append(
                    ft.Text("暂无数据", size=12, color=c.text_muted)
                )
            
            # 最近礼物
            recent_gifts = self._collector.get_recent_gifts(10)
            self.recent_gift_list.controls.clear()
            for record in recent_gifts:
                self.recent_gift_list.controls.append(
                    self._gift_item(
                        record.user.nickname,
                        record.gift.name,
                        record.gift.count,
                        LiveStats.format_gift_value(record.value)
                    )
                )
            if not recent_gifts:
                self.recent_gift_list.controls.append(
                    ft.Text("暂无礼物", size=12, color=c.text_muted)
                )
            
            if self.page:
                self.update()
                
        except Exception as e:
            print(f"[Analytics] 刷新数据错误: {e}")
    
    def set_collector(self, collector: LiveDataCollector):
        """设置数据采集器"""
        self._collector = collector
        self._collector.set_update_callback(self._on_data_update)
    
    def _refresh_sessions(self, e=None):
        """刷新历史场次列表"""
        c = theme.colors
        
        try:
            sessions = self._collector._session_manager.get_recent_sessions(10)
            self.session_list.controls.clear()
            
            for session in sessions:
                # 计算时长
                if session.summary.duration_seconds > 0:
                    mins = session.summary.duration_seconds // 60
                    duration_str = f"{mins}分钟"
                else:
                    duration_str = "进行中" if session.status == "running" else "未知"
                
                # 格式化开始时间
                time_str = session.started_at.strftime("%m-%d %H:%M")
                
                # 状态标签
                status_color = ft.Colors.GREEN_400 if session.status == "running" else c.text_muted
                
                self.session_list.controls.append(
                    ft.Container(
                        content=ft.Row([
                            ft.Text(time_str, size=FONT_SIZE_ITEM, color=c.text_muted, width=75),
                            ft.Text(duration_str, size=FONT_SIZE_ITEM, color=c.text_primary, width=55),
                            ft.Text(f"👥{session.summary.peak_online}", size=FONT_SIZE_ITEM, color=c.text_muted, width=45),
                            ft.Text(f"💬{session.summary.total_chat}", size=FONT_SIZE_ITEM, color=c.text_muted, width=45),
                            ft.Text(LiveStats.format_gift_value(session.summary.total_gift_value), 
                                   size=FONT_SIZE_ITEM, color=ft.Colors.ORANGE_400, text_align=ft.TextAlign.RIGHT),
                        ], spacing=6, vertical_alignment=ft.CrossAxisAlignment.CENTER),
                        padding=ft.padding.symmetric(vertical=3),
                    )
                )
            
            if not sessions:
                self.session_list.controls.append(
                    ft.Text("暂无历史场次", size=FONT_SIZE_ITEM, color=c.text_muted)
                )
            
            if self.page:
                self.update()
                
        except Exception as e:
            print(f"[Analytics] 刷新场次错误: {e}")
    
    def refresh_theme(self):
        """刷新主题"""
        c = theme.colors
        self.bgcolor = c.bg_default
