# -*- coding: utf-8 -*-
"""防检测 Tab"""

import flet as ft
from typing import Callable
from ..themes import theme
from ..components import Card, StatusIndicator


class AntiDetectTab(ft.Container):
    """防检测 Tab"""
    
    def __init__(self,
                 on_visual_toggle: Callable[[bool], None] = None,
                 on_visual_strength: Callable[[str], None] = None,
                 on_typing_toggle: Callable[[bool], None] = None,
                 on_typing_speed: Callable[[str], None] = None,
                 config_manager = None):
        
        self._on_visual_toggle = on_visual_toggle
        self._on_visual_strength = on_visual_strength
        self._on_typing_toggle = on_typing_toggle
        self._on_typing_speed = on_typing_speed
        self.config_manager = config_manager
        self._original_visual_config = {}  # 保存原始画面配置用于变更检测
        self._original_typing_config = {}  # 保存原始打字配置用于变更检测
        
        c = theme.colors
        
        # ========== 画面防检测 ==========
        self.visual_check = ft.Checkbox(
            label="启用画面防检测",
            value=False,
            on_change=self._on_visual_change,
        )
        
        self.visual_strength = ft.Dropdown(
            options=[
                ft.dropdown.Option("low", "低 (几乎不可见)"),
                ft.dropdown.Option("medium", "中 (推荐)"),
                ft.dropdown.Option("high", "高 (明显变化)"),
            ],
            value="medium",
            width=200,
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            on_change=self._on_visual_strength_change,
        )
        
        self.visual_status = StatusIndicator("未启动")
        
        self.visual_card = Card("画面防检测", ft.Column([
            self.visual_check,
            ft.Row([
                ft.Text("处理强度:", size=13, color=c.text_muted, width=80),
                self.visual_strength,
            ], spacing=10),
            self.visual_status,
        ], spacing=12), show_save_button=True, on_save=self._save_config)
        
        # ========== 打字防检测 ==========
        self.typing_check = ft.Checkbox(
            label="启用人类化打字",
            value=False,
            on_change=self._on_typing_change,
        )
        
        self.typing_speed = ft.Dropdown(
            options=[
                ft.dropdown.Option("slow", "慢速 (40 WPM)"),
                ft.dropdown.Option("normal", "正常 (80 WPM)"),
                ft.dropdown.Option("fast", "快速 (120 WPM)"),
            ],
            value="normal",
            width=200,
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            on_change=self._on_typing_speed_change,
        )
        
        self.typing_status = StatusIndicator("驱动状态: 未检测")
        
        self.typing_card = Card("打字防检测", ft.Column([
            self.typing_check,
            ft.Row([
                ft.Text("打字速度:", size=13, color=c.text_muted, width=80),
                self.typing_speed,
            ], spacing=10),
            self.typing_status,
        ], spacing=12), show_save_button=True, on_save=self._save_config)
        
        super().__init__(
            content=ft.Column([self.visual_card, self.typing_card], spacing=15, scroll=ft.ScrollMode.AUTO, expand=True),
            padding=15,
            expand=True,
        )
    
    def _on_visual_change(self, e):
        """UI 变化时只标记待保存，不立即触发功能"""
        self._notify_change()
        # 注意：不再立即调用 _on_visual_toggle，配置只在保存时才生效
    
    def _on_visual_strength_change(self, e):
        """UI 变化时只标记待保存"""
        self._notify_change()
        # 注意：不再立即调用 _on_visual_strength，配置只在保存时才生效
    
    def _on_typing_change(self, e):
        """UI 变化时只标记待保存"""
        self._notify_change()
        # 注意：不再立即调用 _on_typing_toggle，配置只在保存时才生效
    
    def _on_typing_speed_change(self, e):
        """UI 变化时只标记待保存"""
        self._notify_change()
        # 注意：不再立即调用 _on_typing_speed，配置只在保存时才生效
    
    def _notify_change(self):
        """通知配置变化并更新待保存状态"""
        # 分别检测每个卡片的变化
        visual_config = self._get_visual_config()
        typing_config = self._get_typing_config()
        
        self.visual_card.set_has_changes(visual_config != self._original_visual_config)
        self.typing_card.set_has_changes(typing_config != self._original_typing_config)
    
    def _get_visual_config(self) -> dict:
        """获取画面防检测配置（用于变更检测）"""
        return {
            "visual_enabled": self.visual_check.value,
            "visual_strength": self.visual_strength.value,
        }
    
    def _get_typing_config(self) -> dict:
        """获取打字防检测配置（用于变更检测）"""
        return {
            "typing_enabled": self.typing_check.value,
            "typing_speed": self.typing_speed.value,
        }
    
    def _get_config(self) -> dict:
        """获取完整配置（用于保存）"""
        return {
            "visual_enabled": self.visual_check.value,
            "visual_strength": self.visual_strength.value,
            "typing_enabled": self.typing_check.value,
            "typing_speed": self.typing_speed.value,
        }
    
    def _save_config(self, e=None):
        """保存防检测配置 - 保存时才触发功能生效"""
        config = self._get_config()
        
        if self.config_manager:
            self.config_manager.set_anti_detect_config(config)
            self.config_manager.save()
        
        # 保存成功后触发功能生效
        if self._on_visual_toggle:
            self._on_visual_toggle(config["visual_enabled"])
        if self._on_visual_strength:
            self._on_visual_strength(config["visual_strength"])
        if self._on_typing_toggle:
            self._on_typing_toggle(config["typing_enabled"])
        if self._on_typing_speed:
            self._on_typing_speed(config["typing_speed"])
        
        # 更新原始配置
        self._original_visual_config = self._get_visual_config()
        self._original_typing_config = self._get_typing_config()
        self.visual_card.set_has_changes(False)
        self.typing_card.set_has_changes(False)
        
        self._show_save_dialog(True, "防检测配置已保存")
    
    def _show_save_dialog(self, success: bool, message: str):
        """显示保存结果弹窗"""
        def close_dialog(e):
            dialog.open = False
            if self.page:
                self.page.update()
        
        title = "保存成功" if success else "保存失败"
        btn_color = "#4CAF50" if success else "#E53935"
        
        dialog = ft.AlertDialog(
            modal=True,
            title=ft.Text(title, weight=ft.FontWeight.BOLD),
            content=ft.Text(message, size=14),
            actions=[
                ft.ElevatedButton("确定", bgcolor=btn_color, color="#ffffff", on_click=close_dialog),
            ],
            actions_alignment=ft.MainAxisAlignment.END,
        )
        
        if self.page:
            self.page.overlay.append(dialog)
            dialog.open = True
            self.page.update()
    
    def load_config(self, config_manager):
        """从配置加载数据"""
        self.config_manager = config_manager
        
        config = config_manager.get_anti_detect_config()
        self.visual_check.value = config.get("visual_enabled", False)
        self.visual_strength.value = config.get("visual_strength", "medium")
        self.typing_check.value = config.get("typing_enabled", False)
        self.typing_speed.value = config.get("typing_speed", "normal")
        
        # 保存原始配置用于变更检测
        self._original_visual_config = self._get_visual_config()
        self._original_typing_config = self._get_typing_config()
        self.visual_card.set_has_changes(False)
        self.typing_card.set_has_changes(False)
    
    def set_visual_status(self, running: bool, message: str = None):
        """设置画面防检测状态"""
        c = theme.colors
        if running:
            self.visual_status.set_status(message or "已启动 - 按Q退出覆盖层", c.success)
        else:
            self.visual_status.set_status("未启动", c.text_muted)
    
    def set_typing_status(self, available: bool):
        """设置打字驱动状态"""
        c = theme.colors
        if available:
            self.typing_status.set_status("驱动状态: 可用", c.success)
        else:
            self.typing_status.set_status("驱动状态: 不可用", c.error)
