# -*- coding: utf-8 -*-
"""语音回复 Tab"""

import re
import flet as ft
from typing import Callable, Optional, List, Dict
from ..themes import theme
from ..components import Card

# 模板变量定义
TEMPLATE_VARS = {
    "entrance": ["username"],
    "gift": ["username", "gift_name", "count"],
    "follow": ["username"],
}

# 语音列表 (2024年最新 - 仅保留Edge TTS实际支持的音色)
VOICE_LIST = [
    ("晓晓 (温暖)", "zh-CN-XiaoxiaoNeural"),       # 女声，适合新闻/小说
    ("晓伊 (活泼)", "zh-CN-XiaoyiNeural"),         # 女声，适合动画/小说
    ("云希 (阳光)", "zh-CN-YunxiNeural"),          # 男声，阳光活泼
    ("云健 (激情)", "zh-CN-YunjianNeural"),        # 男声，适合体育/小说
    ("云夏 (可爱)", "zh-CN-YunxiaNeural"),         # 男声，适合动画
    ("云扬 (专业)", "zh-CN-YunyangNeural"),        # 男声，适合新闻播报
    ("小北 (东北话)", "zh-CN-liaoning-XiaobeiNeural"),  # 女声，东北方言
    ("小妮 (陕西话)", "zh-CN-shaanxi-XiaoniNeural"),    # 女声，陕西方言
]

RATE_OPTIONS = [
    ("-20%", "-20%"),
    ("-10%", "-10%"),
    ("+0%", "正常"),
    ("+10%", "+10%"),
    ("+20%", "+20%"),
]


class VoiceTab(ft.Container):
    """语音回复 Tab"""
    
    def __init__(self, on_preview: Callable = None, on_config_change: Callable = None, on_save: Callable = None, config_manager = None):
        self._on_preview = on_preview
        self._on_config_change = on_config_change
        self._on_save = on_save
        self.config_manager = config_manager
        self._original_config = {}  # 保存原始语音配置用于变更检测
        self._original_reply_rule = {}  # 保存原始回复规则用于变更检测
        
        c = theme.colors
        
        # ========== TTS 引擎 ==========
        self.engine_dropdown = ft.Dropdown(
            options=[
                ft.dropdown.Option("edge", "Edge-TTS (免费)"),
                ft.dropdown.Option("volcengine", "火山引擎 (克隆)"),
            ],
            value="edge",
            width=200,
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            on_change=self._on_engine_change,
        )
        
        # ========== Edge 配置 ==========
        self.voice_dropdown = ft.Dropdown(
            options=[ft.dropdown.Option(vid, name) for name, vid in VOICE_LIST],
            value=VOICE_LIST[0][1],
            width=200,
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            on_change=self._notify_change,
        )
        
        self.rate_dropdown = ft.Dropdown(
            options=[ft.dropdown.Option(val, name) for val, name in RATE_OPTIONS],
            value="+0%",
            width=150,
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            on_change=self._notify_change,
        )
        
        # ========== 火山引擎配置 ==========
        self.volc_appid_input = ft.TextField(
            hint_text="AppID",
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            width=200,
            on_change=self._notify_change,
        )
        
        self.volc_token_input = ft.TextField(
            hint_text="Access Token",
            password=True,
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            width=200,
            on_change=self._notify_change,
        )
        
        self.volc_voice_input = ft.TextField(
            hint_text="BV001_streaming",
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            width=200,
            on_change=self._notify_change,
        )
        
        # Edge 配置面板
        self.edge_panel = ft.Column([
            ft.Row([
                ft.Text("角色:", size=13, color=c.text_muted, width=60),
                self.voice_dropdown,
            ], spacing=10),
            ft.Row([
                ft.Text("语速:", size=13, color=c.text_muted, width=60),
                self.rate_dropdown,
            ], spacing=10),
        ], spacing=10, visible=True)
        
        # 火山引擎配置面板
        self.volc_panel = ft.Column([
            ft.Row([
                ft.Text("AppID:", size=13, color=c.text_muted, width=60),
                self.volc_appid_input,
            ], spacing=10),
            ft.Row([
                ft.Text("Token:", size=13, color=c.text_muted, width=60),
                self.volc_token_input,
            ], spacing=10),
            ft.Row([
                ft.Text("音色:", size=13, color=c.text_muted, width=60),
                self.volc_voice_input,
            ], spacing=10),
        ], spacing=10, visible=False)
        
        # ========== 音量和试听 ==========
        self.volume_slider = ft.Slider(
            min=0, max=100, value=80, divisions=20, label="{value}%",
            on_change=self._on_volume_change,
        )
        
        self.volume_label = ft.Text("80%", size=12, color=c.text_secondary, width=40)
        
        self.preview_btn = ft.OutlinedButton(
            "试听",
            on_click=self._handle_preview,
            style=ft.ButtonStyle(shape=ft.RoundedRectangleBorder(radius=6)),
        )
        
        # ========== 语音设置卡片 ==========
        self.voice_card = Card("语音设置", ft.Column([
            ft.Row([
                ft.Text("引擎:", size=13, color=c.text_muted, width=60),
                self.engine_dropdown,
            ], spacing=10),
            self.edge_panel,
            self.volc_panel,
            ft.Row([
                ft.Text("音量:", size=13, color=c.text_muted, width=60),
                self.volume_slider,
                self.volume_label,
                self.preview_btn,
            ], spacing=10),
        ], spacing=14), on_save=self._save_voice_config, show_save_button=True)
        
        # ========== 回复规则 ==========
        self.entrance_check = ft.Checkbox(label="入场欢迎", value=True, on_change=self._notify_change)
        self.entrance_cd = ft.TextField(value="3", width=60, text_align=ft.TextAlign.CENTER,
                                        border_color=c.border_default, bgcolor=c.bg_elevated, text_size=12,
                                        content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
                                        on_change=self._notify_change)
        
        self.gift_check = ft.Checkbox(label="礼物感谢", value=True, on_change=self._notify_change)
        self.gift_cd = ft.TextField(value="1", width=60, text_align=ft.TextAlign.CENTER,
                                    border_color=c.border_default, bgcolor=c.bg_elevated, text_size=12,
                                    content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
                                    on_change=self._notify_change)
        
        self.follow_check = ft.Checkbox(label="关注感谢", value=True, on_change=self._notify_change)
        self.follow_cd = ft.TextField(value="2", width=60, text_align=ft.TextAlign.CENTER,
                                      border_color=c.border_default, bgcolor=c.bg_elevated, text_size=12,
                                      content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
                                      on_change=self._notify_change)
        
        self.read_chat_check = ft.Checkbox(label="朗读弹幕", value=True, on_change=self._notify_change)
        
        self.reply_rule_card = Card("回复规则", ft.Column([
            ft.Row([
                ft.Container(ft.Row([self.entrance_check, self.entrance_cd], spacing=5), width=200),
                ft.Container(ft.Row([self.gift_check, self.gift_cd], spacing=5), width=200),
            ], spacing=20),
            ft.Row([
                ft.Container(ft.Row([self.follow_check, self.follow_cd], spacing=5), width=200),
                ft.Container(self.read_chat_check, width=200),
            ], spacing=20),
        ], spacing=10), show_save_button=True, on_save=self._save_reply_rule_config)
        
        # ========== 语音模板 ==========
        # 保存原始模板值用于重置
        self._original_templates = {
            "entrance": "",
            "gift": "",
            "follow": "",
        }
        
        self.entrance_template = ft.TextField(
            hint_text="欢迎 {username} 来到直播间！",
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            expand=True,
            on_change=lambda e: self._validate_template_field("entrance"),
        )
        self.entrance_template_status = ft.Text("", size=11, color=c.text_muted)
        
        self.gift_template = ft.TextField(
            hint_text="感谢 {username} 的 {gift_name}！",
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            expand=True,
            on_change=lambda e: self._validate_template_field("gift"),
        )
        self.gift_template_status = ft.Text("", size=11, color=c.text_muted)
        
        self.follow_template = ft.TextField(
            hint_text="感谢 {username} 的关注！",
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            expand=True,
            on_change=lambda e: self._validate_template_field("follow"),
        )
        self.follow_template_status = ft.Text("", size=11, color=c.text_muted)
        
        # 模板状态文本
        self.template_status = ft.Text("", size=11, color=c.text_muted)
        
        self.template_card = Card("语音模板", ft.Column([
            ft.Text("可用变量: {username} 用户名, {gift_name} 礼物名, {count} 数量", size=11, color=c.text_muted, italic=True),
            ft.Row([ft.Text("入场:", size=12, color=c.text_secondary, width=40), self.entrance_template], spacing=8),
            self.entrance_template_status,
            ft.Row([ft.Text("礼物:", size=12, color=c.text_secondary, width=40), self.gift_template], spacing=8),
            self.gift_template_status,
            ft.Row([ft.Text("关注:", size=12, color=c.text_secondary, width=40), self.follow_template], spacing=8),
            self.follow_template_status,
            self.template_status,
        ], spacing=8), show_save_button=True, on_save=self._save_templates)
        
        # ========== 主容器 ==========
        super().__init__(
            content=ft.Column([self.voice_card, self.reply_rule_card, self.template_card], spacing=15, scroll=ft.ScrollMode.AUTO, expand=True),
            padding=15,
            expand=True,
        )
    
    def _on_engine_change(self, e):
        """引擎切换"""
        is_edge = e.control.value == "edge"
        self.edge_panel.visible = is_edge
        self.volc_panel.visible = not is_edge
        if self.page:
            self.page.update()
        self._notify_change(e)
    
    def _on_volume_change(self, e):
        """音量变化"""
        self.volume_label.value = f"{int(e.control.value)}%"
        self._notify_change(e)
    
    def _handle_preview(self, e):
        """试听"""
        if self._on_preview:
            engine = self.engine_dropdown.value
            volume = self.volume_slider.value / 100.0
            
            if engine == "edge":
                params = {
                    "voice": self.voice_dropdown.value,
                    "rate": self.rate_dropdown.value,
                }
            else:
                params = {
                    "app_id": self.volc_appid_input.value,
                    "access_token": self.volc_token_input.value,
                    "voice_type": self.volc_voice_input.value or "BV001_streaming",
                    "cluster": "volcano_tts",
                }
            
            self._on_preview(engine, params, volume)
    
    def _validate_template(self, template_type: str, template: str) -> tuple:
        """校验模板格式
        :return: (is_valid, error_msg)
        """
        if not template.strip():
            return True, ""  # 空模板使用默认值
        
        # 检查变量格式
        valid_vars = TEMPLATE_VARS.get(template_type, [])
        # 找出所有 {xxx} 格式的变量
        found_vars = re.findall(r'\{(\w+)\}', template)
        
        invalid_vars = [v for v in found_vars if v not in valid_vars]
        if invalid_vars:
            return False, f"无效变量: {{{', '.join(invalid_vars)}}}  可用: {{{', '.join(valid_vars)}}}"
        
        # 检查大括号匹配
        if template.count('{') != template.count('}'):
            return False, "括号不匹配"
        
        return True, ""
    
    def _validate_template_field(self, template_type: str):
        """实时校验模板字段（不保存）"""
        c = theme.colors
        template_map = {
            "entrance": (self.entrance_template, self.entrance_template_status),
            "gift": (self.gift_template, self.gift_template_status),
            "follow": (self.follow_template, self.follow_template_status),
        }
        
        field, status = template_map.get(template_type, (None, None))
        if not field:
            return
        
        template = field.value or ""
        is_valid, error_msg = self._validate_template(template_type, template)
        
        if is_valid:
            status.value = ""
            field.border_color = c.border_default
        else:
            status.value = error_msg
            status.color = c.error
            field.border_color = c.error
        
        # 检测模板是否有变化，设置待保存状态
        current_templates = {
            "entrance": self.entrance_template.value or "",
            "gift": self.gift_template.value or "",
            "follow": self.follow_template.value or "",
        }
        has_changes = current_templates != self._original_templates
        self.template_card.set_has_changes(has_changes)
        
        if self.page:
            self.page.update()
    
    def _save_templates(self, e=None):
        """保存模板"""
        c = theme.colors
        # 校验所有模板
        all_valid = True
        for ttype in ["entrance", "gift", "follow"]:
            template_map = {
                "entrance": self.entrance_template,
                "gift": self.gift_template,
                "follow": self.follow_template,
            }
            template = template_map[ttype].value or ""
            is_valid, _ = self._validate_template(ttype, template)
            if not is_valid:
                all_valid = False
                self._validate_template_field(ttype)
        
        if not all_valid:
            self._show_save_dialog(False, "请修正模板错误后再保存")
            return
        
        # 保存成功，更新原始值
        self._original_templates["entrance"] = self.entrance_template.value or ""
        self._original_templates["gift"] = self.gift_template.value or ""
        self._original_templates["follow"] = self.follow_template.value or ""
        
        # 重置待保存状态
        self.template_card.set_has_changes(False)
        
        # 通知配置变更
        self._notify_change()
        
        self.template_status.value = ""
        self._show_save_dialog(True, "语音模板已保存")
    
    def _show_save_dialog(self, success: bool, message: str):
        """显示保存结果弹窗"""
        def close_dialog(e):
            dialog.open = False
            if self.page:
                self.page.update()
        
        title = "保存成功" if success else "保存失败"
        btn_color = "#4CAF50" if success else "#E53935"
        
        dialog = ft.AlertDialog(
            modal=True,
            title=ft.Text(title, weight=ft.FontWeight.BOLD),
            content=ft.Text(message, size=14),
            actions=[
                ft.ElevatedButton("确定", bgcolor=btn_color, color="#ffffff", on_click=close_dialog),
            ],
            actions_alignment=ft.MainAxisAlignment.END,
        )
        
        if self.page:
            self.page.overlay.append(dialog)
            dialog.open = True
            self.page.update()
    
    def _confirm_reset_templates(self, e=None):
        """确认重置模板"""
        def close_dialog(confirmed):
            dialog.open = False
            if self.page:
                self.page.update()
            if confirmed:
                self._reset_templates()
        
        dialog = ft.AlertDialog(
            modal=True,
            title=ft.Text("确认重置"),
            content=ft.Text("确定要重置模板吗？未保存的修改将丢失。"),
            actions=[
                ft.TextButton("取消", on_click=lambda e: close_dialog(False)),
                ft.ElevatedButton("确认重置", on_click=lambda e: close_dialog(True)),
            ],
            actions_alignment=ft.MainAxisAlignment.END,
        )
        
        if self.page:
            self.page.overlay.append(dialog)
            dialog.open = True
            self.page.update()
    
    def _reset_templates(self):
        """重置模板到上次保存的值"""
        c = theme.colors
        self.entrance_template.value = self._original_templates["entrance"]
        self.gift_template.value = self._original_templates["gift"]
        self.follow_template.value = self._original_templates["follow"]
        
        # 清除错误状态
        for field in [self.entrance_template, self.gift_template, self.follow_template]:
            field.border_color = c.border_default
        for status in [self.entrance_template_status, self.gift_template_status, self.follow_template_status]:
            status.value = ""
        
        self.template_status.value = "✓ 已重置"
        self.template_status.color = c.text_muted
        
        if self.page:
            self.page.update()
    
    def _notify_change(self, e=None):
        """通知配置变化并更新待保存状态 - 不立即更新 config_manager"""
        # 检测语音设置是否有变更
        current_config = self._get_voice_config()
        has_voice_changes = current_config != self._original_config
        self.voice_card.set_has_changes(has_voice_changes)
        
        # 检测回复规则是否有变更
        current_reply_rule = self._get_reply_rule_config()
        has_rule_changes = current_reply_rule != self._original_reply_rule
        self.reply_rule_card.set_has_changes(has_rule_changes)
        
        # 注意：不再调用 _on_config_change，配置只在保存时才生效
    
    def _get_reply_rule_config(self) -> dict:
        """获取回复规则配置（用于变更检测）"""
        return {
            "entrance_enabled": self.entrance_check.value,
            "entrance_cd": self.entrance_cd.value or "3",
            "gift_enabled": self.gift_check.value,
            "gift_cd": self.gift_cd.value or "1",
            "follow_enabled": self.follow_check.value,
            "follow_cd": self.follow_cd.value or "2",
            "read_chat_enabled": self.read_chat_check.value,
        }
    
    def _save_reply_rule_config(self, e=None):
        """保存回复规则配置"""
        if self.config_manager:
            reply_config = {
                "entrance_enabled": self.entrance_check.value,
                "entrance_cd": int(self.entrance_cd.value or 3),
                "gift_enabled": self.gift_check.value,
                "gift_cd": int(self.gift_cd.value or 1),
                "follow_enabled": self.follow_check.value,
                "follow_cd": int(self.follow_cd.value or 2),
                "read_chat_enabled": self.read_chat_check.value,
            }
            self.config_manager.set_reply_config(reply_config)
            self.config_manager.save()
        
        # 更新原始配置
        self._original_reply_rule = self._get_reply_rule_config()
        self.reply_rule_card.set_has_changes(False)
        
        self._show_save_dialog(True, "回复规则已保存")
    
    def _get_voice_config(self) -> dict:
        """获取语音设置配置（用于变更检测）"""
        return {
            "engine": self.engine_dropdown.value,
            "voice": self.voice_dropdown.value,
            "rate": self.rate_dropdown.value,
            "volume": int(self.volume_slider.value),
            "volc_appid": self.volc_appid_input.value or "",
            "volc_token": self.volc_token_input.value or "",
            "volc_voice": self.volc_voice_input.value or "",
        }
    
    def _save_voice_config(self, e=None):
        """保存语音设置"""
        if self.config_manager:
            config = self.get_config()
            # 保存 TTS 配置
            self.config_manager.set_tts_engine(config["tts"]["engine"])
            self.config_manager.set_edge_voice(config["tts"]["edge"]["voice"])
            self.config_manager.set_edge_rate(config["tts"]["edge"]["rate"])
            self.config_manager.set_volume(config["audio"]["volume"])
            
            # 保存火山引擎配置
            volc = config["tts"]["volcengine"]
            self.config_manager.set_volc_config(volc["app_id"], volc["access_token"], volc["voice_type"])
            
            # 保存配置到文件
            self.config_manager.save()
        
        # 更新原始配置
        self._original_config = self._get_voice_config()
        self.voice_card.set_has_changes(False)
        
        self._show_save_dialog(True, "语音设置已保存")
        
        # 通知外部保存回调
        if self._on_save:
            self._on_save(self.get_config())
    
    def get_config(self) -> dict:
        """获取当前配置"""
        return {
            "tts": {
                "engine": self.engine_dropdown.value,
                "edge": {
                    "voice": self.voice_dropdown.value,
                    "rate": self.rate_dropdown.value,
                },
                "volcengine": {
                    "app_id": self.volc_appid_input.value,
                    "access_token": self.volc_token_input.value,
                    "voice_type": self.volc_voice_input.value,
                },
            },
            "audio": {
                "volume": self.volume_slider.value / 100.0,
            },
            "reply": {
                "entrance": {
                    "enabled": self.entrance_check.value,
                    "cooldown": int(self.entrance_cd.value or 3),
                    "templates": [self.entrance_template.value] if self.entrance_template.value else ["欢迎 {username} 来到直播间！"],
                },
                "gift": {
                    "enabled": self.gift_check.value,
                    "cooldown": int(self.gift_cd.value or 1),
                    "templates": [self.gift_template.value] if self.gift_template.value else ["感谢 {username} 的 {gift_name}！"],
                },
                "follow": {
                    "enabled": self.follow_check.value,
                    "cooldown": int(self.follow_cd.value or 2),
                    "templates": [self.follow_template.value] if self.follow_template.value else ["感谢 {username} 的关注！"],
                },
                "chat": {
                    "enabled": True,
                    "read_chat": self.read_chat_check.value,
                },
            },
        }
    
    def load_config(self, config_manager):
        """从配置加载数据"""
        self.config_manager = config_manager
        
        self.engine_dropdown.value = config_manager.get_tts_engine()
        self.voice_dropdown.value = config_manager.get_edge_voice()
        self.rate_dropdown.value = config_manager.get_edge_rate()
        self.volume_slider.value = config_manager.get_volume() * 100
        self.volume_label.value = f"{int(self.volume_slider.value)}%"
        
        # 切换面板
        is_edge = self.engine_dropdown.value == "edge"
        self.edge_panel.visible = is_edge
        self.volc_panel.visible = not is_edge
        
        # 回复规则
        reply = config_manager.get_reply_config()
        
        # 支持旧版和新版配置格式
        entrance_cfg = reply.get("entrance", {})
        if isinstance(entrance_cfg, dict):
            self.entrance_check.value = entrance_cfg.get("enabled", True)
            self.entrance_cd.value = str(entrance_cfg.get("cooldown", 3))
            templates = entrance_cfg.get("templates", [])
            self.entrance_template.value = templates[0] if templates else ""
        else:
            self.entrance_check.value = reply.get("entrance_enabled", True)
            self.entrance_cd.value = str(reply.get("entrance_cd", 3))
        
        gift_cfg = reply.get("gift", {})
        if isinstance(gift_cfg, dict):
            self.gift_check.value = gift_cfg.get("enabled", True)
            self.gift_cd.value = str(gift_cfg.get("cooldown", 1))
            templates = gift_cfg.get("templates", [])
            self.gift_template.value = templates[0] if templates else ""
        else:
            self.gift_check.value = reply.get("gift_enabled", True)
            self.gift_cd.value = str(reply.get("gift_cd", 1))
        
        follow_cfg = reply.get("follow", {})
        if isinstance(follow_cfg, dict):
            self.follow_check.value = follow_cfg.get("enabled", True)
            self.follow_cd.value = str(follow_cfg.get("cooldown", 2))
            templates = follow_cfg.get("templates", [])
            self.follow_template.value = templates[0] if templates else ""
        else:
            self.follow_check.value = reply.get("follow_enabled", True)
            self.follow_cd.value = str(reply.get("follow_cd", 2))
        
        chat_cfg = reply.get("chat", {})
        if isinstance(chat_cfg, dict):
            self.read_chat_check.value = chat_cfg.get("read_chat", True)
        else:
            self.read_chat_check.value = reply.get("read_chat_enabled", True)
        
        # 保存原始模板值用于重置
        self._original_templates["entrance"] = self.entrance_template.value or ""
        self._original_templates["gift"] = self.gift_template.value or ""
        self._original_templates["follow"] = self.follow_template.value or ""
        
        # 保存原始语音配置用于变更检测
        self._original_config = self._get_voice_config()
        self.voice_card.set_has_changes(False)
        
        # 保存原始回复规则配置用于变更检测
        self._original_reply_rule = self._get_reply_rule_config()
        self.reply_rule_card.set_has_changes(False)
        
        # 重置模板卡片状态
        self.template_card.set_has_changes(False)
    
    def set_preview_state(self, playing: bool):
        """设置试听状态"""
        self.preview_btn.text = "播放中..." if playing else "试听"
        self.preview_btn.disabled = playing
