# -*- coding: utf-8 -*-
"""
音频播放模块
支持队列播放、音量控制和口型同步
"""

import asyncio
import threading
from typing import Optional, List, Callable
from pathlib import Path
from queue import Queue, Empty
import os

from src.utils.logger import logger


class AudioPlayer:
    """
    音频播放器
    支持队列播放，避免多个音频同时播放
    支持口型同步回调
    """
    
    def __init__(
        self,
        volume: float = 1.0,
        queue_max_size: int = 20,
        auto_delete: bool = True,
        lipsync_callback: Optional[Callable[[float], None]] = None
    ):
        """
        初始化音频播放器
        
        Args:
            volume: 音量 (0.0 - 1.0)
            queue_max_size: 队列最大长度
            auto_delete: 播放完成后是否自动删除文件
            lipsync_callback: 口型同步回调函数
        """
        self.volume = max(0.0, min(1.0, volume))
        self.queue_max_size = queue_max_size
        self.auto_delete = auto_delete
        self.lipsync_callback = lipsync_callback
        
        self._queue: Queue = Queue(maxsize=queue_max_size)
        self._playing = False
        self._running = False
        self._player_thread: Optional[threading.Thread] = None
        self._pygame_initialized = False
        self._lipsync_thread: Optional[threading.Thread] = None
        
        # 尝试初始化pygame
        self._init_pygame()
    
    def _init_pygame(self) -> bool:
        """初始化pygame音频模块"""
        try:
            import pygame
            pygame.mixer.init()
            self._pygame_initialized = True
            logger.info("音频播放器初始化成功 (pygame)")
            return True
        except Exception as e:
            logger.warning(f"pygame初始化失败: {e}，将使用系统播放器")
            self._pygame_initialized = False
            return False
    
    def start(self) -> None:
        """启动播放器"""
        if self._running:
            return
        
        self._running = True
        self._player_thread = threading.Thread(target=self._player_loop)
        self._player_thread.daemon = True
        self._player_thread.start()
        logger.info("音频播放器已启动")
    
    def stop(self) -> None:
        """停止播放器并停止当前播放"""
        self._running = False
        self._playing = False
        
        # 停止当前播放
        if self._pygame_initialized:
            try:
                import pygame
                pygame.mixer.music.stop()
            except:
                pass
        
        # 清空队列
        self.clear_queue()
        logger.info("音频播放器已停止")
    
    def clear_queue(self) -> None:
        """清空播放队列"""
        while not self._queue.empty():
            try:
                path = self._queue.get_nowait()
                # 删除队列中的文件
                if self.auto_delete:
                    self._delete_file(path)
            except Empty:
                break
    
    def play(self, audio_path: str, priority: bool = False) -> bool:
        """
        添加音频到播放队列
        
        Args:
            audio_path: 音频文件路径
            priority: 是否优先播放（插入队首）
        
        Returns:
            bool: 是否成功添加到队列
        """
        if not audio_path or not Path(audio_path).exists():
            logger.warning(f"音频文件不存在: {audio_path}")
            return False
        
        try:
            if self._queue.full():
                logger.warning("播放队列已满，丢弃音频")
                return False
            
            self._queue.put(audio_path)
            logger.debug(f"添加到播放队列: {audio_path}")
            return True
            
        except Exception as e:
            logger.error(f"添加到队列失败: {e}")
            return False
    
    async def play_async(self, audio_path: str) -> bool:
        """异步播放（添加到队列）"""
        return self.play(audio_path)
    
    def _player_loop(self) -> None:
        """播放器主循环"""
        while self._running:
            try:
                # 从队列获取音频
                audio_path = self._queue.get(timeout=0.5)
                
                # 播放音频
                self._playing = True
                self._play_audio(audio_path)
                self._playing = False
                
                # 自动删除
                if self.auto_delete:
                    self._delete_file(audio_path)
                
            except Empty:
                continue
            except Exception as e:
                logger.error(f"播放错误: {e}")
                self._playing = False
    
    def _play_audio(self, audio_path: str) -> None:
        """播放单个音频文件"""
        if self._pygame_initialized:
            self._play_with_pygame(audio_path)
        else:
            self._play_with_system(audio_path)
    
    def _play_with_pygame(self, audio_path: str) -> None:
        """使用pygame播放，同时进行口型同步"""
        try:
            import pygame
            import time
            
            # 始终使用 pydub 标准化并增强音量
            boosted_path = self._boost_audio_volume(audio_path)
            
            pygame.mixer.music.load(boosted_path)
            pygame.mixer.music.set_volume(1.0)  # pygame 音量设为最大，由 pydub 控制
            pygame.mixer.music.play()
            
            # 启动口型同步（如果有回调）
            if self.lipsync_callback:
                self._start_lipsync(audio_path)
            
            # 等待播放完成
            while pygame.mixer.music.get_busy() and self._running:
                time.sleep(0.05)
            
            # 停止口型同步
            if self.lipsync_callback:
                self.lipsync_callback(0.0)
                
        except Exception as e:
            logger.error(f"pygame播放失败: {e}")
            if self.lipsync_callback:
                self.lipsync_callback(0.0)
    
    def _start_lipsync(self, audio_path: str):
        """启动口型同步分析"""
        import time
        import random
        
        def analyze():
            try:
                # 尝试使用 pydub 精确分析
                try:
                    from pydub import AudioSegment
                    audio = AudioSegment.from_file(audio_path)
                    duration_ms = len(audio)
                    interval_ms = 50  # 50ms 采样
                    
                    start_time = time.time()
                    
                    for i in range(0, duration_ms, interval_ms):
                        if not self._playing:
                            break
                        
                        chunk = audio[i:i + interval_ms]
                        rms = chunk.rms
                        # 归一化并放大
                        volume = min(1.0, rms / 8000)
                        
                        if self.lipsync_callback:
                            self.lipsync_callback(volume)
                        
                        # 时间同步
                        elapsed = time.time() - start_time
                        expected = (i + interval_ms) / 1000
                        sleep_time = expected - elapsed
                        if sleep_time > 0:
                            time.sleep(sleep_time)
                    
                    return
                    
                except ImportError:
                    pass
                
                # 备用：简单的模拟口型
                while self._playing:
                    # 随机生成口型值模拟说话
                    value = random.uniform(0.3, 0.9) if random.random() > 0.2 else random.uniform(0.0, 0.3)
                    if self.lipsync_callback:
                        self.lipsync_callback(value)
                    time.sleep(0.08)
                    
            except Exception as e:
                logger.debug(f"口型分析错误: {e}")
        
        self._lipsync_thread = threading.Thread(target=analyze, daemon=True)
        self._lipsync_thread.start()
    
    def _play_with_system(self, audio_path: str) -> None:
        """使用系统播放器播放"""
        import subprocess
        import platform
        
        try:
            system = platform.system()
            
            if system == "Windows":
                # Windows使用PowerShell播放
                subprocess.run(
                    ["powershell", "-c", f"(New-Object Media.SoundPlayer '{audio_path}').PlaySync()"],
                    shell=False,
                    capture_output=True
                )
            elif system == "Darwin":  # macOS
                subprocess.run(["afplay", audio_path], capture_output=True)
            else:  # Linux
                subprocess.run(["aplay", audio_path], capture_output=True)
                
        except Exception as e:
            logger.error(f"系统播放器失败: {e}")
    
    def _boost_audio_volume(self, audio_path: str) -> str:
        """使用 pydub 标准化并增强音频音量"""
        try:
            from pydub import AudioSegment
            from pydub.effects import normalize
            
            audio = AudioSegment.from_file(audio_path)
            
            # 1. 首先标准化音频（将峰值提升到 0dB）
            normalized = normalize(audio)
            
            # 2. 然后根据用户音量设置额外增益
            # volume 0.5 -> +6dB, volume 1.0 -> +18dB
            extra_gain_db = 6 + (self.volume - 0.5) * 24
            boosted = normalized + extra_gain_db
            
            # 3. 防止削波：如果峰值超过 0dB，进行限幅
            if boosted.max_dBFS > 0:
                boosted = boosted - boosted.max_dBFS
            
            # 保存到临时文件
            boosted_path = audio_path.replace('.mp3', '_boosted.mp3').replace('.wav', '_boosted.wav')
            boosted.export(boosted_path, format="mp3" if audio_path.endswith('.mp3') else "wav")
            
            logger.debug(f"音频标准化+增益: {boosted_path}")
            return boosted_path
            
        except ImportError:
            logger.debug("pydub 未安装，无法增加音量增益")
            return audio_path
        except Exception as e:
            logger.debug(f"音频增益处理失败: {e}")
            return audio_path
    
    def _delete_file(self, file_path: str) -> None:
        """删除文件"""
        try:
            Path(file_path).unlink()
            logger.debug(f"已删除临时文件: {file_path}")
            
            # 同时删除增益版本
            boosted_path = file_path.replace('.mp3', '_boosted.mp3').replace('.wav', '_boosted.wav')
            if Path(boosted_path).exists():
                Path(boosted_path).unlink()
        except Exception as e:
            logger.debug(f"删除文件失败: {e}")
    
    @property
    def is_playing(self) -> bool:
        """是否正在播放"""
        return self._playing
    
    @property
    def queue_size(self) -> int:
        """当前队列大小"""
        return self._queue.qsize()
    
    def set_volume(self, volume: float) -> None:
        """设置音量 (0.0 - 1.0)"""
        self.volume = max(0.0, min(1.0, volume))
        if self._pygame_initialized:
            try:
                import pygame
                pygame.mixer.music.set_volume(self.volume)
            except:
                pass


# 全局播放器实例
_player: Optional[AudioPlayer] = None


def get_audio_player() -> AudioPlayer:
    """获取音频播放器实例"""
    global _player
    if _player is None:
        _player = AudioPlayer()
    return _player


def init_audio_player(volume: float = 1.0, queue_max_size: int = 20, auto_delete: bool = True,
                      lipsync_callback: Optional[Callable[[float], None]] = None) -> AudioPlayer:
    """初始化音频播放器"""
    global _player
    _player = AudioPlayer(
        volume=volume, 
        queue_max_size=queue_max_size, 
        auto_delete=auto_delete,
        lipsync_callback=lipsync_callback
    )
    _player.start()
    return _player
