# -*- coding: utf-8 -*-
"""
自动更新器
集成版本检查、下载、安装的完整更新流程

功能：
1. 启动时自动检查更新
2. 后台静默下载
3. 显示更新提示弹窗
4. 支持增量更新和安装包更新
5. 更新完成后自动重启
"""

import os
import sys
import logging
import threading
import subprocess
from pathlib import Path
from typing import Callable, Optional
from dataclasses import dataclass
from enum import Enum

logger = logging.getLogger(__name__)

# 导入更新检查器
try:
    from .update_checker import UpdateChecker, UpdateInfo, UpdateStatus, get_update_checker
    UPDATE_CHECKER_AVAILABLE = True
except ImportError:
    UPDATE_CHECKER_AVAILABLE = False
    logger.warning("UpdateChecker 不可用")


class UpdateMode(Enum):
    """更新模式"""
    INCREMENTAL = "incremental"  # 增量更新（替换文件）
    INSTALLER = "installer"       # 安装包更新（下载新安装包）


@dataclass
class AutoUpdateConfig:
    """自动更新配置"""
    enabled: bool = True                    # 是否启用自动更新
    check_on_startup: bool = True           # 启动时检查
    check_interval: int = 3600              # 检查间隔（秒）
    silent_download: bool = True            # 静默下载
    auto_install: bool = False              # 自动安装（不提示）
    update_url: str = ""                    # 更新服务器 URL
    software_code: str = "DOUYIN_LIVE_ASSISTANT"  # 软件编码


class AutoUpdater:
    """自动更新器"""
    
    _instance = None
    
    def __new__(cls):
        if cls._instance is None:
            cls._instance = super(AutoUpdater, cls).__new__(cls)
            cls._instance._initialized = False
        return cls._instance
    
    def __init__(self):
        if self._initialized:
            return
        
        self._config = AutoUpdateConfig()
        self._checker: Optional[UpdateChecker] = None
        self._update_info: Optional[UpdateInfo] = None
        self._download_progress: float = 0
        self._is_checking: bool = False
        self._is_downloading: bool = False
        self._is_installing: bool = False
        
        # 回调函数
        self._on_update_available: Optional[Callable] = None
        self._on_download_progress: Optional[Callable] = None
        self._on_download_complete: Optional[Callable] = None
        self._on_install_complete: Optional[Callable] = None
        self._on_error: Optional[Callable] = None
        
        self._init_checker()
        self._initialized = True
    
    def _init_checker(self):
        """初始化更新检查器"""
        if not UPDATE_CHECKER_AVAILABLE:
            return
        
        self._checker = get_update_checker()
        
        # 设置进度回调
        def on_progress(progress: float):
            self._download_progress = progress
            if self._on_download_progress:
                self._on_download_progress(progress)
        
        self._checker.set_progress_callback(on_progress)
    
    def configure(self, config: AutoUpdateConfig):
        """配置更新器"""
        self._config = config
        
        if self._checker and config.update_url:
            self._checker.set_update_url(config.update_url)
            self._checker.SOFTWARE_CODE = config.software_code
    
    def set_callbacks(self,
                      on_update_available: Callable = None,
                      on_download_progress: Callable = None,
                      on_download_complete: Callable = None,
                      on_install_complete: Callable = None,
                      on_error: Callable = None):
        """设置回调函数"""
        self._on_update_available = on_update_available
        self._on_download_progress = on_download_progress
        self._on_download_complete = on_download_complete
        self._on_install_complete = on_install_complete
        self._on_error = on_error
    
    def check_for_updates(self, callback: Callable = None):
        """
        检查更新（异步）
        
        Args:
            callback: 检查完成回调，参数为 UpdateInfo 或 None
        """
        if not self._checker or not self._config.enabled:
            if callback:
                callback(None)
            return
        
        if self._is_checking:
            logger.debug("正在检查更新中...")
            return
        
        self._is_checking = True
        
        def on_check_complete(update_info: Optional[UpdateInfo]):
            self._is_checking = False
            self._update_info = update_info
            
            if update_info and self._on_update_available:
                self._on_update_available(update_info)
            
            if callback:
                callback(update_info)
        
        self._checker.check_for_updates(callback=on_check_complete)
    
    def download_update(self, callback: Callable = None):
        """
        下载更新（异步）
        
        Args:
            callback: 下载完成回调，参数为 bool（是否成功）
        """
        if not self._checker or not self._update_info:
            if callback:
                callback(False)
            return
        
        if self._is_downloading:
            logger.debug("正在下载更新中...")
            return
        
        self._is_downloading = True
        
        def on_download_complete(success: bool):
            self._is_downloading = False
            
            if self._on_download_complete:
                self._on_download_complete(success)
            
            if callback:
                callback(success)
        
        self._checker.download_and_apply_updates(callback=on_download_complete)
    
    def install_update(self, restart: bool = True):
        """
        安装更新
        
        Args:
            restart: 安装完成后是否重启应用
        """
        if self._is_installing:
            return
        
        self._is_installing = True
        
        try:
            # 增量更新已经在下载时完成了文件替换
            # 这里只需要处理重启逻辑
            
            if restart:
                self._restart_application()
            
            if self._on_install_complete:
                self._on_install_complete(True)
                
        except Exception as e:
            logger.error(f"安装更新失败: {e}")
            if self._on_error:
                self._on_error(str(e))
        finally:
            self._is_installing = False
    
    def download_installer(self, installer_url: str, callback: Callable = None):
        """
        下载安装包（用于大版本更新）
        
        Args:
            installer_url: 安装包下载地址
            callback: 下载完成回调，参数为本地文件路径或 None
        """
        import requests
        
        def download_task():
            try:
                # 下载到临时目录
                app_dir = self._get_app_dir()
                temp_dir = app_dir / ".update_temp"
                temp_dir.mkdir(exist_ok=True)
                
                filename = installer_url.split("/")[-1]
                local_path = temp_dir / filename
                
                logger.info(f"下载安装包: {installer_url}")
                
                response = requests.get(installer_url, stream=True, timeout=300)
                response.raise_for_status()
                
                total_size = int(response.headers.get('content-length', 0))
                downloaded = 0
                
                with open(local_path, 'wb') as f:
                    for chunk in response.iter_content(chunk_size=8192):
                        f.write(chunk)
                        downloaded += len(chunk)
                        if total_size > 0:
                            progress = downloaded / total_size * 100
                            if self._on_download_progress:
                                self._on_download_progress(progress)
                
                logger.info(f"安装包下载完成: {local_path}")
                
                if callback:
                    callback(str(local_path))
                    
            except Exception as e:
                logger.error(f"下载安装包失败: {e}")
                if self._on_error:
                    self._on_error(str(e))
                if callback:
                    callback(None)
        
        thread = threading.Thread(target=download_task, daemon=True)
        thread.start()
    
    def run_installer(self, installer_path: str, silent: bool = False):
        """
        运行安装包
        
        Args:
            installer_path: 安装包路径
            silent: 是否静默安装
        """
        if not os.path.exists(installer_path):
            logger.error(f"安装包不存在: {installer_path}")
            return
        
        try:
            # 构建安装命令
            args = [installer_path]
            if silent:
                args.append("/SILENT")
            
            # 启动安装程序
            logger.info(f"启动安装程序: {' '.join(args)}")
            subprocess.Popen(args, shell=True)
            
            # 退出当前程序
            sys.exit(0)
            
        except Exception as e:
            logger.error(f"启动安装程序失败: {e}")
            if self._on_error:
                self._on_error(str(e))
    
    def rollback(self):
        """回滚到上一版本"""
        if self._checker:
            return self._checker.rollback()
        return False
    
    def _restart_application(self):
        """重启应用程序"""
        try:
            # 获取当前可执行文件路径
            if getattr(sys, 'frozen', False):
                # PyInstaller 打包后
                exe_path = sys.executable
            else:
                # 开发模式
                exe_path = sys.executable
                args = [exe_path] + sys.argv
                subprocess.Popen(args)
                sys.exit(0)
                return
            
            # 启动新进程
            logger.info(f"重启应用: {exe_path}")
            subprocess.Popen([exe_path])
            
            # 退出当前进程
            sys.exit(0)
            
        except Exception as e:
            logger.error(f"重启失败: {e}")
    
    def _get_app_dir(self) -> Path:
        """获取应用目录"""
        if getattr(sys, 'frozen', False):
            return Path(sys.executable).parent
        else:
            return Path(__file__).parent.parent.parent
    
    @property
    def update_info(self) -> Optional[UpdateInfo]:
        """获取更新信息"""
        return self._update_info
    
    @property
    def download_progress(self) -> float:
        """获取下载进度"""
        return self._download_progress
    
    @property
    def is_checking(self) -> bool:
        """是否正在检查"""
        return self._is_checking
    
    @property
    def is_downloading(self) -> bool:
        """是否正在下载"""
        return self._is_downloading
    
    @property
    def current_version(self) -> str:
        """获取当前版本"""
        if self._checker:
            return self._checker.current_version
        return "0.0.0"


# 全局实例
_auto_updater: Optional[AutoUpdater] = None


def get_auto_updater() -> AutoUpdater:
    """获取自动更新器实例"""
    global _auto_updater
    if _auto_updater is None:
        _auto_updater = AutoUpdater()
    return _auto_updater


def check_for_updates_on_startup(
    update_url: str,
    on_update_available: Callable = None,
    on_error: Callable = None
):
    """
    启动时检查更新（便捷函数）
    
    Args:
        update_url: 更新服务器 URL
        on_update_available: 发现更新时的回调
        on_error: 错误回调
    """
    updater = get_auto_updater()
    
    config = AutoUpdateConfig(
        enabled=True,
        check_on_startup=True,
        update_url=update_url
    )
    updater.configure(config)
    updater.set_callbacks(
        on_update_available=on_update_available,
        on_error=on_error
    )
    
    # 异步检查更新
    updater.check_for_updates()
