# -*- coding: utf-8 -*-
"""
动态模块加载器
支持从外部目录加载和热更新 Python 模块

使用场景：
1. 单文件打包后，业务模块放在外部目录，支持增量更新
2. 运行时动态加载/重载模块，无需重启程序
"""

import os
import sys
import logging
import importlib
import importlib.util
from pathlib import Path
from typing import Dict, Any, Optional, List, Callable
from dataclasses import dataclass, field
from datetime import datetime

logger = logging.getLogger(__name__)


@dataclass
class ModuleInfo:
    """模块信息"""
    name: str                    # 模块名
    path: str                    # 文件路径
    loaded_at: datetime = None   # 加载时间
    version: str = "1.0.0"       # 模块版本
    module: Any = None           # 模块对象
    file_hash: str = ""          # 文件哈希（用于检测更新）


class ModuleLoader:
    """动态模块加载器"""
    
    _instance = None
    
    def __new__(cls):
        if cls._instance is None:
            cls._instance = super(ModuleLoader, cls).__new__(cls)
            cls._instance._initialized = False
        return cls._instance
    
    def __init__(self):
        if self._initialized:
            return
        
        self._modules: Dict[str, ModuleInfo] = {}
        self._module_dirs: List[str] = []
        self._on_reload_callbacks: Dict[str, List[Callable]] = {}
        
        # 初始化默认模块目录
        self._init_module_dirs()
        self._initialized = True
    
    def _init_module_dirs(self):
        """初始化模块搜索目录"""
        # 获取应用目录
        if getattr(sys, 'frozen', False):
            # PyInstaller 打包后
            app_dir = Path(sys.executable).parent
        else:
            # 开发模式
            app_dir = Path(__file__).parent.parent.parent
        
        # 默认模块目录
        default_dirs = [
            app_dir / "modules",      # 外部可更新模块
            app_dir / "plugins",      # 插件目录
            app_dir / "extensions",   # 扩展目录
        ]
        
        for d in default_dirs:
            if d.exists():
                self.add_module_dir(str(d))
    
    def add_module_dir(self, path: str):
        """添加模块搜索目录"""
        abs_path = os.path.abspath(path)
        if abs_path not in self._module_dirs:
            self._module_dirs.append(abs_path)
            # 添加到 sys.path
            if abs_path not in sys.path:
                sys.path.insert(0, abs_path)
            logger.info(f"添加模块目录: {abs_path}")
    
    def load_module(self, module_name: str, reload: bool = False) -> Optional[Any]:
        """
        加载模块
        
        Args:
            module_name: 模块名（如 'live_handler' 或 'modules.live_handler'）
            reload: 是否强制重载
        
        Returns:
            加载的模块对象，失败返回 None
        """
        # 检查是否已加载
        if module_name in self._modules and not reload:
            return self._modules[module_name].module
        
        try:
            # 尝试从模块目录加载
            module_path = self._find_module_path(module_name)
            
            if module_path:
                # 从文件路径加载
                module = self._load_from_path(module_name, module_path, reload)
            else:
                # 尝试标准导入
                if reload and module_name in sys.modules:
                    module = importlib.reload(sys.modules[module_name])
                else:
                    module = importlib.import_module(module_name)
                module_path = getattr(module, '__file__', '')
            
            if module:
                # 记录模块信息
                self._modules[module_name] = ModuleInfo(
                    name=module_name,
                    path=module_path or '',
                    loaded_at=datetime.now(),
                    version=getattr(module, '__version__', '1.0.0'),
                    module=module,
                    file_hash=self._calculate_hash(module_path) if module_path else ''
                )
                
                logger.info(f"模块加载成功: {module_name}")
                
                # 触发重载回调
                if reload and module_name in self._on_reload_callbacks:
                    for callback in self._on_reload_callbacks[module_name]:
                        try:
                            callback(module)
                        except Exception as e:
                            logger.error(f"重载回调执行失败: {e}")
                
                return module
            
        except Exception as e:
            logger.error(f"模块加载失败 {module_name}: {e}")
        
        return None
    
    def _find_module_path(self, module_name: str) -> Optional[str]:
        """在模块目录中查找模块文件"""
        # 处理带点的模块名（如 modules.live_handler）
        parts = module_name.split('.')
        relative_path = os.path.join(*parts) + '.py'
        
        for module_dir in self._module_dirs:
            # 尝试直接匹配
            full_path = os.path.join(module_dir, relative_path)
            if os.path.isfile(full_path):
                return full_path
            
            # 尝试作为包
            pkg_path = os.path.join(module_dir, *parts, '__init__.py')
            if os.path.isfile(pkg_path):
                return pkg_path
            
            # 尝试简单名称
            simple_path = os.path.join(module_dir, parts[-1] + '.py')
            if os.path.isfile(simple_path):
                return simple_path
        
        return None
    
    def _load_from_path(self, module_name: str, path: str, reload: bool = False) -> Optional[Any]:
        """从指定路径加载模块"""
        try:
            spec = importlib.util.spec_from_file_location(module_name, path)
            if spec is None or spec.loader is None:
                return None
            
            # 如果重载，先移除旧模块
            if reload and module_name in sys.modules:
                del sys.modules[module_name]
            
            module = importlib.util.module_from_spec(spec)
            sys.modules[module_name] = module
            spec.loader.exec_module(module)
            
            return module
            
        except Exception as e:
            logger.error(f"从路径加载模块失败 {path}: {e}")
            return None
    
    def reload_module(self, module_name: str) -> Optional[Any]:
        """重载模块"""
        return self.load_module(module_name, reload=True)
    
    def reload_all(self) -> Dict[str, bool]:
        """重载所有已加载的模块"""
        results = {}
        for name in list(self._modules.keys()):
            module = self.reload_module(name)
            results[name] = module is not None
        return results
    
    def check_updates(self) -> List[str]:
        """
        检查哪些模块文件已更新
        
        Returns:
            需要重载的模块名列表
        """
        updated = []
        for name, info in self._modules.items():
            if info.path and os.path.isfile(info.path):
                current_hash = self._calculate_hash(info.path)
                if current_hash != info.file_hash:
                    updated.append(name)
        return updated
    
    def auto_reload_updated(self) -> Dict[str, bool]:
        """自动重载已更新的模块"""
        updated = self.check_updates()
        results = {}
        for name in updated:
            module = self.reload_module(name)
            results[name] = module is not None
        return results
    
    def on_reload(self, module_name: str, callback: Callable):
        """注册模块重载回调"""
        if module_name not in self._on_reload_callbacks:
            self._on_reload_callbacks[module_name] = []
        self._on_reload_callbacks[module_name].append(callback)
    
    def get_module(self, module_name: str) -> Optional[Any]:
        """获取已加载的模块"""
        info = self._modules.get(module_name)
        return info.module if info else None
    
    def get_module_info(self, module_name: str) -> Optional[ModuleInfo]:
        """获取模块信息"""
        return self._modules.get(module_name)
    
    def list_modules(self) -> List[ModuleInfo]:
        """列出所有已加载的模块"""
        return list(self._modules.values())
    
    def unload_module(self, module_name: str):
        """卸载模块"""
        if module_name in self._modules:
            del self._modules[module_name]
        if module_name in sys.modules:
            del sys.modules[module_name]
        logger.info(f"模块已卸载: {module_name}")
    
    def _calculate_hash(self, filepath: str) -> str:
        """计算文件哈希"""
        if not filepath or not os.path.isfile(filepath):
            return ""
        
        import hashlib
        sha256 = hashlib.sha256()
        try:
            with open(filepath, 'rb') as f:
                for chunk in iter(lambda: f.read(8192), b''):
                    sha256.update(chunk)
            return sha256.hexdigest()
        except:
            return ""
    
    def scan_modules(self, pattern: str = "*.py") -> List[str]:
        """
        扫描模块目录中的所有模块
        
        Args:
            pattern: 文件匹配模式
        
        Returns:
            可加载的模块名列表
        """
        from glob import glob
        
        modules = []
        for module_dir in self._module_dirs:
            for filepath in glob(os.path.join(module_dir, pattern)):
                if os.path.basename(filepath).startswith('_'):
                    continue
                module_name = os.path.splitext(os.path.basename(filepath))[0]
                modules.append(module_name)
        
        return list(set(modules))


# 全局实例
_module_loader: Optional[ModuleLoader] = None


def get_module_loader() -> ModuleLoader:
    """获取模块加载器实例"""
    global _module_loader
    if _module_loader is None:
        _module_loader = ModuleLoader()
    return _module_loader


def load_module(module_name: str, reload: bool = False) -> Optional[Any]:
    """便捷函数：加载模块"""
    return get_module_loader().load_module(module_name, reload)


def reload_module(module_name: str) -> Optional[Any]:
    """便捷函数：重载模块"""
    return get_module_loader().reload_module(module_name)
