# -*- coding: utf-8 -*-
"""
启动时更新检查
集成到应用启动流程，在主界面显示前检查更新

使用方法（在 app_flet.py 中）:
    from src.core.startup_update import check_update_on_startup
    
    def main(page: ft.Page):
        # 启动时检查更新
        check_update_on_startup(page, UPDATE_URL)
        
        # ... 其他初始化代码
"""

import logging
from typing import Optional, Callable
import threading

logger = logging.getLogger(__name__)

# 延迟导入避免循环依赖
_flet_module = None
_update_dialog_module = None


def _get_flet():
    global _flet_module
    if _flet_module is None:
        import flet as ft
        _flet_module = ft
    return _flet_module


def _get_update_dialog():
    global _update_dialog_module
    if _update_dialog_module is None:
        from gui.update_dialog import UpdateDialog, UpdateDialogInfo
        _update_dialog_module = (UpdateDialog, UpdateDialogInfo)
    return _update_dialog_module


def check_update_on_startup(
    page,  # ft.Page
    update_url: str,
    software_code: str = "DOUYIN_LIVE_ASSISTANT",
    on_complete: Callable = None,
    delay_seconds: float = 2.0,
):
    """
    启动时检查更新（非阻塞）
    
    Args:
        page: Flet 页面对象
        update_url: 更新服务器 URL
        software_code: 软件编码
        on_complete: 检查完成回调
        delay_seconds: 延迟检查时间（秒），给主界面加载时间
    """
    from .auto_updater import get_auto_updater, AutoUpdateConfig
    
    def do_check():
        try:
            # 配置更新器
            updater = get_auto_updater()
            config = AutoUpdateConfig(
                enabled=True,
                update_url=update_url,
                software_code=software_code,
            )
            updater.configure(config)
            
            logger.info(f"检查更新: {update_url}")
            
            # 检查更新
            def on_update_found(update_info):
                if update_info:
                    logger.info(f"发现新版本: {update_info.version}")
                    # 在主线程显示更新弹窗
                    page.run_thread(lambda: _show_update_dialog(page, updater, update_info))
                else:
                    logger.info("当前已是最新版本")
                
                if on_complete:
                    on_complete(update_info)
            
            updater.check_for_updates(callback=on_update_found)
            
        except Exception as e:
            logger.error(f"检查更新失败: {e}")
            if on_complete:
                on_complete(None)
    
    # 延迟检查，避免阻塞启动
    import time
    def delayed_check():
        time.sleep(delay_seconds)
        do_check()
    
    thread = threading.Thread(target=delayed_check, daemon=True)
    thread.start()


def _show_update_dialog(page, updater, update_info):
    """显示更新弹窗"""
    try:
        UpdateDialog, UpdateDialogInfo = _get_update_dialog()
        
        dialog = UpdateDialog(page)
        info = UpdateDialogInfo(
            version=update_info.version,
            changelog=update_info.changelog,
            total_size=update_info.total_size,
            current_version=updater.current_version,
            force_update=False,  # 从 update_info 获取
        )
        
        def on_confirm():
            # 开始下载
            def on_progress(progress):
                try:
                    dialog.update_progress(progress)
                except:
                    pass
            
            def on_download_complete(success):
                if success:
                    dialog.show_complete(True, "更新完成，点击重启生效")
                else:
                    dialog.show_complete(False, "下载失败，请重试")
            
            updater.set_callbacks(
                on_download_progress=on_progress,
                on_download_complete=on_download_complete,
            )
            updater.download_update(callback=on_download_complete)
        
        def on_skip():
            logger.info("用户跳过更新")
        
        dialog.show(info, on_confirm=on_confirm, on_skip=on_skip)
        
    except Exception as e:
        logger.error(f"显示更新弹窗失败: {e}")


def integrate_update_check(app_main_func: Callable) -> Callable:
    """
    装饰器：为 Flet 应用添加启动更新检查
    
    使用方法:
        @integrate_update_check
        def main(page: ft.Page):
            # 应用主逻辑
            pass
        
        ft.app(target=main)
    """
    def wrapper(page):
        # 从配置读取更新 URL
        update_url = get_update_url_from_config()
        
        if update_url:
            check_update_on_startup(page, update_url)
        
        # 执行原始主函数
        return app_main_func(page)
    
    return wrapper


def get_update_url_from_config() -> Optional[str]:
    """从配置文件读取更新 URL（基于 server.api_base_url 构建，必须配置）"""
    import sys
    import yaml
    from pathlib import Path
    
    # 确定根目录（兼容打包环境）
    if getattr(sys, 'frozen', False):
        root_dir = Path(sys._MEIPASS)
    else:
        root_dir = Path(__file__).parent.parent.parent.absolute()
    
    # 用户配置目录
    user_config_dir = Path.home() / ".douyin_live_assistant"
    
    config_paths = [
        root_dir / "config" / "config.yaml",
        user_config_dir / "config.yaml",
    ]
    
    for config_path in config_paths:
        if config_path.exists():
            try:
                with open(config_path, 'r', encoding='utf-8') as f:
                    config = yaml.safe_load(f)
                    if config and 'server' in config:
                        url = config['server'].get('api_base_url')
                        if url:
                            return f"{url.rstrip('/')}/biz/license/client"
            except Exception as e:
                logger.error(f"读取配置文件失败 {config_path}: {e}")
    
    logger.error(f"未找到 API 配置，请在 config/config.yaml 中配置 server.api_base_url，已检查路径: {config_paths}")
    return None
