# -*- coding: utf-8 -*-
"""
更新服务端 API
可部署到任何支持 Python 的服务器（如 Flask/FastAPI）

功能：
1. 提供版本检查接口
2. 提供文件清单接口
3. 提供文件下载接口
4. 支持增量更新

部署方式：
1. 独立 Flask 服务
2. 集成到现有后端（如 Spring Boot 通过静态资源）
3. 使用 CDN/OSS 静态托管
"""

import os
import json
import hashlib
from datetime import datetime
from typing import Dict, List, Optional
from dataclasses import dataclass, asdict
import logging

logger = logging.getLogger(__name__)


@dataclass
class UpdateConfig:
    """更新服务配置"""
    # 更新文件存储目录
    updates_dir: str = "./updates"
    # 当前最新版本
    latest_version: str = "1.0.0"
    # 更新服务基础 URL
    base_url: str = "https://your-server.com/updates"
    # 是否强制更新
    force_update: bool = False
    # 更新公告
    announcement: str = ""


class UpdateServerAPI:
    """
    更新服务 API 实现
    
    可以：
    1. 作为独立 Flask 应用运行
    2. 生成静态 JSON 文件托管到 CDN/OSS
    """
    
    def __init__(self, config: UpdateConfig = None):
        self.config = config or UpdateConfig()
        self.updates_dir = self.config.updates_dir
        os.makedirs(self.updates_dir, exist_ok=True)
        
    def calculate_file_hash(self, filepath: str) -> str:
        """计算文件 SHA256"""
        sha256 = hashlib.sha256()
        with open(filepath, 'rb') as f:
            for chunk in iter(lambda: f.read(8192), b''):
                sha256.update(chunk)
        return sha256.hexdigest()
    
    def generate_version_manifest(self, version_dir: str, version: str) -> Dict:
        """
        为指定版本生成文件清单
        
        Args:
            version_dir: 版本文件目录
            version: 版本号
            
        Returns:
            清单字典
        """
        manifest = {
            "version": version,
            "generated": datetime.now().isoformat(),
            "files": {}
        }
        
        for root, dirs, files in os.walk(version_dir):
            # 排除隐藏目录
            dirs[:] = [d for d in dirs if not d.startswith('.')]
            
            for filename in files:
                if filename.startswith('.'):
                    continue
                    
                filepath = os.path.join(root, filename)
                rel_path = os.path.relpath(filepath, version_dir)
                # 统一使用正斜杠
                rel_path = rel_path.replace('\\', '/')
                
                try:
                    stat = os.stat(filepath)
                    manifest["files"][rel_path] = {
                        "hash": self.calculate_file_hash(filepath),
                        "size": stat.st_size,
                        "modified": datetime.fromtimestamp(stat.st_mtime).isoformat()
                    }
                except Exception as e:
                    logger.warning(f"处理文件失败 {rel_path}: {e}")
        
        return manifest
    
    def get_version_info(self) -> Dict:
        """
        获取版本信息接口
        
        客户端调用: GET /api/updates/version
        
        Returns:
            {
                "latest_version": "1.2.0",
                "force_update": false,
                "announcement": "新增美颜功能",
                "download_url": "https://..."
            }
        """
        return {
            "latest_version": self.config.latest_version,
            "force_update": self.config.force_update,
            "announcement": self.config.announcement,
            "manifest_url": f"{self.config.base_url}/manifest.json",
            "files_base_url": f"{self.config.base_url}/files"
        }
    
    def get_manifest(self, version: str = None) -> Optional[Dict]:
        """
        获取指定版本的文件清单
        
        客户端调用: GET /api/updates/manifest?version=1.2.0
        
        Args:
            version: 版本号，默认为最新版本
            
        Returns:
            文件清单字典
        """
        version = version or self.config.latest_version
        manifest_path = os.path.join(self.updates_dir, version, "manifest.json")
        
        if os.path.exists(manifest_path):
            with open(manifest_path, 'r', encoding='utf-8') as f:
                return json.load(f)
        return None
    
    def get_diff(self, client_version: str, target_version: str = None) -> Dict:
        """
        获取两个版本之间的差异
        
        客户端调用: GET /api/updates/diff?from=1.0.0&to=1.2.0
        
        Args:
            client_version: 客户端当前版本
            target_version: 目标版本，默认为最新版本
            
        Returns:
            {
                "from_version": "1.0.0",
                "to_version": "1.2.0",
                "added": [...],
                "modified": [...],
                "deleted": [...],
                "total_size": 12345678
            }
        """
        target_version = target_version or self.config.latest_version
        
        client_manifest = self.get_manifest(client_version)
        target_manifest = self.get_manifest(target_version)
        
        if not target_manifest:
            return {"error": f"Target version {target_version} not found"}
        
        # 如果客户端清单不存在，返回完整更新
        if not client_manifest:
            client_files = {}
        else:
            client_files = client_manifest.get("files", {})
        
        target_files = target_manifest.get("files", {})
        
        added = []
        modified = []
        deleted = []
        total_size = 0
        
        # 检查新增和修改的文件
        for path, info in target_files.items():
            if path not in client_files:
                added.append({
                    "path": path,
                    "hash": info["hash"],
                    "size": info["size"]
                })
                total_size += info["size"]
            elif client_files[path]["hash"] != info["hash"]:
                modified.append({
                    "path": path,
                    "hash": info["hash"],
                    "size": info["size"]
                })
                total_size += info["size"]
        
        # 检查删除的文件
        for path in client_files:
            if path not in target_files:
                deleted.append(path)
        
        return {
            "from_version": client_version,
            "to_version": target_version,
            "added": added,
            "modified": modified,
            "deleted": deleted,
            "total_size": total_size,
            "files_base_url": f"{self.config.base_url}/{target_version}/files"
        }
    
    def publish_version(self, source_dir: str, version: str, changelog: str = "") -> str:
        """
        发布新版本
        
        Args:
            source_dir: 源文件目录（打包后的目录）
            version: 版本号
            changelog: 更新日志
            
        Returns:
            发布目录路径
        """
        import shutil
        
        # 创建版本目录
        version_dir = os.path.join(self.updates_dir, version)
        files_dir = os.path.join(version_dir, "files")
        
        if os.path.exists(version_dir):
            shutil.rmtree(version_dir)
        
        os.makedirs(files_dir)
        
        # 复制文件
        for root, dirs, files in os.walk(source_dir):
            dirs[:] = [d for d in dirs if not d.startswith('.') and d not in ['__pycache__', 'build', 'dist']]
            
            for filename in files:
                if filename.startswith('.') or filename.endswith('.pyc'):
                    continue
                    
                src_path = os.path.join(root, filename)
                rel_path = os.path.relpath(src_path, source_dir)
                dst_path = os.path.join(files_dir, rel_path)
                
                os.makedirs(os.path.dirname(dst_path), exist_ok=True)
                shutil.copy2(src_path, dst_path)
        
        # 生成清单
        manifest = self.generate_version_manifest(files_dir, version)
        manifest["changelog"] = changelog
        
        manifest_path = os.path.join(version_dir, "manifest.json")
        with open(manifest_path, 'w', encoding='utf-8') as f:
            json.dump(manifest, f, indent=2, ensure_ascii=False)
        
        # 更新最新版本配置
        self.config.latest_version = version
        
        # 保存版本信息
        version_info_path = os.path.join(version_dir, "version.json")
        with open(version_info_path, 'w', encoding='utf-8') as f:
            json.dump(self.get_version_info(), f, indent=2, ensure_ascii=False)
        
        logger.info(f"版本 {version} 发布成功: {version_dir}")
        return version_dir


# ============ Flask 应用示例 ============

def create_flask_app(config: UpdateConfig = None):
    """
    创建 Flask 应用
    
    使用方式：
        from update_server import create_flask_app, UpdateConfig
        
        config = UpdateConfig(
            updates_dir="./updates",
            latest_version="1.2.0",
            base_url="https://your-server.com/updates"
        )
        app = create_flask_app(config)
        app.run(host='0.0.0.0', port=5000)
    """
    try:
        from flask import Flask, jsonify, request, send_from_directory
    except ImportError:
        logger.error("Flask not installed. Run: pip install flask")
        return None
    
    app = Flask(__name__)
    api = UpdateServerAPI(config)
    
    @app.route('/api/updates/version', methods=['GET'])
    def get_version():
        """获取最新版本信息"""
        return jsonify(api.get_version_info())
    
    @app.route('/api/updates/manifest', methods=['GET'])
    def get_manifest():
        """获取版本清单"""
        version = request.args.get('version')
        manifest = api.get_manifest(version)
        if manifest:
            return jsonify(manifest)
        return jsonify({"error": "Version not found"}), 404
    
    @app.route('/api/updates/diff', methods=['GET'])
    def get_diff():
        """获取版本差异"""
        from_version = request.args.get('from', '0.0.0')
        to_version = request.args.get('to')
        return jsonify(api.get_diff(from_version, to_version))
    
    @app.route('/updates/<version>/files/<path:filepath>', methods=['GET'])
    def download_file(version, filepath):
        """下载更新文件"""
        files_dir = os.path.join(api.updates_dir, version, "files")
        return send_from_directory(files_dir, filepath)
    
    return app


# ============ 静态文件生成（用于 CDN/OSS 托管）============

def generate_static_update_files(source_dir: str, output_dir: str, version: str, 
                                  base_url: str, changelog: str = ""):
    """
    生成静态更新文件，可直接上传到 CDN/OSS
    
    Args:
        source_dir: 源文件目录
        output_dir: 输出目录
        version: 版本号
        base_url: CDN/OSS 基础 URL
        changelog: 更新日志
    
    生成结构:
        output_dir/
        ├── version.json          # 版本信息
        ├── manifest.json         # 文件清单
        └── files/                # 更新文件
            ├── src/...
            ├── gui/...
            └── ...
    """
    config = UpdateConfig(
        updates_dir=output_dir,
        latest_version=version,
        base_url=base_url
    )
    
    api = UpdateServerAPI(config)
    api.publish_version(source_dir, version, changelog)
    
    print(f"静态更新文件已生成: {output_dir}")
    print(f"上传到 CDN/OSS 后，客户端访问: {base_url}/version.json")
    
    return output_dir


if __name__ == "__main__":
    # 示例：启动 Flask 服务
    config = UpdateConfig(
        updates_dir="./updates",
        latest_version="1.0.0",
        base_url="http://localhost:5000/updates"
    )
    
    app = create_flask_app(config)
    if app:
        print("更新服务启动: http://localhost:5000")
        print("API 端点:")
        print("  GET /api/updates/version - 获取版本信息")
        print("  GET /api/updates/manifest - 获取文件清单")
        print("  GET /api/updates/diff?from=1.0.0 - 获取版本差异")
        app.run(host='0.0.0.0', port=5000, debug=True)
