# -*- coding: utf-8 -*-
"""
代理模式弹幕客户端
通过DouyinBarrageGrab WebSocket代理获取弹幕
更稳定，推荐生产环境使用
"""

import json
import asyncio
from typing import Optional, Dict, Any

import websockets

from .base import BaseDanmakuClient
from src.utils.events import (
    EventType, User, Gift, Event,
    ChatEvent, GiftEvent, EntranceEvent, FollowEvent, LikeEvent, RoomStatsEvent
)
from src.utils.logger import logger


class ProxyDanmakuClient(BaseDanmakuClient):
    """
    代理模式弹幕客户端
    连接到DouyinBarrageGrab的WebSocket服务
    """
    
    # 消息类型映射
    MSG_TYPES = {
        1: "弹幕消息",
        2: "点赞消息",
        3: "进直播间",
        4: "关注消息",
        5: "礼物消息",
        6: "直播间统计",
        7: "粉丝团消息",
        8: "直播间分享",
        9: "下播",
    }
    
    def __init__(self, room_id: str, ws_url: str = "ws://127.0.0.1:8888"):
        super().__init__(room_id)
        self.ws_url = ws_url
        self._ws: Optional[websockets.WebSocketClientProtocol] = None
        self._reconnect_interval = 5
        self._max_reconnect_times = 10
    
    async def connect(self) -> bool:
        """连接到WebSocket服务"""
        try:
            logger.info(f"正在连接到 {self.ws_url}...")
            self._ws = await websockets.connect(
                self.ws_url,
                ping_interval=None,
                ping_timeout=None
            )
            self.is_connected = True
            logger.info(f"✅ 已连接到弹幕代理服务 {self.ws_url}")
            
            # 触发连接事件
            await self.emit(Event(type=EventType.CONNECTED))
            return True
            
        except Exception as e:
            logger.error(f"连接失败: {e}")
            self.is_connected = False
            return False
    
    async def disconnect(self) -> None:
        """断开连接"""
        self._running = False
        self.is_connected = False
        if self._ws:
            await self._ws.close()
            self._ws = None
        logger.info("已断开连接")
    
    async def start(self) -> None:
        """开始监听弹幕"""
        self._running = True
        reconnect_count = 0
        
        while self._running:
            try:
                if not self.is_connected:
                    if not await self.connect():
                        reconnect_count += 1
                        if reconnect_count >= self._max_reconnect_times:
                            logger.error(f"已达到最大重连次数 {self._max_reconnect_times}，停止重连")
                            break
                        logger.warning(f"将在 {self._reconnect_interval} 秒后重试... ({reconnect_count}/{self._max_reconnect_times})")
                        await asyncio.sleep(self._reconnect_interval)
                        continue
                    else:
                        reconnect_count = 0
                
                # 接收消息
                async for message in self._ws:
                    await self._handle_message(message)
                    
            except websockets.exceptions.ConnectionClosed as e:
                logger.warning(f"连接已关闭: {e}")
                self.is_connected = False
                await self.emit(Event(type=EventType.DISCONNECTED))
                
            except Exception as e:
                logger.error(f"发生错误: {e}")
                self.is_connected = False
                
            if self._running:
                await asyncio.sleep(self._reconnect_interval)
    
    async def _handle_message(self, message: str) -> None:
        """处理接收到的消息"""
        try:
            data = json.loads(message)
            
            if "Type" not in data:
                logger.debug(f"收到无效消息: {message[:100]}")
                return
            
            msg_type = data["Type"]
            
            # 下播消息
            if msg_type == 9:
                logger.warning("📴 直播已结束")
                await self.emit(Event(type=EventType.LIVE_END))
                return
            
            # 解析Data字段
            if "Data" not in data:
                return
            
            try:
                data_dict = json.loads(data["Data"])
            except json.JSONDecodeError:
                logger.debug(f"无法解析Data字段")
                return
            
            # 根据消息类型处理
            handlers = {
                1: self._handle_chat,
                2: self._handle_like,
                3: self._handle_entrance,
                4: self._handle_follow,
                5: self._handle_gift,
                6: self._handle_stats,
            }
            
            handler = handlers.get(msg_type)
            if handler:
                await handler(data_dict)
                
        except json.JSONDecodeError:
            logger.debug(f"无效的JSON: {message[:100]}")
        except Exception as e:
            logger.error(f"处理消息错误: {e}")
    
    async def _handle_chat(self, data: Dict[str, Any]) -> None:
        """处理弹幕消息"""
        user = self._parse_user(data.get("User", {}))
        content = data.get("Content", "")
        
        event = ChatEvent(user=user, content=content)
        logger.info(f"💬 [{user.nickname}]: {content}")
        await self.emit(event)
    
    async def _handle_like(self, data: Dict[str, Any]) -> None:
        """处理点赞消息"""
        user = self._parse_user(data.get("User", {}))
        count = data.get("Count", 0)
        total = data.get("Total", 0)
        
        event = LikeEvent(user=user, count=count, total=total)
        logger.debug(f"👍 {user.nickname} 点赞 x{count}")
        await self.emit(event)
    
    async def _handle_entrance(self, data: Dict[str, Any]) -> None:
        """处理进入直播间消息"""
        user = self._parse_user(data.get("User", {}))
        
        event = EntranceEvent(user=user, content="进入直播间")
        logger.info(f"🚪 {user.nickname} 进入直播间")
        await self.emit(event)
    
    async def _handle_follow(self, data: Dict[str, Any]) -> None:
        """处理关注消息"""
        user = self._parse_user(data.get("User", {}))
        
        event = FollowEvent(user=user, content="关注了主播")
        logger.info(f"➕ {user.nickname} 关注了主播")
        await self.emit(event)
    
    async def _handle_gift(self, data: Dict[str, Any]) -> None:
        """处理礼物消息"""
        user = self._parse_user(data.get("User", {}))
        
        gift = Gift(
            id=str(data.get("GiftId", "")),
            name=data.get("GiftName", "礼物"),
            count=data.get("GiftCount", 1) or data.get("RepeatCount", 1),
            price=data.get("DiamondCount", 0)
        )
        
        event = GiftEvent(user=user, gift=gift)
        logger.info(f"🎁 {user.nickname} 送出 {gift.name} x{gift.count}")
        await self.emit(event)
    
    async def _handle_stats(self, data: Dict[str, Any]) -> None:
        """处理直播间统计"""
        event = RoomStatsEvent(
            online_count=data.get("OnlineUserCount", 0),
            total_count=data.get("TotalUserCount", 0)
        )
        logger.debug(f"📊 在线: {event.online_count}, 累计: {event.total_count}")
        await self.emit(event)
    
    def _parse_user(self, user_data: Dict[str, Any]) -> User:
        """解析用户信息"""
        return User(
            id=str(user_data.get("Id", user_data.get("UserId", ""))),
            nickname=user_data.get("Nickname", user_data.get("NickName", "匿名用户")),
            avatar=user_data.get("HeadImgUrl", ""),
            gender=user_data.get("Gender", 0),
            level=user_data.get("Level", 0)
        )
