# -*- coding: utf-8 -*-
"""
抖音直播弹幕发送模块
使用 DrissionPage 实现 - 可接管用户已登录的浏览器，不会被检测

增强功能:
- 支持人类化打字模拟（防检测）
- 随机延迟 + 正态分布
- 偶尔打错并删除
"""
import json
import logging
import time
import random
import sys
from pathlib import Path
from typing import Optional
from datetime import datetime

logger = logging.getLogger(__name__)

# ========== 日志配置 ==========
# 日志文件路径：用户目录/.douyin_live_assistant/logs/danmu_sender.log
LOG_DIR = Path.home() / '.douyin_live_assistant' / 'logs'
LOG_FILE = LOG_DIR / 'danmu_sender.log'

# 确保日志目录存在
try:
    LOG_DIR.mkdir(parents=True, exist_ok=True)
except:
    pass

# 创建专用的文件日志器
_file_logger = None

def _get_file_logger():
    """获取文件日志器（单例）"""
    global _file_logger
    if _file_logger is None:
        _file_logger = logging.getLogger('danmu_sender_file')
        _file_logger.setLevel(logging.DEBUG)
        _file_logger.propagate = False  # 不传播到父日志器
        
        # 清除已有的处理器
        _file_logger.handlers.clear()
        
        try:
            # 文件处理器 - 追加模式，UTF-8编码
            file_handler = logging.FileHandler(
                LOG_FILE, 
                mode='a', 
                encoding='utf-8'
            )
            file_handler.setLevel(logging.DEBUG)
            
            # 日志格式
            formatter = logging.Formatter(
                '[%(asctime)s.%(msecs)03d][%(levelname)s] %(message)s',
                datefmt='%Y-%m-%d %H:%M:%S'
            )
            file_handler.setFormatter(formatter)
            _file_logger.addHandler(file_handler)
        except Exception as e:
            print(f"[警告] 无法创建日志文件: {e}")
    
    return _file_logger

def danmu_log(msg, level="INFO", module="弹幕发送"):
    """
    写入日志到文件
    
    Args:
        msg: 日志消息
        level: 日志级别 (DEBUG, INFO, WARN, ERROR)
        module: 模块名称
    """
    try:
        file_logger = _get_file_logger()
        full_msg = f"[{module}] {msg}"
        
        if level == "DEBUG":
            file_logger.debug(full_msg)
        elif level == "INFO":
            file_logger.info(full_msg)
        elif level == "WARN":
            file_logger.warning(full_msg)
        elif level == "ERROR":
            file_logger.error(full_msg)
        else:
            file_logger.info(full_msg)
    except:
        pass  # 日志失败不影响主流程

def get_log_file_path() -> str:
    """获取日志文件路径"""
    return str(LOG_FILE)

# 尝试导入防检测键盘模块
try:
    from src.utils.keyboard_anti_detection import KeyboardAntiDetection, TypingConfig
    ANTI_DETECTION_AVAILABLE = True
except ImportError:
    ANTI_DETECTION_AVAILABLE = False
    logger.warning("键盘防检测模块不可用")

# Cookies 保存路径
COOKIES_FILE = Path(__file__).parent.parent.parent / "config" / "douyin_cookies.json"

# 弹幕发送间隔（秒）- 抖音限制约5秒一条
SEND_INTERVAL = 5.0


class DanmuSender:
    """
    抖音直播弹幕发送器
    使用 DrissionPage 接管浏览器，避免自动化检测
    """
    
    def __init__(self, room_id: str, use_existing: bool = True, chrome_path: str = None):
        """
        初始化弹幕发送器
        
        Args:
            room_id: 直播间ID
            use_existing: 是否接管已打开的浏览器（推荐True，使用已登录的浏览器）
            chrome_path: Chrome 浏览器路径（留空自动查找）
        """
        self.room_id = room_id
        self.use_existing = use_existing
        self.chrome_path = chrome_path
        self._page = None
        self._running = False
        self._logged_in = False
        self._last_send_time = 0  # 上次发送时间
        
        # 人类化打字配置
        self.human_typing_enabled = False  # 默认关闭，使用DrissionPage原生输入更稳定
        self._keyboard = None  # 延迟创建，只有启用人类化打字时才初始化
        
    def start(self):
        """启动浏览器"""
        import os
        import io
        import socket
        import traceback
        
        def log(msg, level="INFO"):
            """写入日志到文件"""
            danmu_log(msg, level, "浏览器启动")
        
        def log_traceback():
            """将 traceback 写入日志"""
            tb_str = io.StringIO()
            traceback.print_exc(file=tb_str)
            log(f"堆栈跟踪:\n{tb_str.getvalue()}", "ERROR")
        
        log("=" * 60)
        log("开始启动浏览器流程")
        log("=" * 60)
        
        # 打印环境信息
        log(f"Python 版本: {sys.version}")
        log(f"操作系统: {os.name} / {sys.platform}")
        log(f"当前工作目录: {os.getcwd()}")
        log(f"room_id: {self.room_id}")
        log(f"use_existing: {self.use_existing}")
        log(f"chrome_path 参数: {self.chrome_path}")
        log(f"用户目录: {Path.home()}")
        
        # 检查 DrissionPage
        log("-" * 40)
        log("步骤1: 检查 DrissionPage 模块")
        try:
            from DrissionPage import ChromiumPage, ChromiumOptions
            try:
                import DrissionPage
                dp_version = getattr(DrissionPage, '__version__', '未知')
                log(f"✓ DrissionPage 已安装, 版本: {dp_version}")
            except:
                log("✓ DrissionPage 已安装, 版本: 无法获取")
        except ImportError as e:
            log(f"✗ DrissionPage 未安装: {e}", "ERROR")
            log("请运行: pip install DrissionPage", "ERROR")
            return False
        except Exception as e:
            log(f"✗ 导入 DrissionPage 出错: {e}", "ERROR")
            log_traceback()
            return False
        
        try:
            # 端口检测函数
            def is_port_in_use(port):
                """检查端口是否被占用"""
                try:
                    with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as s:
                        s.settimeout(1)
                        result = s.connect_ex(('127.0.0.1', port))
                        return result == 0
                except Exception as e:
                    log(f"端口检测异常 {port}: {e}", "WARN")
                    return False
            
            # 检查端口状态
            log("-" * 40)
            log("步骤2: 检查端口状态")
            port_9222_used = is_port_in_use(9222)
            port_9333_used = is_port_in_use(9333)
            log(f"端口 9222 状态: {'已占用' if port_9222_used else '空闲'}")
            log(f"端口 9333 状态: {'已占用' if port_9333_used else '空闲'}")
            
            # 1. 优先尝试接管已打开的浏览器
            if port_9222_used:
                log("-" * 40)
                log("步骤3a: 尝试接管端口9222的浏览器")
                try:
                    log("创建 ChromiumOptions...")
                    co = ChromiumOptions()
                    co.set_local_port(9222)
                    log("设置超时时间: base=5, page_load=10")
                    co.set_timeouts(base=5, page_load=10)
                    
                    log("创建 ChromiumPage 连接...")
                    page = ChromiumPage(addr_or_opts=co)
                    log(f"ChromiumPage 创建完成, page={page}")
                    
                    page_url = None
                    try:
                        page_url = page.url if page else None
                        log(f"页面 URL: {page_url}")
                    except Exception as e:
                        log(f"获取页面 URL 失败: {e}", "WARN")
                    
                    if page and page_url:
                        self._page = page
                        log("✓ 成功接管现有浏览器 (端口9222)")
                        
                        # 设置窗口大小
                        try:
                            page.set.window.size(1280, 800)
                            log("✓ 已调整窗口大小为 1280x800")
                        except Exception as e:
                            log(f"调整窗口大小失败: {e}", "WARN")
                        
                        # 判断是否需要跳转
                        if "live.douyin.com" in page_url:
                            log(f"✓ 已在直播间页面: {page_url}")
                        elif self.room_id and self.room_id not in page_url:
                            url = f"https://live.douyin.com/{self.room_id}"
                            log(f"跳转到直播间: {url}")
                            self._page.get(url)
                        
                        self._running = True
                        self._check_login()
                        log("=" * 60)
                        log("✓ 浏览器启动成功 (接管模式)")
                        log("=" * 60)
                        return True
                    else:
                        log("接管的浏览器无有效页面，尝试关闭...")
                        try:
                            if page:
                                page.quit()
                                log("已关闭无效页面对象")
                        except Exception as e:
                            log(f"关闭页面对象失败: {e}", "WARN")
                        log("将启动新浏览器")
                except Exception as e:
                    log(f"✗ 接管浏览器失败: {e}", "ERROR")
                    log(f"异常类型: {type(e).__name__}")
                    log_traceback()
                    log("将启动新浏览器")
            else:
                log("-" * 40)
                log("步骤3a: 跳过接管（端口9222无浏览器监听）")

            # 2. 启动新浏览器
            log("-" * 40)
            log("步骤3b: 启动新浏览器")
            
            # 查找浏览器路径
            log("查找 Chrome/Edge 浏览器路径...")
            chrome_path_to_use = None
            
            # 用户指定路径
            if self.chrome_path:
                normalized_path = os.path.normpath(self.chrome_path)
                log(f"检查用户指定路径: {normalized_path}")
                if os.path.exists(normalized_path):
                    chrome_path_to_use = normalized_path
                    log(f"✓ 用户指定路径有效")
                else:
                    log(f"✗ 用户指定路径不存在", "WARN")
            
            # 自动查找
            if not chrome_path_to_use:
                log("自动查找浏览器路径...")
                chrome_paths = [
                    r"C:\Program Files\Google\Chrome\Application\chrome.exe",
                    r"C:\Program Files (x86)\Google\Chrome\Application\chrome.exe",
                    str(Path.home() / "AppData" / "Local" / "Google" / "Chrome" / "Application" / "chrome.exe"),
                    r"C:\Program Files (x86)\Microsoft\Edge\Application\msedge.exe",
                    r"C:\Program Files\Microsoft\Edge\Application\msedge.exe",
                ]
                
                for path in chrome_paths:
                    exists = os.path.exists(path)
                    log(f"  检查: {path} -> {'存在' if exists else '不存在'}")
                    if exists and not chrome_path_to_use:
                        chrome_path_to_use = path
                        log(f"✓ 选择浏览器: {path}")
            
            if not chrome_path_to_use:
                log("✗ 未找到任何可用的浏览器！", "ERROR")
                log("请安装 Chrome 或 Edge 浏览器", "ERROR")
                return False
            
            log(f"使用浏览器: {chrome_path_to_use}")
            
            # 检查用户数据目录
            user_data_dir = Path.home() / '.douyin_browser'
            log(f"用户数据目录: {user_data_dir}")
            try:
                user_data_dir.mkdir(exist_ok=True)
                log(f"✓ 用户数据目录已就绪")
            except Exception as e:
                log(f"✗ 创建用户数据目录失败: {e}", "ERROR")
                return False
            
            # 配置 ChromiumOptions
            log("配置 ChromiumOptions...")
            co = ChromiumOptions()
            co.set_argument('--window-size=1280,800')
            co.headless(False)
            co.set_local_port(9333)
            co.set_argument('--remote-debugging-port=9333')
            co.set_browser_path(chrome_path_to_use)
            co.set_user_data_path(str(user_data_dir))
            
            # 添加一些调试参数
            co.set_argument('--no-first-run')
            co.set_argument('--no-default-browser-check')
            co.set_argument('--disable-popup-blocking')
            
            log("ChromiumOptions 配置完成:")
            log(f"  - 窗口大小: 1280x800")
            log(f"  - headless: False")
            log(f"  - 调试端口: 9333")
            log(f"  - 浏览器路径: {chrome_path_to_use}")
            log(f"  - 用户数据目录: {user_data_dir}")
            
            # 创建页面
            log("-" * 40)
            log("步骤4: 创建 ChromiumPage")
            log("正在启动浏览器进程，请稍候...")
            
            start_time = time.time()
            try:
                self._page = ChromiumPage(addr_or_opts=co)
                elapsed = time.time() - start_time
                log(f"✓ ChromiumPage 创建成功 (耗时: {elapsed:.2f}秒)")
            except Exception as e:
                elapsed = time.time() - start_time
                log(f"✗ ChromiumPage 创建失败 (耗时: {elapsed:.2f}秒)", "ERROR")
                log(f"异常类型: {type(e).__name__}", "ERROR")
                log(f"异常信息: {e}", "ERROR")
                log_traceback()
                return False
            
            # 打开直播间
            log("-" * 40)
            log("步骤5: 打开直播间页面")
            url = f"https://live.douyin.com/{self.room_id}"
            log(f"目标 URL: {url}")
            
            try:
                start_time = time.time()
                self._page.get(url)
                elapsed = time.time() - start_time
                log(f"✓ 页面加载完成 (耗时: {elapsed:.2f}秒)")
                
                # 获取当前 URL 确认
                try:
                    current_url = self._page.url
                    log(f"当前页面 URL: {current_url}")
                except:
                    pass
                    
            except Exception as e:
                log(f"✗ 页面加载失败: {e}", "ERROR")
                log_traceback()
                return False
            
            self._running = True
            
            # 等待页面稳定
            log("等待页面稳定 (3秒)...")
            time.sleep(3)
            
            # 检查登录状态
            log("-" * 40)
            log("步骤6: 检查登录状态")
            self._check_login()
            
            log("=" * 60)
            log("✓ 浏览器启动流程完成")
            log(f"  - 运行状态: {self._running}")
            log(f"  - 登录状态: {self._logged_in}")
            log("=" * 60)
            
            return True
            
        except Exception as e:
            log(f"✗ 启动浏览器过程中发生未预期的异常", "ERROR")
            log(f"异常类型: {type(e).__name__}", "ERROR")
            log(f"异常信息: {e}", "ERROR")
            import io
            import traceback as tb
            # 将 traceback 写入日志
            tb_str = io.StringIO()
            tb.print_exc(file=tb_str)
            log(f"堆栈跟踪:\n{tb_str.getvalue()}", "ERROR")
            return False
        
    def _check_login(self):
        """检查登录状态"""
        import traceback
        import io
        
        def log(msg, level="INFO"):
            """写入日志到文件"""
            danmu_log(msg, level, "登录检测")
        
        log("开始检查登录状态...")
        
        try:
            # 先检查页面对象是否有效
            if not self._page:
                log("页面对象为空，无法检查登录状态", "ERROR")
                self._logged_in = False
                return
            
            # 尝试获取当前页面URL
            try:
                current_url = self._page.url
                log(f"当前页面 URL: {current_url}")
            except Exception as e:
                log(f"获取页面 URL 失败: {e}", "WARN")
            
            # 最准确的方法：检查页面上是否有"需先登录"文本
            log("查找'需先登录，才能开始聊天'文本...")
            try:
                need_login = self._page.ele('text:需先登录，才能开始聊天', timeout=2)
                log(f"查找结果: {need_login}")
                if need_login:
                    self._logged_in = False
                    log("✗ 页面显示'需先登录，才能开始聊天'，判定为未登录")
                    return
            except Exception as e:
                log(f"查找'需先登录，才能开始聊天'时异常: {e}", "WARN")
            
            # 检查是否有"需先登录"提示
            log("查找'需先登录'文本...")
            try:
                need_login2 = self._page.ele('text:需先登录', timeout=1)
                log(f"查找结果: {need_login2}")
                if need_login2:
                    self._logged_in = False
                    log("✗ 检测到'需先登录'提示，判定为未登录")
                    return
            except Exception as e:
                log(f"查找'需先登录'时异常: {e}", "WARN")
            
            # 如果没有"需先登录"提示，认为已登录
            self._logged_in = True
            log("✓ 未检测到登录提示，判定为已登录")
            
            # 已登录时自动最小化浏览器
            self._minimize_browser()
            
        except Exception as e:
            # 如果检测出错，假定可能已登录，让用户尝试发送
            self._logged_in = True
            log(f"检查登录状态时发生异常: {e}", "ERROR")
            log(f"异常类型: {type(e).__name__}", "ERROR")
            # 将 traceback 写入日志
            tb_str = io.StringIO()
            traceback.print_exc(file=tb_str)
            log(f"堆栈跟踪:\n{tb_str.getvalue()}", "ERROR")
            log("由于异常，默认假定已登录，让用户尝试发送")
    
    def _minimize_browser(self):
        """最小化浏览器窗口"""
        import sys
        try:
            if self._page:
                self._page.set.window.mini()
                print("[弹幕发送] ✓ 浏览器已最小化")
                sys.stdout.flush()
        except Exception as e:
            print(f"[弹幕发送] 最小化浏览器失败: {e}")
            sys.stdout.flush()
    
    def refresh_page(self, new_room_id: str = None):
        """刷新浏览器页面（用于切换主播ID）"""
        import sys
        try:
            if not self._page:
                print("[弹幕发送] 浏览器未启动，无法刷新")
                sys.stdout.flush()
                return False
            
            # 如果提供了新的房间ID，更新并跳转
            if new_room_id and new_room_id != self.room_id:
                self.room_id = new_room_id
                url = f"https://live.douyin.com/{new_room_id}"
                print(f"[弹幕发送] 正在切换直播间: {url}")
                sys.stdout.flush()
                self._page.get(url)
            else:
                # 刷新当前页面
                print("[弹幕发送] 正在刷新页面...")
                sys.stdout.flush()
                self._page.refresh()
            
            # 恢复窗口并等待页面加载
            try:
                self._page.set.window.normal()
            except:
                pass
            
            time.sleep(2)
            self._check_login()
            print("[弹幕发送] ✓ 页面已刷新")
            sys.stdout.flush()
            return True
        except Exception as e:
            print(f"[弹幕发送] 刷新页面失败: {e}")
            sys.stdout.flush()
            return False
            
    def wait_for_login(self, timeout: int = 180) -> bool:
        """
        等待用户登录
        
        Args:
            timeout: 超时时间（秒）
        """
        import sys
        if self._logged_in:
            return True
            
        print(f"[弹幕发送] 等待登录，超时时间 {timeout} 秒...")
        print("[弹幕发送] 请在弹出的浏览器中扫码或输入验证码登录抖音")
        sys.stdout.flush()
        
        try:
            # 等待登录完成
            start_time = time.time()
            check_count = 0
            while (time.time() - start_time) < timeout:
                check_count += 1
                try:
                    # 检查是否还有"需先登录"提示
                    need_login = self._page.ele('text:需先登录，才能开始聊天', timeout=1)
                    
                    if check_count % 5 == 0:  # 每10秒打印一次
                        print(f"[弹幕发送] 检查登录状态... (需登录提示={bool(need_login)})")
                        sys.stdout.flush()
                    
                    if not need_login:
                        # 没有登录提示了，说明已登录
                        self._logged_in = True
                        print("[弹幕发送] ✓ 登录成功！")
                        sys.stdout.flush()
                        
                        # 登录成功后自动最小化浏览器
                        self._minimize_browser()
                        return True
                        
                except Exception as e:
                    if check_count % 5 == 0:
                        print(f"[弹幕发送] 检查异常: {e}")
                        sys.stdout.flush()
                
                time.sleep(2)
            
            print("[弹幕发送] ✗ 登录超时")
            sys.stdout.flush()
            return False
            
        except Exception as e:
            print(f"[弹幕发送] 等待登录失败: {e}")
            sys.stdout.flush()
            return False
            
    def send(self, message: str) -> bool:
        """
        发送弹幕
        
        Args:
            message: 弹幕内容
            
        Returns:
            是否发送成功
        """
        import sys
        if not self._running or not self._page:
            print("[弹幕发送] 发送器未启动")
            return False
            
        if not self._logged_in:
            print("[弹幕发送] 未登录，无法发送弹幕")
            return False
        
        # 检查发送频率限制
        now = time.time()
        elapsed = now - self._last_send_time
        if elapsed < SEND_INTERVAL:
            wait_time = SEND_INTERVAL - elapsed
            print(f"[弹幕发送] 发送太频繁，等待 {wait_time:.1f} 秒...")
            sys.stdout.flush()
            time.sleep(wait_time)
            
        try:
            # 先检查是否真的已登录（页面上没有"需先登录"提示）
            need_login = self._page.ele('text:需先登录，才能开始聊天', timeout=1)
            if need_login:
                print("[弹幕发送] ✗ 页面显示需要登录，无法发送弹幕")
                print("[弹幕发送] 请先在浏览器中登录抖音账号")
                sys.stdout.flush()
                return False
            
            # ---------------------------------------------------------
            # 优化策略：直接使用坐标点击（最快、最稳）
            # ---------------------------------------------------------
            textarea = None
            
            try:
                # 获取窗口大小
                win_size = self._page.run_js("return {w: window.innerWidth, h: window.innerHeight}")

                if win_size:
                    w, h = win_size['w'], win_size['h']
                    # 输入框通常在右下角，大概位置：右侧 20% 区域，底部 50px 区域
                    # 尝试点击几个可能的点
                    click_points = [
                        (w - 150, h - 30), # 右下角 (首选)
                        (w - 200, h - 30),
                        (w - 100, h - 30),
                    ]
                    
                    try:
                        from DrissionPage.common import Keys
                    except ImportError:
                        class Keys:
                            ENTER = '\n'
                    
                    for x, y in click_points:
                        # 使用 JS 模拟点击
                        self._page.run_js(f"document.elementFromPoint({x}, {y}).click();")
                        time.sleep(0.2)
                        
                        # 检查是否有元素获得焦点
                        active_ele = self._page.run_js("return document.activeElement;")
                        if active_ele:
                            tag = active_ele.tag
                            editable = active_ele.attr('contenteditable')
                            
                            if tag == 'textarea' or editable == 'true' or tag == 'input':
                                textarea = active_ele
                                
                                # 直接在这里处理输入和发送，因为已经激活了
                                try:
                                    # 清空输入框
                                    self._page.run_js("document.activeElement.innerText = '';")
                                    time.sleep(0.1)
                                    
                                    # 使用人类化打字或普通输入
                                    if self.human_typing_enabled and self._keyboard:
                                        try:
                                            print(f"[弹幕发送] 使用人类化打字模式...")
                                            sys.stdout.flush()
                                            self._human_type_text(message)
                                        except Exception as typing_err:
                                            print(f"[弹幕发送] 人类化打字失败，回退到普通模式: {typing_err}")
                                            textarea.input(message)
                                    else:
                                        # 普通输入
                                        textarea.input(message)
                                    
                                    time.sleep(0.1)
                                    
                                    # 发送
                                    textarea.input(Keys.ENTER)
                                    print(f"[弹幕发送] ✓ 弹幕已发送: {message}")
                                    sys.stdout.flush()
                                    
                                    # 更新发送时间
                                    self._last_send_time = time.time()
                                    return True
                                    
                                except Exception as input_err:
                                    print(f"[弹幕发送] 输入/发送失败: {input_err}")
                                    
                                break
            except Exception as e:
                print(f"[弹幕发送] 坐标点击失败: {e}")

            if not textarea:
                print("[弹幕发送] ✗ 找不到弹幕输入框")
                print("[弹幕发送] 可能的原因：1.未登录 2.页面未完全加载 3.直播间不支持弹幕")
                sys.stdout.flush()
                return False
            
            return True # 如果上面已经发送成功，这里不会执行到
            
        except Exception as e:
            danmu_log(f"✗ 发送弹幕失败: {e}", "ERROR", "弹幕发送")
            import traceback
            import io
            tb_str = io.StringIO()
            traceback.print_exc(file=tb_str)
            danmu_log(f"堆栈跟踪:\n{tb_str.getvalue()}", "ERROR", "弹幕发送")
            return False
            
    def _human_type_text(self, text: str):
        """
        人类化打字输入
        模拟真实人类打字，包含随机延迟、偶尔打错等特征
        """
        import sys
        
        if not self._keyboard:
            # 如果没有键盘模块，使用简单的随机延迟
            for char in text:
                # 模拟按键
                try:
                    import pyautogui
                    pyautogui.press(char) if len(char) == 1 else None
                except:
                    pass
                
                # 随机延迟 (50-150ms)
                delay = random.gauss(0.08, 0.03)
                delay = max(0.03, min(delay, 0.2))
                time.sleep(delay)
            return
        
        # 使用防检测键盘模块
        try:
            self._keyboard.type_text(text)
        except Exception as e:
            print(f"[弹幕发送] 人类化打字失败: {e}，回退到普通输入")
            sys.stdout.flush()
            # 回退到简单随机延迟
            for char in text:
                try:
                    import pyautogui
                    pyautogui.typewrite(char, interval=random.uniform(0.05, 0.15))
                except:
                    pass
    
    def set_human_typing(self, enabled: bool, speed: str = "normal"):
        """
        设置人类化打字模式
        
        Args:
            enabled: 是否启用
            speed: 速度 ("slow", "normal", "fast")
        """
        self.human_typing_enabled = enabled
        print(f"[弹幕发送] 人类化打字: {'开启' if enabled else '关闭'}")
        
        # 只有启用时才创建键盘防检测实例（延迟初始化）
        if enabled and self._keyboard is None and ANTI_DETECTION_AVAILABLE:
            try:
                config = TypingConfig(
                    wpm=80,
                    error_rate=0.02,
                    pause_rate=0.03,
                    use_interception=True
                )
                self._keyboard = KeyboardAntiDetection(config)
                print("[弹幕发送] 键盘防检测模块已初始化")
            except Exception as e:
                print(f"[弹幕发送] 键盘防检测初始化失败: {e}")
        
        if self._keyboard:
            try:
                self._keyboard.set_speed(speed)
                status = self._keyboard.get_status()
                print(f"[弹幕发送] 输入方法: {status.get('输入方法', '未知')}")
                print(f"[弹幕发送] 打字速度: {status.get('打字速度', '未知')}")
            except Exception as e:
                print(f"[弹幕发送] 设置打字速度出错: {e}")
    
    def stop(self):
        """停止发送器（不关闭浏览器）"""
        self._running = False
        # 注意：不关闭浏览器，让用户继续使用
        logger.info("弹幕发送器已停止（浏览器保持打开）")
        
    @property
    def is_logged_in(self) -> bool:
        return self._logged_in
        
    @property
    def is_running(self) -> bool:
        return self._running


class DanmuSenderThread:
    """弹幕发送器的线程包装"""
    
    def __init__(self, room_id: str, use_existing: bool = True, chrome_path: str = None):
        self.room_id = room_id
        self.use_existing = use_existing
        self.chrome_path = chrome_path
        self._sender: Optional[DanmuSender] = None
        self._started = False
        
    def start(self):
        """启动发送器"""
        self._sender = DanmuSender(self.room_id, self.use_existing, self.chrome_path)
        result = self._sender.start()
        self._started = result
        return result
        
    def wait_for_login(self, timeout: int = 180) -> bool:
        """等待登录"""
        if not self._sender:
            return False
        return self._sender.wait_for_login(timeout)
        
    def send(self, message: str) -> bool:
        """发送弹幕"""
        if not self._sender:
            return False
        return self._sender.send(message)
            
    def stop(self):
        """停止"""
        if self._sender:
            self._sender.stop()
    
    def refresh_page(self, new_room_id: str = None):
        """刷新浏览器页面"""
        if not self._sender:
            return False
        if new_room_id:
            self.room_id = new_room_id
        return self._sender.refresh_page(new_room_id)
            
    @property
    def is_logged_in(self) -> bool:
        return self._sender.is_logged_in if self._sender else False


# 全局实例
_sender_instance: Optional[DanmuSenderThread] = None


def get_danmu_sender(room_id: str = None, use_existing: bool = True, chrome_path: str = None) -> Optional[DanmuSenderThread]:
    """获取弹幕发送器实例"""
    global _sender_instance
    
    if _sender_instance is None and room_id:
        _sender_instance = DanmuSenderThread(room_id, use_existing, chrome_path)
        
    return _sender_instance


def init_danmu_sender(room_id: str, use_existing: bool = True, chrome_path: str = None) -> DanmuSenderThread:
    """初始化弹幕发送器"""
    global _sender_instance
    
    if _sender_instance:
        _sender_instance.stop()
        
    _sender_instance = DanmuSenderThread(room_id, use_existing, chrome_path)
    return _sender_instance


def has_saved_cookies() -> bool:
    """检查是否有保存的 cookies"""
    return COOKIES_FILE.exists()
