# -*- coding: utf-8 -*-
from PyQt6.QtWidgets import QMainWindow, QWidget
from PyQt6.QtWebEngineWidgets import QWebEngineView
from PyQt6.QtWebEngineCore import QWebEnginePage
from PyQt6.QtCore import Qt, QUrl, QPoint, QEvent, QTimer, pyqtSignal
from PyQt6.QtGui import QColor, QCursor, QRegion

import sys

class TransparentWebPage(QWebEnginePage):
    """自定义 WebPage 以支持透明背景，并输出 JS 控制台日志"""
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setBackgroundColor(QColor(0, 0, 0, 0))
        
    def javaScriptConsoleMessage(self, level, message, lineNumber, sourceID):
        # 将 JS 控制台日志输出到 Python 控制台
        print(f"[Live2D Console] {message} (Line {lineNumber})")

class DragOverlay(QWidget):
    """透明拖动覆盖层"""
    def __init__(self, parent):
        super().__init__(parent)
        self.parent_window = parent
        self.setAttribute(Qt.WidgetAttribute.WA_TransparentForMouseEvents, False)
        self.setStyleSheet("background: transparent;")
        self._drag_pos = None
        
    def mousePressEvent(self, event):
        if event.button() == Qt.MouseButton.LeftButton:
            self._drag_pos = event.globalPosition().toPoint() - self.parent_window.frameGeometry().topLeft()
            self.setCursor(QCursor(Qt.CursorShape.ClosedHandCursor))
            event.accept()
        else:
            event.ignore()
            
    def mouseMoveEvent(self, event):
        if event.buttons() == Qt.MouseButton.LeftButton and self._drag_pos:
            self.parent_window.move(event.globalPosition().toPoint() - self._drag_pos)
            event.accept()
        else:
            event.ignore()
            
    def mouseReleaseEvent(self, event):
        self._drag_pos = None
        self.setCursor(QCursor(Qt.CursorShape.ArrowCursor))
        event.ignore()
    
    def mouseDoubleClickEvent(self, event):
        event.ignore()
    
    def wheelEvent(self, event):
        # 转发滚轮事件给父窗口
        self.parent_window.wheelEvent(event)

    def contextMenuEvent(self, event):
        # 转发右键菜单事件给父窗口
        self.parent_window.contextMenuEvent(event)

class AvatarWindow(QMainWindow):
    model_changed = pyqtSignal(str)  # 新增信号
    
    def __init__(self, url="http://127.0.0.1:8080/live2d/index.html", available_models=None):
        super().__init__()
        
        # 可用模型列表
        self._available_models = available_models or ["Hiyori", "Haru"]
        self._current_model_index = 0
        self._current_model = self._available_models[0] if self._available_models else "Hiyori"
        
        # 无边框，置顶，工具窗口
        self.setWindowFlags(
            Qt.WindowType.FramelessWindowHint | 
            Qt.WindowType.WindowStaysOnTopHint | 
            Qt.WindowType.Tool
        )
        
        # 透明背景设置
        self.setAttribute(Qt.WidgetAttribute.WA_TranslucentBackground, True)
        self.setAttribute(Qt.WidgetAttribute.WA_NoSystemBackground, True)
        self.setStyleSheet("background: transparent;")
        
        # 初始尺寸
        self._base_size = 400
        self.resize(self._base_size, int(self._base_size * 1.5))
        
        # 默认放在屏幕右下角
        screen = self.screen().availableGeometry()
        self.move(screen.width() - self._base_size - 50, screen.height() - int(self._base_size * 1.5) - 50)
        
        # 中央 Widget
        central = QWidget(self)
        central.setStyleSheet("background: transparent;")
        self.setCentralWidget(central)
        
        # Web View
        self.webview = QWebEngineView(central)
        self.webview.setStyleSheet("background: transparent;")
        self.webview.setContextMenuPolicy(Qt.ContextMenuPolicy.NoContextMenu)
        
        # 使用自定义透明页面
        page = TransparentWebPage(self.webview)
        self.webview.setPage(page)
        
        self.webview.load(QUrl(url))
        
        # 透明拖动覆盖层
        self.drag_overlay = DragOverlay(self)
        self.drag_overlay.hide() # 默认隐藏（锁定状态）
        
        # 缩放比例
        self._scale = 1.0
        self._min_scale = 0.3
        self._max_scale = 2.0
        
        # 初始布局
        self._update_layout()
        
        # 在 Windows 上设置点击穿透 (延后执行，防止卡死)
        if sys.platform == 'win32':
            QTimer.singleShot(500, self._setup_click_through)
        
    def _setup_click_through(self):
        """设置 Windows 点击穿透"""
        try:
            import ctypes
            from ctypes import wintypes
            
            hwnd = int(self.winId())
            
            # 获取当前扩展样式
            GWL_EXSTYLE = -20
            WS_EX_LAYERED = 0x80000
            WS_EX_TRANSPARENT = 0x20
            
            user32 = ctypes.windll.user32
            current_style = user32.GetWindowLongW(hwnd, GWL_EXSTYLE)
            
            # 添加透明和穿透样式
            new_style = current_style | WS_EX_LAYERED | WS_EX_TRANSPARENT
            user32.SetWindowLongW(hwnd, GWL_EXSTYLE, new_style)
            
            self._click_through_enabled = True
        except Exception as e:
            print(f"设置点击穿透失败: {e}")
            self._click_through_enabled = False
    
    def _disable_click_through(self):
        """禁用点击穿透（用于交互）"""
        if sys.platform != 'win32':
            return
        try:
            import ctypes
            hwnd = int(self.winId())
            GWL_EXSTYLE = -20
            WS_EX_TRANSPARENT = 0x20
            
            user32 = ctypes.windll.user32
            current_style = user32.GetWindowLongW(hwnd, GWL_EXSTYLE)
            new_style = current_style & ~WS_EX_TRANSPARENT
            user32.SetWindowLongW(hwnd, GWL_EXSTYLE, new_style)
        except:
            pass
    
    def _enable_click_through(self):
        """启用点击穿透"""
        if sys.platform != 'win32':
            return
        try:
            import ctypes
            hwnd = int(self.winId())
            GWL_EXSTYLE = -20
            WS_EX_TRANSPARENT = 0x20
            
            user32 = ctypes.windll.user32
            current_style = user32.GetWindowLongW(hwnd, GWL_EXSTYLE)
            new_style = current_style | WS_EX_TRANSPARENT
            user32.SetWindowLongW(hwnd, GWL_EXSTYLE, new_style)
        except:
            pass
        
    def resizeEvent(self, event):
        super().resizeEvent(event)
        self._update_layout()
        
    def _update_layout(self):
        """更新子控件布局"""
        size = self.size()
        if hasattr(self, 'webview'):
            self.webview.setGeometry(0, 0, size.width(), size.height())
        if hasattr(self, 'drag_overlay'):
            self.drag_overlay.setGeometry(0, 0, size.width(), size.height())
            self.drag_overlay.raise_()  # 确保在最上层
        
    def load_model(self, model_name):
        base_url = "http://127.0.0.1:8080/live2d/index.html"
        url = f"{base_url}?model={model_name}"
        self._current_model = model_name
        if model_name in self._available_models:
            self._current_model_index = self._available_models.index(model_name)
        self.webview.load(QUrl(url))
        
    def wheelEvent(self, event):
        """滚轮缩放 - 需要先禁用穿透才能接收"""
        delta = event.angleDelta().y()
        
        if delta > 0:
            self._scale = min(self._scale + 0.1, self._max_scale)
        else:
            self._scale = max(self._scale - 0.1, self._min_scale)
        
        new_width = int(self._base_size * self._scale)
        new_height = int(self._base_size * 1.5 * self._scale)
        
        old_center = self.geometry().center()
        self.resize(new_width, new_height)
        new_rect = self.geometry()
        new_rect.moveCenter(old_center)
        self.setGeometry(new_rect)
        
        event.accept()
    
    def contextMenuEvent(self, event):
        """右键菜单"""
        from PyQt6.QtWidgets import QMenu
        from PyQt6.QtGui import QAction
        
        # 临时禁用穿透以显示菜单
        self._disable_click_through()
        
        menu = QMenu(self)
        menu.setWindowFlags(menu.windowFlags() | Qt.WindowType.NoDropShadowWindowHint)
        menu.setStyleSheet("""
            QMenu {
                background-color: #2a2a2a;
                color: #ddd;
                border: 1px solid #444;
                border-radius: 6px;
                padding: 5px;
            }
            QMenu::item {
                padding: 8px 25px;
                border-radius: 4px;
            }
            QMenu::item:selected {
                background-color: #3a3a3a;
            }
            QMenu::separator {
                height: 1px;
                background-color: #444;
                margin: 5px 10px;
            }
        """)
        
        # 锁定/解锁
        if getattr(self, '_click_through_enabled', False):
            lock_action = QAction("🔓 解锁 (可拖动/缩放)", self)
            lock_action.triggered.connect(self._unlock_interaction)
        else:
            lock_action = QAction("🔒 锁定 (点击穿透)", self)
            lock_action.triggered.connect(self._lock_interaction)
        menu.addAction(lock_action)
        
        menu.addSeparator()
        
        # 模型子菜单
        model_menu = menu.addMenu(f"🔄 换个模型 (当前: {self._current_model})")
        model_menu.setStyleSheet(menu.styleSheet())
        
        for model_name in self._available_models:
            action = QAction(model_name, self)
            action.triggered.connect(lambda checked, m=model_name: self._switch_model(m))
            if model_name == self._current_model:
                action.setEnabled(False)
            model_menu.addAction(action)
        
        menu.addSeparator()
        
        reset_action = QAction("📐 重置大小", self)
        reset_action.triggered.connect(self._reset_size)
        menu.addAction(reset_action)
        
        menu.addSeparator()
        
        close_action = QAction("❌ 关闭", self)
        close_action.triggered.connect(self.close)
        menu.addAction(close_action)
        
        menu.exec(event.globalPos())
        
        # 菜单关闭后恢复穿透状态
        if getattr(self, '_click_through_enabled', False):
            QTimer.singleShot(100, self._enable_click_through)
    
    def _lock_interaction(self):
        """锁定交互（启用点击穿透）"""
        self._click_through_enabled = True
        self.drag_overlay.hide()  # 隐藏拖动层
        self._enable_click_through()
    
    def _unlock_interaction(self):
        """解锁交互（禁用点击穿透）"""
        self._click_through_enabled = False
        self._disable_click_through()
        self.drag_overlay.show()  # 显示拖动层
        self.drag_overlay.raise_()
    
    def _reset_size(self):
        """重置为默认大小"""
        self._scale = 1.0
        self.resize(self._base_size, int(self._base_size * 1.5))
    
    def _switch_model(self, model_name):
        """切换到指定模型"""
        if model_name == self._current_model:
            return
        self.load_model(model_name)
        self.model_changed.emit(model_name)
    
    def start_speaking(self, duration=None):
        """开始说话口型动画"""
        if duration:
            js = f"window.startSpeaking({duration});"
        else:
            js = "window.startSpeaking();"
        self.webview.page().runJavaScript(js)
    
    def stop_speaking(self):
        """停止说话口型动画"""
        self.webview.page().runJavaScript("window.stopSpeaking();")
