# -*- coding: utf-8 -*-
"""
独立运行 Live2D 桌面形象窗口
用法: python run_avatar.py [model_name]
"""

import sys
import os
from pathlib import Path

# 处理打包后的路径
if getattr(sys, 'frozen', False):
    # PyInstaller 打包后
    if hasattr(sys, '_MEIPASS'):
        ROOT_DIR = Path(sys._MEIPASS)
    else:
        ROOT_DIR = Path(sys.executable).parent
else:
    # 开发环境
    ROOT_DIR = Path(__file__).parent.parent.parent

sys.path.insert(0, str(ROOT_DIR))

def main():
    try:
        from PyQt6.QtWidgets import QApplication
        from PyQt6.QtGui import QIcon
        
        # 根据环境导入 avatar_window
        if getattr(sys, 'frozen', False):
            # 打包后，avatar_window 已经在路径中
            try:
                from src.gui.avatar_window import AvatarWindow
            except ImportError:
                from avatar_window import AvatarWindow
        else:
            from src.gui.avatar_window import AvatarWindow
    except ImportError as e:
        print(f"错误: 需要安装 PyQt6-WebEngine")
        print(f"运行: pip install PyQt6 PyQt6-WebEngine")
        print(f"详情: {e}")
        sys.exit(1)
    
    # 获取模型名
    model = sys.argv[1] if len(sys.argv) > 1 else "Hiyori"
    
    # 扫描可用模型
    def scan_models():
        try:
            model_dir = ROOT_DIR / "static" / "live2d" / "live2d-model"
            if not model_dir.exists():
                # 尝试其他路径
                model_dir = Path(sys.executable).parent / "static" / "live2d" / "live2d-model"
            models = [p.name for p in model_dir.iterdir() if p.is_dir()]
            if not models:
                return ["Hiyori"]
            return sorted(models)
        except Exception as e:
            print(f"扫描模型失败: {e}")
            return ["Hiyori"]
    
    app = QApplication(sys.argv)
    
    # 设置图标
    icon_path = ROOT_DIR / "static" / "icons" / "icon.ico"
    if not icon_path.exists():
        icon_path = Path(sys.executable).parent / "static" / "icons" / "icon.ico"
    if icon_path.exists():
        app.setWindowIcon(QIcon(str(icon_path)))
    
    window = AvatarWindow(available_models=scan_models())
    window.load_model(model)
    window.show()
    
    # 延迟解锁
    from PyQt6.QtCore import QTimer
    QTimer.singleShot(500, window._unlock_interaction)
    
    sys.exit(app.exec())

if __name__ == "__main__":
    main()
