# -*- coding: utf-8 -*-
"""
JS注入模块 - 控制抖音直播伴侣
通过Chrome DevTools Protocol (CDP)控制Electron应用
"""

import os
import re
import json
import time
import shutil
import subprocess
import threading
from pathlib import Path
from typing import Optional, Callable

try:
    import requests
    REQUESTS_AVAILABLE = True
except ImportError:
    REQUESTS_AVAILABLE = False

try:
    import websocket
    WEBSOCKET_AVAILABLE = True
except ImportError:
    WEBSOCKET_AVAILABLE = False


class LiveCompanionController:
    """直播伴侣控制器 - 通过JS注入控制"""
    
    DEBUG_PORT = 9222  # Chrome DevTools调试端口
    
    def __init__(self, companion_path: str, logger: Callable = None):
        self.companion_path = companion_path
        self._logger = logger or print
        self._ws = None
        self._msg_id = 0
    
    def log(self, msg: str):
        print(f"[JSInject] {msg}")
        if self._logger:
            self._logger(msg)
    
    def get_index_js_path(self) -> Optional[str]:
        """获取直播伴侣的index.js路径"""
        if not self.companion_path:
            return None
        
        # 直播伴侣目录结构: xxx/直播伴侣.exe -> xxx/resources/app/index.js
        exe_dir = Path(self.companion_path).parent
        index_js = exe_dir / "resources" / "app" / "index.js"
        
        if index_js.exists():
            return str(index_js)
        
        self.log(f"未找到index.js: {index_js}")
        return None
    
    def backup_index_js(self, index_js_path: str) -> bool:
        """备份index.js"""
        backup_path = index_js_path + ".bak"
        if not os.path.exists(backup_path):
            try:
                shutil.copy2(index_js_path, backup_path)
                self.log(f"已备份: {backup_path}")
                return True
            except Exception as e:
                self.log(f"备份失败: {e}")
                return False
        return True
    
    def inject_debug_port(self, index_js_path: str) -> bool:
        """注入远程调试端口参数，使Electron支持CDP"""
        try:
            # 如果存在备份且当前文件没有完整配置，从备份恢复
            backup_path = index_js_path + ".bak"
            
            with open(index_js_path, 'r', encoding='utf-8') as f:
                content = f.read()
            
            # 检查是否已经注入了完整的配置
            if 'remote-allow-origins' in content and 'remote-debugging-port' in content:
                self.log("调试参数已完整注入")
                return True
            
            # 如果只有部分注入，从备份恢复
            if 'remote-debugging-port' in content and 'remote-allow-origins' not in content:
                if os.path.exists(backup_path):
                    self.log("发现不完整注入，从备份恢复...")
                    with open(backup_path, 'r', encoding='utf-8') as f:
                        content = f.read()
            
            # 备份（如果还没有）
            self.backup_index_js(index_js_path)
            
            # 查找注入点 - 在 no-sandbox 之前插入
            pattern = r'(\w+)\.commandLine\.appendSwitch\("no-sandbox"\)'
            match = re.search(pattern, content)
            
            if match:
                var_name = match.group(1)
                # 注入调试端口和允许所有origin
                inject_code = (
                    f'{var_name}.commandLine.appendSwitch("remote-debugging-port","{self.DEBUG_PORT}"),'
                    f'{var_name}.commandLine.appendSwitch("remote-allow-origins","*"),'
                )
                new_code = inject_code + match.group(0)
                content = content.replace(match.group(0), new_code)
                
                with open(index_js_path, 'w', encoding='utf-8') as f:
                    f.write(content)
                
                self.log(f"已注入调试端口: {self.DEBUG_PORT} (允许所有origin)")
                return True
            else:
                self.log("未找到注入点")
                return False
                
        except Exception as e:
            self.log(f"注入失败: {e}")
            return False
    
    def start_companion_with_debug(self) -> bool:
        """启动带调试端口的直播伴侣"""
        if not self.companion_path or not os.path.exists(self.companion_path):
            self.log("直播伴侣路径无效")
            return False
        
        # 先注入调试端口
        index_js = self.get_index_js_path()
        if index_js:
            self.inject_debug_port(index_js)
        
        # 启动直播伴侣
        try:
            work_dir = os.path.dirname(self.companion_path)
            subprocess.Popen(
                [self.companion_path],
                cwd=work_dir,
                creationflags=subprocess.DETACHED_PROCESS | subprocess.CREATE_NEW_PROCESS_GROUP
            )
            self.log("直播伴侣已启动")
            return True
        except Exception as e:
            self.log(f"启动失败: {e}")
            return False
    
    def wait_for_debug_port(self, timeout: int = 30) -> bool:
        """等待调试端口就绪"""
        if not REQUESTS_AVAILABLE:
            self.log("requests库不可用")
            return False
        
        start_time = time.time()
        while time.time() - start_time < timeout:
            try:
                resp = requests.get(f"http://127.0.0.1:{self.DEBUG_PORT}/json", timeout=2)
                if resp.status_code == 200:
                    self.log("调试端口已就绪")
                    return True
            except:
                pass
            time.sleep(1)
        
        self.log("等待调试端口超时")
        return False
    
    def get_page_ws_url(self) -> Optional[str]:
        """获取页面的WebSocket调试URL"""
        try:
            resp = requests.get(f"http://127.0.0.1:{self.DEBUG_PORT}/json", timeout=5)
            pages = resp.json()
            
            for page in pages:
                # 查找主窗口
                if page.get('type') == 'page':
                    ws_url = page.get('webSocketDebuggerUrl')
                    if ws_url:
                        self.log(f"找到页面: {page.get('title', 'Unknown')}")
                        return ws_url
            
            self.log("未找到可调试页面")
            return None
        except Exception as e:
            self.log(f"获取页面失败: {e}")
            return None
    
    def connect_ws(self, ws_url: str) -> bool:
        """连接WebSocket"""
        if not WEBSOCKET_AVAILABLE:
            self.log("websocket-client库不可用")
            return False
        
        try:
            # 设置origin头解决403问题
            self._ws = websocket.create_connection(
                ws_url, 
                timeout=10,
                suppress_origin=True,  # 不发送origin
                header={
                    "Origin": "http://127.0.0.1:9222"
                }
            )
            self.log("WebSocket已连接")
            return True
        except Exception as e:
            self.log(f"WebSocket连接失败: {e}")
            # 尝试不带header连接
            try:
                self._ws = websocket.create_connection(ws_url, timeout=10, suppress_origin=True)
                self.log("WebSocket已连接(无origin)")
                return True
            except Exception as e2:
                self.log(f"重试连接也失败: {e2}")
            return False
    
    def send_command(self, method: str, params: dict = None) -> dict:
        """发送CDP命令"""
        if not self._ws:
            return {}
        
        self._msg_id += 1
        msg = {
            "id": self._msg_id,
            "method": method,
            "params": params or {}
        }
        
        try:
            self._ws.send(json.dumps(msg))
            result = self._ws.recv()
            return json.loads(result)
        except Exception as e:
            self.log(f"发送命令失败: {e}")
            return {}
    
    def execute_js(self, script: str) -> dict:
        """执行JavaScript代码"""
        return self.send_command("Runtime.evaluate", {
            "expression": script,
            "returnByValue": True
        })
    
    def click_start_button(self) -> bool:
        """点击开始直播按钮"""
        # 查找并点击"开始直播"按钮
        js_code = '''
        (function() {
            // 方法1: 通过文本查找按钮
            const buttons = document.querySelectorAll('button, div[role="button"], span');
            for (const btn of buttons) {
                const text = btn.innerText || btn.textContent || '';
                if (text.includes('开始直播') || text.includes('开播')) {
                    btn.click();
                    return 'clicked: ' + text;
                }
            }
            
            // 方法2: 通过class查找
            const startBtn = document.querySelector('.start-live-btn, .live-start-btn, [class*="start"]');
            if (startBtn) {
                startBtn.click();
                return 'clicked by class';
            }
            
            return 'button not found';
        })();
        '''
        
        result = self.execute_js(js_code)
        self.log(f"点击结果: {result}")
        
        if result.get('result', {}).get('value', '').startswith('clicked'):
            return True
        return False
    
    def auto_start_broadcast(self) -> bool:
        """全自动开播流程"""
        self.log("========== JS注入自动开播 ==========")
        
        # 1. 启动直播伴侣
        self.log("步骤1: 启动直播伴侣...")
        if not self.start_companion_with_debug():
            return False
        
        # 2. 等待调试端口
        self.log("步骤2: 等待调试端口就绪...")
        if not self.wait_for_debug_port(timeout=30):
            self.log("调试端口未就绪，尝试直接连接...")
        
        # 3. 获取WebSocket URL
        self.log("步骤3: 连接调试接口...")
        time.sleep(3)  # 额外等待界面加载
        
        ws_url = self.get_page_ws_url()
        if not ws_url:
            self.log("无法获取调试接口")
            return False
        
        # 4. 连接WebSocket
        if not self.connect_ws(ws_url):
            return False
        
        # 5. 等待页面加载完成
        self.log("步骤4: 等待页面加载...")
        time.sleep(5)
        
        # 6. 点击开始直播按钮
        self.log("步骤5: 点击开始直播...")
        if self.click_start_button():
            self.log("========== 开播成功 ==========")
            return True
        
        self.log("点击按钮失败")
        return False
    
    def close(self):
        """关闭连接"""
        if self._ws:
            try:
                self._ws.close()
            except:
                pass
            self._ws = None


def check_dependencies() -> dict:
    """检查依赖"""
    return {
        'requests': REQUESTS_AVAILABLE,
        'websocket': WEBSOCKET_AVAILABLE,
    }


if __name__ == '__main__':
    print("依赖检查:", check_dependencies())
    
    controller = LiveCompanionController(
        companion_path=r"D:\soft\zhibobanlv\webcast_mate\11.1.3.263047209\直播伴侣.exe"
    )
    controller.auto_start_broadcast()
