# -*- coding: utf-8 -*-
"""
Edge TTS 语音合成引擎
使用微软Edge浏览器的免费TTS服务
"""

import subprocess
import sys
from typing import Optional
from pathlib import Path

import edge_tts

from .base import BaseTTS
from src.utils.logger import logger


def _try_upgrade_edge_tts() -> bool:
    """尝试自动升级 edge-tts 库"""
    try:
        logger.info("正在尝试自动更新 edge-tts 库...")
        result = subprocess.run(
            [sys.executable, "-m", "pip", "install", "--upgrade", "edge-tts"],
            capture_output=True,
            text=True,
            timeout=60
        )
        if result.returncode == 0:
            logger.info("edge-tts 更新成功，请重启程序")
            return True
        else:
            logger.error(f"edge-tts 更新失败: {result.stderr}")
            return False
    except Exception as e:
        logger.error(f"自动更新 edge-tts 异常: {e}")
        return False


class EdgeTTSEngine(BaseTTS):
    """
    Edge TTS 引擎
    
    支持的中文语音 (2024年最新):
    - zh-CN-XiaoxiaoNeural (女声，温暖，推荐)
    - zh-CN-XiaoyiNeural (女声，活泼)
    - zh-CN-YunxiNeural (男声，阳光)
    - zh-CN-YunjianNeural (男声，激情)
    - zh-CN-YunxiaNeural (男声，可爱)
    - zh-CN-YunyangNeural (男声，专业)
    - zh-CN-liaoning-XiaobeiNeural (女声，东北话)
    - zh-CN-shaanxi-XiaoniNeural (女声，陕西话)
    """
    
    # 常用语音列表 (2024年最新)
    VOICES = {
        "xiaoxiao": "zh-CN-XiaoxiaoNeural",   # 女声，温暖，适合新闻/小说
        "xiaoyi": "zh-CN-XiaoyiNeural",       # 女声，活泼，适合动画/小说
        "yunxi": "zh-CN-YunxiNeural",         # 男声，阳光活泼，适合小说
        "yunjian": "zh-CN-YunjianNeural",     # 男声，激情，适合体育/小说
        "yunxia": "zh-CN-YunxiaNeural",       # 男声，可爱，适合动画
        "yunyang": "zh-CN-YunyangNeural",     # 男声，专业，适合新闻播报
        "xiaobei": "zh-CN-liaoning-XiaobeiNeural",  # 女声，东北话
        "xiaoni": "zh-CN-shaanxi-XiaoniNeural",     # 女声，陕西话
    }
    
    def __init__(
        self,
        output_dir: str = "./output",
        voice: str = "zh-CN-XiaoxiaoNeural",
        rate: str = "+0%",
        volume: str = "+0%",
        proxy: str = None
    ):
        """
        初始化Edge TTS引擎
        
        Args:
            output_dir: 输出目录
            voice: 语音角色
            rate: 语速调整 (如 "+10%", "-10%")
            volume: 音量调整
            proxy: 代理地址（可选）
        """
        super().__init__(output_dir)
        
        # 支持简写
        self.voice = self.VOICES.get(voice.lower(), voice)
        self.rate = rate
        self.volume = volume
        self.proxy = proxy if proxy else None
        
        logger.info(f"Edge TTS 初始化: voice={self.voice}, rate={self.rate}")
    
    async def synthesize(self, text: str, filename: str = None) -> Optional[str]:
        """
        合成语音
        
        Args:
            text: 要合成的文本
            filename: 输出文件名（可选）
        
        Returns:
            str: 音频文件路径，失败返回None
        """
        if not text or not text.strip():
            logger.warning("合成文本为空")
            return None
        
        # 清理文本中的特殊字符
        text = self._clean_text(text)
        
        if not filename:
            filename = self._generate_filename("edge")
        
        output_path = self.output_dir / filename
        
        try:
            communicate = edge_tts.Communicate(
                text=text,
                voice=self.voice,
                rate=self.rate,
                volume=self.volume,
                proxy=self.proxy
            )
            
            await communicate.save(str(output_path))
            logger.debug(f"语音合成成功: {output_path}")
            return str(output_path)
            
        except Exception as e:
            error_str = str(e)
            logger.error(f"语音合成失败: {e}")
            
            # 检测 403 错误（Token 过期）
            if "403" in error_str or "Invalid response status" in error_str:
                logger.warning("检测到 Edge TTS Token 过期（403错误）")
                
                # 标记需要更新（避免重复更新）
                if not getattr(self, '_upgrade_attempted', False):
                    self._upgrade_attempted = True
                    if _try_upgrade_edge_tts():
                        logger.info("✓ edge-tts 已自动更新，请重启程序后生效")
                    else:
                        logger.error("✗ 自动更新失败，请手动执行: pip install --upgrade edge-tts")
            
            return None
    
    def _clean_text(self, text: str) -> str:
        """清理文本中的特殊字符"""
        # 移除可能导致问题的字符
        text = text.replace('"', '').replace("'", '')
        text = text.replace('\n', ' ').replace('\r', '')
        # 限制长度
        if len(text) > 500:
            text = text[:500] + "..."
        return text.strip()
    
    @classmethod
    async def list_voices(cls, language: str = "zh") -> list:
        """
        列出可用的语音
        
        Args:
            language: 语言代码筛选
        
        Returns:
            list: 语音列表
        """
        voices = await edge_tts.list_voices()
        if language:
            voices = [v for v in voices if language.lower() in v["Locale"].lower()]
        return voices
