# -*- coding: utf-8 -*-
"""
桌面覆盖层防检测
在整个屏幕上覆盖一个透明的处理层
实时对屏幕添加噪点等效果

原理：创建一个置顶的半透明窗口，覆盖整个屏幕
直播伴侣录屏时会把这个覆盖层一起录进去
"""

import sys
import random
import time
import threading
from typing import Optional

# PyQt6/5
try:
    from PyQt6.QtWidgets import QApplication, QWidget, QLabel
    from PyQt6.QtCore import Qt, QTimer, QPoint
    from PyQt6.QtGui import QPainter, QColor, QPen, QBrush
    PYQT_AVAILABLE = True
except ImportError:
    try:
        from PyQt5.QtWidgets import QApplication, QWidget, QLabel
        from PyQt5.QtCore import Qt, QTimer, QPoint
        from PyQt5.QtGui import QPainter, QColor, QPen, QBrush
        PYQT_AVAILABLE = True
    except ImportError:
        PYQT_AVAILABLE = False
        print("需要 PyQt6 或 PyQt5")


class NoiseOverlay(QWidget):
    """噪点覆盖层窗口"""
    
    def __init__(self, strength: str = "medium"):
        super().__init__()
        
        self.strength = strength
        self.noise_points = []
        self.frame_count = 0
        self.hue_r = 0
        self.hue_g = 0
        self.hue_b = 0
        self.color_shift = 0
        self.brightness_alpha = 0
        
        # 设置窗口属性 (兼容PyQt5/6)
        try:
            # PyQt6
            self.setWindowFlags(
                Qt.WindowType.FramelessWindowHint |
                Qt.WindowType.WindowStaysOnTopHint |
                Qt.WindowType.Tool |
                Qt.WindowType.WindowTransparentForInput
            )
            self.setAttribute(Qt.WidgetAttribute.WA_TranslucentBackground)
            self.setAttribute(Qt.WidgetAttribute.WA_TransparentForMouseEvents)
        except AttributeError:
            # PyQt5
            self.setWindowFlags(
                Qt.FramelessWindowHint |
                Qt.WindowStaysOnTopHint |
                Qt.Tool |
                Qt.WindowTransparentForInput
            )
            self.setAttribute(Qt.WA_TranslucentBackground)
            self.setAttribute(Qt.WA_TransparentForMouseEvents)
        
        # Windows API: 确保窗口完全可穿透
        self._make_click_through()
        
        # 全屏
        screen = QApplication.primaryScreen().geometry()
        self.setGeometry(screen)
        
        # 生成噪点参数
        self._update_noise_params()
        
        # 定时刷新 (每100ms刷新一次 = 10fps)
        self.timer = QTimer()
        self.timer.timeout.connect(self._on_timer)
        self.timer.start(100)
        
        # 状态标签 - 极简
        self.status_label = QLabel(self)
        self.status_label.setStyleSheet("color: rgba(0,200,0,100); font-size: 9px; background: transparent;")
        self.status_label.move(5, 5)
        self.status_label.setText("F0")
        
    def _update_noise_params(self):
        """更新噪点和色彩参数"""
        self.frame_count += 1
        
        # 根据强度决定参数
        if self.strength == "low":
            num_points = 50
            max_size = 2
            max_alpha = 30
            self.color_shift = random.randint(-3, 3)  # 色彩偏移
            self.brightness_alpha = random.randint(0, 5)  # 亮度层透明度
        elif self.strength == "medium":
            num_points = 150
            max_size = 3
            max_alpha = 50
            self.color_shift = random.randint(-5, 5)
            self.brightness_alpha = random.randint(0, 8)
        else:  # high
            num_points = 300
            max_size = 4
            max_alpha = 80
            self.color_shift = random.randint(-10, 10)
            self.brightness_alpha = random.randint(0, 15)
        
        # 随机色调 (每30帧变化一次)
        if self.frame_count % 30 == 0:
            self.hue_r = random.randint(-8, 8)
            self.hue_g = random.randint(-8, 8)
            self.hue_b = random.randint(-8, 8)
        
        # 生成随机噪点
        screen = QApplication.primaryScreen().geometry()
        self.noise_points = []
        for _ in range(num_points):
            x = random.randint(0, screen.width())
            y = random.randint(0, screen.height())
            size = random.randint(1, max_size)
            alpha = random.randint(10, max_alpha)
            # 随机颜色偏移
            r = random.randint(-20, 20)
            g = random.randint(-20, 20)
            b = random.randint(-20, 20)
            self.noise_points.append((x, y, size, alpha, r, g, b))
    
    def _on_timer(self):
        """定时器回调"""
        self._update_noise_params()
        self.status_label.setText(f"F{self.frame_count}")
        self.update()  # 触发重绘
    
    def paintEvent(self, event):
        """绘制噪点和色彩滤镜"""
        painter = QPainter(self)
        try:
            painter.setRenderHint(QPainter.RenderHint.Antialiasing)
            no_pen = Qt.PenStyle.NoPen
        except AttributeError:
            painter.setRenderHint(QPainter.Antialiasing)
            no_pen = Qt.NoPen
        
        # 1. 绘制全屏半透明色彩层 (模拟色调变化)
        if self.brightness_alpha > 0:
            screen = self.geometry()
            hue_color = QColor(
                max(0, min(255, 128 + self.hue_r)),
                max(0, min(255, 128 + self.hue_g)),
                max(0, min(255, 128 + self.hue_b)),
                self.brightness_alpha
            )
            painter.setPen(no_pen)
            painter.setBrush(QBrush(hue_color))
            painter.drawRect(0, 0, screen.width(), screen.height())
        
        # 2. 绘制随机噪点
        for x, y, size, alpha, r, g, b in self.noise_points:
            # 基础灰色 + 随机偏移
            base = 128
            color = QColor(
                max(0, min(255, base + r + self.color_shift)),
                max(0, min(255, base + g + self.color_shift)),
                max(0, min(255, base + b + self.color_shift)),
                alpha
            )
            painter.setPen(no_pen)
            painter.setBrush(QBrush(color))
            painter.drawEllipse(x, y, size, size)
    
    def _make_click_through(self):
        """使用Windows API确保窗口可穿透点击"""
        try:
            import ctypes
            from ctypes import wintypes
            
            # 获取窗口句柄
            hwnd = int(self.winId())
            
            # 获取当前窗口扩展样式
            GWL_EXSTYLE = -20
            WS_EX_LAYERED = 0x80000
            WS_EX_TRANSPARENT = 0x20
            
            user32 = ctypes.windll.user32
            style = user32.GetWindowLongW(hwnd, GWL_EXSTYLE)
            
            # 添加透明和可穿透样式
            user32.SetWindowLongW(hwnd, GWL_EXSTYLE, style | WS_EX_LAYERED | WS_EX_TRANSPARENT)
            print("[覆盖层] 已设置鼠标穿透 (Windows API)")
        except Exception as e:
            print(f"[覆盖层] 设置鼠标穿透失败: {e}")
    
    def set_strength(self, strength: str):
        """设置强度"""
        self.strength = strength
        print(f"[覆盖层] 强度: {strength}")
    
    def keyPressEvent(self, event):
        """按键处理"""
        key = event.key()
        try:
            # PyQt6
            if key == Qt.Key.Key_Q:
                self.close()
            elif key == Qt.Key.Key_1:
                self.set_strength("low")
            elif key == Qt.Key.Key_2:
                self.set_strength("medium")
            elif key == Qt.Key.Key_3:
                self.set_strength("high")
        except AttributeError:
            # PyQt5
            if key == Qt.Key_Q:
                self.close()
            elif key == Qt.Key_1:
                self.set_strength("low")
            elif key == Qt.Key_2:
                self.set_strength("medium")
            elif key == Qt.Key_3:
                self.set_strength("high")


class DesktopAntiDetect:
    """桌面防检测控制器"""
    
    def __init__(self):
        self.app = None
        self.overlay = None
        self.running = False
    
    def start(self, strength: str = "medium"):
        """启动覆盖层"""
        if not PYQT_AVAILABLE:
            print("错误: PyQt5 未安装")
            return False
        
        print("=" * 50)
        print("桌面防检测覆盖层已启动")
        print("=" * 50)
        print("操作: 按 Q 退出 | 1/2/3 切换强度")
        print("=" * 50)
        
        self.app = QApplication.instance() or QApplication(sys.argv)
        self.overlay = NoiseOverlay(strength)
        self.overlay.show()
        self.running = True
        
        return True
    
    def run(self):
        """运行事件循环"""
        if self.app:
            try:
                self.app.exec()  # PyQt6
            except AttributeError:
                self.app.exec_()  # PyQt5
    
    def stop(self):
        """停止"""
        self.running = False
        if self.overlay:
            self.overlay.close()


def main():
    """主函数"""
    import argparse
    parser = argparse.ArgumentParser(description="桌面防检测覆盖层")
    parser.add_argument("--strength", type=str, default="medium", 
                       choices=["low", "medium", "high"], help="覆盖强度")
    args = parser.parse_args()
    
    print(f"正在启动桌面防检测覆盖层 (强度: {args.strength})...")
    
    controller = DesktopAntiDetect()
    if controller.start(args.strength):
        controller.run()
    
    print("覆盖层已关闭")


if __name__ == "__main__":
    main()
