# -*- coding: utf-8 -*-
"""
事件系统模块
提供事件发布/订阅机制
"""

from enum import Enum, auto
from typing import Callable, Dict, List, Any, Optional
from dataclasses import dataclass, field
from datetime import datetime
import asyncio


class EventType(Enum):
    """事件类型枚举"""
    # 连接事件
    CONNECTED = auto()          # 连接成功
    DISCONNECTED = auto()       # 断开连接
    RECONNECTING = auto()       # 重连中
    
    # 直播间事件
    CHAT = auto()               # 聊天弹幕
    GIFT = auto()               # 礼物
    LIKE = auto()               # 点赞
    ENTRANCE = auto()           # 进入直播间
    FOLLOW = auto()             # 关注
    SHARE = auto()              # 分享
    SUBSCRIBE = auto()          # 粉丝团
    
    # 统计事件
    ROOM_STATS = auto()         # 直播间统计
    
    # 直播状态
    LIVE_START = auto()         # 开播
    LIVE_END = auto()           # 下播
    
    # 系统事件
    ERROR = auto()              # 错误


@dataclass
class User:
    """用户信息"""
    id: str = ""
    nickname: str = ""
    avatar: str = ""
    gender: int = 0  # 0未知, 1男, 2女
    level: int = 0
    
    @property
    def gender_str(self) -> str:
        return {0: "未知", 1: "男", 2: "女"}.get(self.gender, "未知")


@dataclass
class Gift:
    """礼物信息"""
    id: str = ""
    name: str = ""
    count: int = 1
    price: int = 0  # 抖币价格
    icon: str = ""


@dataclass
class Event:
    """事件基类"""
    type: EventType
    timestamp: datetime = field(default_factory=datetime.now)
    user: Optional[User] = None
    content: str = ""
    raw_data: Dict[str, Any] = field(default_factory=dict)


@dataclass
class ChatEvent(Event):
    """聊天事件"""
    type: EventType = EventType.CHAT


@dataclass
class GiftEvent(Event):
    """礼物事件"""
    type: EventType = EventType.GIFT
    gift: Optional[Gift] = None


@dataclass
class EntranceEvent(Event):
    """进入直播间事件"""
    type: EventType = EventType.ENTRANCE


@dataclass
class FollowEvent(Event):
    """关注事件"""
    type: EventType = EventType.FOLLOW


@dataclass
class LikeEvent(Event):
    """点赞事件"""
    type: EventType = EventType.LIKE
    count: int = 0
    total: int = 0


@dataclass
class RoomStatsEvent(Event):
    """直播间统计事件"""
    type: EventType = EventType.ROOM_STATS
    online_count: int = 0
    total_count: int = 0
    like_count: int = 0


class EventEmitter:
    """事件发射器"""
    
    def __init__(self):
        self._listeners: Dict[EventType, List[Callable]] = {}
        self._async_listeners: Dict[EventType, List[Callable]] = {}
    
    def on(self, event_type: EventType, callback: Callable) -> None:
        """注册事件监听器"""
        if asyncio.iscoroutinefunction(callback):
            if event_type not in self._async_listeners:
                self._async_listeners[event_type] = []
            self._async_listeners[event_type].append(callback)
        else:
            if event_type not in self._listeners:
                self._listeners[event_type] = []
            self._listeners[event_type].append(callback)
    
    def off(self, event_type: EventType, callback: Callable) -> None:
        """移除事件监听器"""
        if event_type in self._listeners:
            if callback in self._listeners[event_type]:
                self._listeners[event_type].remove(callback)
        if event_type in self._async_listeners:
            if callback in self._async_listeners[event_type]:
                self._async_listeners[event_type].remove(callback)
    
    def emit(self, event: Event) -> None:
        """同步触发事件"""
        event_type = event.type
        if event_type in self._listeners:
            for callback in self._listeners[event_type]:
                try:
                    callback(event)
                except Exception as e:
                    print(f"事件处理错误: {e}")
    
    async def emit_async(self, event: Event) -> None:
        """异步触发事件"""
        event_type = event.type
        
        # 触发同步监听器
        self.emit(event)
        
        # 触发异步监听器
        if event_type in self._async_listeners:
            tasks = []
            for callback in self._async_listeners[event_type]:
                try:
                    tasks.append(callback(event))
                except Exception as e:
                    print(f"异步事件处理错误: {e}")
            
            if tasks:
                await asyncio.gather(*tasks, return_exceptions=True)
    
    def clear(self, event_type: EventType = None) -> None:
        """清除监听器"""
        if event_type:
            self._listeners.pop(event_type, None)
            self._async_listeners.pop(event_type, None)
        else:
            self._listeners.clear()
            self._async_listeners.clear()


# 全局事件发射器
event_emitter = EventEmitter()


def get_event_emitter() -> EventEmitter:
    """获取事件发射器实例"""
    return event_emitter
