# -*- coding: utf-8 -*-
"""
人脸关键点检测器
基于 MediaPipe Face Mesh

提供 478 个 3D 人脸关键点，用于高精度贴纸和大眼瘦脸
"""

import cv2
import numpy as np
import mediapipe as mp
import time
from typing import Optional, List, Tuple, NamedTuple

class FaceLandmarks(NamedTuple):
    """人脸关键点数据"""
    landmarks: np.ndarray  # (N, 2) 像素坐标
    landmarks_3d: np.ndarray  # (N, 3) 归一化坐标
    bbox: Tuple[int, int, int, int]  # x, y, w, h


class FaceLandmarker:
    """MediaPipe 人脸关键点检测封装"""
    
    _instance = None
    
    def __new__(cls):
        if cls._instance is None:
            cls._instance = super(FaceLandmarker, cls).__new__(cls)
            cls._instance._initialized = False
        return cls._instance
    
    def __init__(self):
        if self._initialized:
            return
            
        try:
            self.mp_face_mesh = mp.solutions.face_mesh
            self.face_mesh = self.mp_face_mesh.FaceMesh(
                max_num_faces=1,
                refine_landmarks=True,  # 包含瞳孔关键点 (478个点)
                min_detection_confidence=0.5,
                min_tracking_confidence=0.5
            )
            self.enabled = True
            print("MediaPipe Face Mesh initialized successfully")
        except Exception as e:
            print(f"Failed to initialize MediaPipe: {e}")
            self.enabled = False
            
        self._initialized = True
        
    def process(self, frame: np.ndarray) -> Optional[FaceLandmarks]:
        """处理一帧图像，返回关键点"""
        if not self.enabled:
            return None
            
        try:
            h, w = frame.shape[:2]
            
            # MediaPipe 需要 RGB
            # 注意：输入frame通常已经是RGB（如果在CameraWindow里转过）
            # 但为了安全，我们假设输入可能是BGR或RGB，这里不做转换，由调用者保证RGB
            # 或者我们在这里统一处理，但会有性能损耗。
            # 约定：传入的 frame 必须是 RGB 格式
            
            # 为了性能，将图像标记为不可写
            frame.flags.writeable = False
            results = self.face_mesh.process(frame)
            frame.flags.writeable = True
            
            if not results.multi_face_landmarks:
                return None
                
            # 获取第一个人脸
            face_landmarks = results.multi_face_landmarks[0]
            
            # 转换为 numpy 数组 (像素坐标)
            landmarks_np = np.array([
                (int(lm.x * w), int(lm.y * h)) 
                for lm in face_landmarks.landmark
            ])
            
            # 3D 归一化坐标
            landmarks_3d = np.array([
                (lm.x, lm.y, lm.z) 
                for lm in face_landmarks.landmark
            ])
            
            # 计算边界框
            x_min = np.min(landmarks_np[:, 0])
            x_max = np.max(landmarks_np[:, 0])
            y_min = np.min(landmarks_np[:, 1])
            y_max = np.max(landmarks_np[:, 1])
            
            bbox = (x_min, y_min, x_max - x_min, y_max - y_min)
            
            return FaceLandmarks(landmarks_np, landmarks_3d, bbox)
            
        except Exception as e:
            print(f"Landmark detection error: {e}")
            return None
            
    def close(self):
        """释放资源"""
        if self.enabled:
            self.face_mesh.close()

# 关键点索引常量
class LandmarkIndices:
    # 轮廓
    FACE_OVAL = [10, 338, 297, 332, 284, 251, 389, 356, 454, 323, 361, 288, 397, 365, 379, 378, 400, 377, 152, 148, 176, 149, 150, 136, 172, 58, 132, 93, 234, 127, 162, 21, 54, 103, 67, 109]
    
    # 眼睛 (用于大眼)
    LEFT_EYE = [33, 246, 161, 160, 159, 158, 157, 173, 133, 155, 154, 153, 145, 144, 163, 7]
    RIGHT_EYE = [362, 398, 384, 385, 386, 387, 388, 466, 263, 249, 390, 373, 374, 380, 381, 382]
    
    LEFT_EYE_CENTER = 468
    RIGHT_EYE_CENTER = 473
    
    # 眉毛
    LEFT_EYEBROW = [70, 63, 105, 66, 107, 55, 65, 52, 53, 46]
    RIGHT_EYEBROW = [336, 296, 334, 293, 300, 276, 283, 282, 295, 285]
    
    # 嘴唇
    LIPS = [61, 146, 91, 181, 84, 17, 314, 405, 321, 375, 291, 308, 324, 318, 402, 317, 14, 87, 178, 88, 95]
    
    # 鼻子
    NOSE_TIP = 1
    NOSE_TOP = 6
    
    # 脸颊 (用于瘦脸)
    LEFT_CHEEK = 234
    RIGHT_CHEEK = 454
    CHIN = 152
    FOREHEAD = 10
