# -*- coding: utf-8 -*-
"""
键盘输入防检测模块
模拟人类打字行为，绕过平台自动化检测

核心技术:
1. Interception驱动 - 内核级输入，无INJECTED标记
2. 人类化延迟 - 随机延迟+正态分布
3. 打字节奏模拟 - 模拟真实打字速度波动
4. 偶尔错误 - 打错并删除，增加真实性
5. 思考停顿 - 随机停顿模拟思考
6. 拼音输入 - 中文使用拼音逐字输入

参考项目:
- pyinterception: https://github.com/kennyhml/pyinterception
- KeymouseGo: https://github.com/taojy123/KeymouseGo
"""

import random
import time
import threading
from typing import Optional, Callable
from dataclasses import dataclass
import logging

logger = logging.getLogger(__name__)


@dataclass
class TypingConfig:
    """打字配置"""
    wpm: int = 80                    # 每分钟单词数 (60-120为正常人速度)
    delay_variance: float = 0.3      # 延迟波动比例
    key_hold_min: float = 0.02       # 按键最短按下时间
    key_hold_max: float = 0.08       # 按键最长按下时间
    error_rate: float = 0.02         # 错误率 (0-0.1)
    pause_rate: float = 0.05         # 停顿频率 (0-0.2)
    pause_min: float = 0.3           # 最短停顿时间
    pause_max: float = 1.0           # 最长停顿时间
    use_interception: bool = True    # 优先使用Interception驱动


class KeyboardAntiDetection:
    """键盘输入防检测处理器"""
    
    def __init__(self, config: Optional[TypingConfig] = None):
        self.config = config or TypingConfig()
        self._interception_available = False
        self._interception = None
        self._fallback_method = None
        
        # 初始化输入方法
        self._init_input_method()
    
    def _init_input_method(self):
        """初始化输入方法 - 优先使用Interception"""
        if self.config.use_interception:
            try:
                # 强制添加当前目录到 PATH，确保能找到 interception.dll
                import os
                import sys
                
                # 获取DLL路径：优先检查 PyInstaller 解压目录
                if getattr(sys, 'frozen', False):
                    base_path = sys._MEIPASS
                else:
                    base_path = os.getcwd()
                
                dll_path = os.path.join(base_path, "interception.dll")
                
                # 如果存在DLL，尝试预加载
                if os.path.exists(dll_path):
                    try:
                        import ctypes
                        # 添加目录到DLL搜索路径 (Python 3.8+)
                        if hasattr(os, 'add_dll_directory'):
                            os.add_dll_directory(base_path)
                        
                        ctypes.CDLL(dll_path)
                        logger.info(f"预加载DLL成功: {dll_path}")
                    except Exception as e:
                        logger.warning(f"预加载DLL失败: {e}")

                # 尝试导入并初始化
                import interception
                
                # 验证驱动是否可用 (尝试创建一个context)
                try:
                    context = interception.Interception()
                    context.destroy() # 立即销毁，只是为了测试
                except Exception as e:
                    raise ImportError(f"Interception驱动加载失败(DLL路径: {dll_path}): {e}")
                
                interception.auto_capture_devices()
                self._interception = interception
                self._interception_available = True
                logger.info("Interception驱动加载成功 - 使用内核级输入")
            except Exception as e:
                logger.warning(f"Interception驱动加载失败: {e}，使用备用方案")
                self._interception_available = False
        
        # 备用方案: pynput
        if not self._interception_available:
            try:
                from pynput.keyboard import Controller, Key
                self._fallback_controller = Controller()
                self._fallback_key = Key
                self._fallback_method = "pynput"
                logger.info("使用pynput作为备用输入方案")
            except ImportError:
                pass
        
        # 备用方案2: pyautogui
        if not self._interception_available and not self._fallback_method:
            try:
                import pyautogui
                pyautogui.PAUSE = 0
                self._pyautogui = pyautogui
                self._fallback_method = "pyautogui"
                logger.info("使用pyautogui作为备用输入方案")
            except ImportError:
                logger.error("没有可用的输入方案！请安装 interception-python 或 pynput")
    
    def _get_char_delay(self) -> float:
        """获取字符间延迟 - 使用正态分布模拟人类打字"""
        base_delay = 60 / (self.config.wpm * 5)  # 平均每字符延迟
        delay = random.gauss(base_delay, base_delay * self.config.delay_variance)
        return max(0.02, min(delay, base_delay * 3))
    
    def _get_key_hold_time(self) -> float:
        """获取按键按下时长"""
        return random.uniform(self.config.key_hold_min, self.config.key_hold_max)
    
    def _should_make_error(self) -> bool:
        """是否应该产生错误"""
        return random.random() < self.config.error_rate
    
    def _should_pause(self) -> bool:
        """是否应该停顿"""
        return random.random() < self.config.pause_rate
    
    def _get_pause_duration(self) -> float:
        """获取停顿时长"""
        return random.uniform(self.config.pause_min, self.config.pause_max)
    
    def _press_key_interception(self, key: str):
        """使用Interception驱动按键"""
        if len(key) == 1:
            self._interception.press(key)
        else:
            # 特殊键处理
            self._interception.press(key)
    
    def _press_key_pynput(self, key: str):
        """使用pynput按键"""
        if len(key) == 1:
            self._fallback_controller.press(key)
            time.sleep(self._get_key_hold_time())
            self._fallback_controller.release(key)
        else:
            # 特殊键
            special_key = getattr(self._fallback_key, key, None)
            if special_key:
                self._fallback_controller.press(special_key)
                time.sleep(self._get_key_hold_time())
                self._fallback_controller.release(special_key)
    
    def _press_key_pyautogui(self, key: str):
        """使用pyautogui按键"""
        self._pyautogui.press(key)
    
    def _press_key(self, key: str):
        """按下按键 - 自动选择最佳方法"""
        if self._interception_available:
            self._press_key_interception(key)
        elif self._fallback_method == "pynput":
            self._press_key_pynput(key)
        elif self._fallback_method == "pyautogui":
            self._press_key_pyautogui(key)
    
    def _delete_char(self):
        """删除一个字符"""
        self._press_key("backspace")
    
    def _get_wrong_char(self, correct_char: str) -> str:
        """获取一个错误字符（相邻键）"""
        # 键盘相邻键映射
        adjacent_keys = {
            'a': 'sqwz', 'b': 'vghn', 'c': 'xdfv', 'd': 'erfcxs',
            'e': 'rdsw', 'f': 'rtgvcd', 'g': 'tyhbvf', 'h': 'yujnbg',
            'i': 'uojk', 'j': 'uikmnh', 'k': 'iolmj', 'l': 'opk',
            'm': 'njk', 'n': 'bhjm', 'o': 'iplk', 'p': 'ol',
            'q': 'wa', 'r': 'etdf', 's': 'wedxza', 't': 'ryfg',
            'u': 'yihj', 'v': 'cfgb', 'w': 'qeas', 'x': 'zsdc',
            'y': 'tugh', 'z': 'asx',
        }
        lower = correct_char.lower()
        if lower in adjacent_keys:
            wrong = random.choice(adjacent_keys[lower])
            return wrong.upper() if correct_char.isupper() else wrong
        return correct_char
    
    def type_text(self, text: str, callback: Optional[Callable[[int, int], None]] = None):
        """
        人类化打字输入
        
        Args:
            text: 要输入的文本
            callback: 进度回调 (当前位置, 总长度)
        """
        total = len(text)
        
        for i, char in enumerate(text):
            # 回调进度
            if callback:
                callback(i, total)
            
            # 模拟思考停顿
            if self._should_pause():
                time.sleep(self._get_pause_duration())
            
            # 模拟打错并删除
            if self._should_make_error() and char.isalpha():
                wrong_char = self._get_wrong_char(char)
                self._press_key(wrong_char)
                time.sleep(random.uniform(0.1, 0.3))  # 发现错误的延迟
                self._delete_char()
                time.sleep(random.uniform(0.05, 0.15))
            
            # 输入正确字符
            if char == '\n':
                self._press_key("enter")
            elif char == '\t':
                self._press_key("tab")
            elif char == ' ':
                self._press_key("space")
            else:
                self._press_key(char)
            
            # 字符间延迟
            time.sleep(self._get_char_delay())
        
        if callback:
            callback(total, total)
    
    def type_text_async(self, text: str, callback: Optional[Callable[[int, int], None]] = None):
        """异步打字"""
        thread = threading.Thread(target=self.type_text, args=(text, callback), daemon=True)
        thread.start()
        return thread
    
    def set_speed(self, level: str):
        """
        设置打字速度
        
        Args:
            level: "slow", "normal", "fast"
        """
        if level == "slow":
            self.config.wpm = 40
            self.config.pause_rate = 0.1
            self.config.error_rate = 0.03
        elif level == "normal":
            self.config.wpm = 80
            self.config.pause_rate = 0.05
            self.config.error_rate = 0.02
        elif level == "fast":
            self.config.wpm = 120
            self.config.pause_rate = 0.02
            self.config.error_rate = 0.01
    
    def get_status(self) -> dict:
        """获取当前状态"""
        return {
            "输入方法": "Interception驱动" if self._interception_available else self._fallback_method,
            "防检测": self._interception_available,
            "打字速度": f"{self.config.wpm} WPM",
            "错误率": f"{self.config.error_rate * 100:.1f}%",
            "停顿率": f"{self.config.pause_rate * 100:.1f}%"
        }


class ChineseInputSimulator:
    """中文输入模拟器 - 模拟拼音输入"""
    
    def __init__(self, keyboard: KeyboardAntiDetection):
        self.keyboard = keyboard
        self._pypinyin_available = False
        
        try:
            from pypinyin import pinyin, Style
            self._pinyin = pinyin
            self._style = Style
            self._pypinyin_available = True
        except ImportError:
            logger.warning("pypinyin未安装，中文将直接输入")
    
    def type_chinese(self, text: str, callback: Optional[Callable[[int, int], None]] = None):
        """
        模拟拼音输入中文
        
        Args:
            text: 中文文本
            callback: 进度回调
        """
        if not self._pypinyin_available:
            # 直接输入
            self.keyboard.type_text(text, callback)
            return
        
        total = len(text)
        for i, char in enumerate(text):
            if callback:
                callback(i, total)
            
            if '\u4e00' <= char <= '\u9fff':  # 中文字符
                # 获取拼音
                py_list = self._pinyin(char, style=self._style.NORMAL)
                if py_list and py_list[0]:
                    py = py_list[0][0]
                    # 逐字输入拼音
                    for letter in py:
                        self.keyboard._press_key(letter)
                        time.sleep(random.uniform(0.05, 0.12))
                    # 按空格/数字选择
                    time.sleep(random.uniform(0.1, 0.2))
                    self.keyboard._press_key("space")
                    time.sleep(random.uniform(0.05, 0.1))
            else:
                # 非中文字符直接输入
                self.keyboard._press_key(char)
                time.sleep(self.keyboard._get_char_delay())
        
        if callback:
            callback(total, total)


# 全局实例
_keyboard_instance: Optional[KeyboardAntiDetection] = None
_chinese_instance: Optional[ChineseInputSimulator] = None


def get_keyboard() -> KeyboardAntiDetection:
    """获取键盘防检测实例"""
    global _keyboard_instance
    if _keyboard_instance is None:
        _keyboard_instance = KeyboardAntiDetection()
    return _keyboard_instance


def get_chinese_input() -> ChineseInputSimulator:
    """获取中文输入模拟器"""
    global _chinese_instance
    if _chinese_instance is None:
        _chinese_instance = ChineseInputSimulator(get_keyboard())
    return _chinese_instance


def human_type(text: str, speed: str = "normal"):
    """
    便捷函数 - 人类化打字
    
    Args:
        text: 要输入的文本
        speed: 速度 ("slow", "normal", "fast")
    """
    kb = get_keyboard()
    kb.set_speed(speed)
    kb.type_text(text)


def human_type_chinese(text: str):
    """
    便捷函数 - 人类化中文输入
    
    Args:
        text: 要输入的中文文本
    """
    ci = get_chinese_input()
    ci.type_chinese(text)
