# -*- coding: utf-8 -*-
"""
视频防检测/去重处理器
实时对画面进行随机微调，防止被平台检测为重复画面

核心技术 (基于TiktokTools和videohash调研):
1. 随机噪点 - 每帧添加不可见噪点，改变像素指纹
2. 随机位移 - 画面微小偏移，改变帧结构
3. 随机缩放 - 画面微小缩放，改变分辨率特征
4. 随机色调 - HSV微调，改变色彩分布
5. 随机亮度/对比度/饱和度/伽马 - 综合色彩调整
6. 轻微锐化 - unsharp效果，改变边缘特征
7. 隐形水印 - 添加不可见的随机水印
8. 周期性参数变化 - 避免检测到固定模式

参考项目:
- TiktokTools: https://github.com/Mouseww/TiktokTools
- videohash: https://github.com/akamhy/videohash
- GhostCut: https://github.com/JollyToday/GhostCut_Remake_Video
"""

import cv2
import numpy as np
import random
import time
from typing import Optional, Tuple


class VideoAntiDetection:
    """视频防检测处理器"""
    
    def __init__(self):
        # 随机种子基于时间
        self.seed = int(time.time())
        random.seed(self.seed)
        
        # 处理参数（可调整强度）- 基于TiktokTools验证有效的参数
        self.noise_strength = 3      # 噪点强度 (0-10)
        self.shift_range = 2         # 位移范围像素 (0-5)
        self.scale_range = 0.02      # 缩放范围 (0-0.05)
        self.hue_range = 3           # 色调变化范围 (0-10)
        self.brightness_range = 5    # 亮度变化范围 (0-20)
        self.contrast_range = 0.02   # 对比度变化范围 (0-0.1)
        self.saturation_range = 0.03 # 饱和度变化范围 (0-0.1)
        self.gamma_range = 0.02      # 伽马值变化范围 (0-0.1)
        self.sharpen_strength = 0.3  # 锐化强度 (0-1.5)
        
        # 计数器，用于周期性变化
        self.frame_count = 0
        self.change_interval = 30  # 每30帧变化一次参数
        
        # 启用的处理选项
        self.enable_noise = True
        self.enable_shift = True
        self.enable_scale = True
        self.enable_color = True
        self.enable_sharpen = True
        self.enable_watermark = True
        
        # 当前随机参数
        self._update_random_params()
    
    def _update_random_params(self):
        """更新随机参数 - 每个周期随机生成新参数"""
        self.current_shift_x = random.randint(-self.shift_range, self.shift_range)
        self.current_shift_y = random.randint(-self.shift_range, self.shift_range)
        self.current_scale = 1.0 + random.uniform(-self.scale_range, self.scale_range)
        self.current_hue = random.randint(-self.hue_range, self.hue_range)
        self.current_brightness = random.randint(-self.brightness_range, self.brightness_range)
        self.current_contrast = 1.0 + random.uniform(-self.contrast_range, self.contrast_range)
        self.current_saturation = 1.0 + random.uniform(-self.saturation_range, self.saturation_range)
        self.current_gamma = 1.0 + random.uniform(-self.gamma_range, self.gamma_range)
        self.current_sharpen = random.uniform(0.1, self.sharpen_strength)
    
    def add_noise(self, frame: np.ndarray) -> np.ndarray:
        """添加随机噪点"""
        if self.noise_strength <= 0:
            return frame
        
        noise = np.random.randint(
            -self.noise_strength, 
            self.noise_strength + 1, 
            frame.shape, 
            dtype=np.int16
        )
        noisy = np.clip(frame.astype(np.int16) + noise, 0, 255).astype(np.uint8)
        return noisy
    
    def apply_shift(self, frame: np.ndarray) -> np.ndarray:
        """应用随机位移"""
        if self.shift_range <= 0:
            return frame
        
        rows, cols = frame.shape[:2]
        M = np.float32([
            [1, 0, self.current_shift_x],
            [0, 1, self.current_shift_y]
        ])
        shifted = cv2.warpAffine(frame, M, (cols, rows), borderMode=cv2.BORDER_REPLICATE)
        return shifted
    
    def apply_scale(self, frame: np.ndarray) -> np.ndarray:
        """应用随机缩放"""
        if self.scale_range <= 0:
            return frame
        
        rows, cols = frame.shape[:2]
        # 缩放后居中
        new_cols = int(cols * self.current_scale)
        new_rows = int(rows * self.current_scale)
        
        scaled = cv2.resize(frame, (new_cols, new_rows))
        
        # 裁剪或填充回原尺寸
        if self.current_scale > 1.0:
            # 裁剪
            start_x = (new_cols - cols) // 2
            start_y = (new_rows - rows) // 2
            result = scaled[start_y:start_y+rows, start_x:start_x+cols]
        else:
            # 填充
            result = np.zeros_like(frame)
            start_x = (cols - new_cols) // 2
            start_y = (rows - new_rows) // 2
            result[start_y:start_y+new_rows, start_x:start_x+new_cols] = scaled
        
        return result
    
    def apply_color_adjustment(self, frame: np.ndarray) -> np.ndarray:
        """应用颜色调整（色调、亮度、对比度、饱和度、伽马）
        
        基于TiktokTools的滤镜参数:
        eq=contrast=X:brightness=Y:saturation=Z:gamma=W
        """
        # 转换到HSV
        hsv = cv2.cvtColor(frame, cv2.COLOR_BGR2HSV).astype(np.float32)
        
        # 调整色调 (H通道)
        hsv[:, :, 0] = np.clip(hsv[:, :, 0] + self.current_hue, 0, 179)
        
        # 调整饱和度 (S通道)
        hsv[:, :, 1] = np.clip(hsv[:, :, 1] * self.current_saturation, 0, 255)
        
        # 调整亮度 (V通道)
        hsv[:, :, 2] = np.clip(hsv[:, :, 2] + self.current_brightness, 0, 255)
        
        result = cv2.cvtColor(hsv.astype(np.uint8), cv2.COLOR_HSV2BGR)
        
        # 调整对比度
        result = cv2.convertScaleAbs(result, alpha=self.current_contrast, beta=0)
        
        # 应用伽马校正
        if abs(self.current_gamma - 1.0) > 0.001:
            inv_gamma = 1.0 / self.current_gamma
            table = np.array([((i / 255.0) ** inv_gamma) * 255 
                            for i in np.arange(0, 256)]).astype("uint8")
            result = cv2.LUT(result, table)
        
        return result
    
    def apply_sharpen(self, frame: np.ndarray) -> np.ndarray:
        """应用锐化 - 基于TiktokTools的unsharp滤镜
        
        FFmpeg等效命令:
        -vf unsharp=luma_msize_x=7:luma_msize_y=7:luma_amount=X
        """
        if self.current_sharpen <= 0:
            return frame
        
        # 使用unsharp mask锐化
        blurred = cv2.GaussianBlur(frame, (7, 7), 0)
        sharpened = cv2.addWeighted(frame, 1 + self.current_sharpen, 
                                    blurred, -self.current_sharpen, 0)
        return sharpened
    
    def add_invisible_watermark(self, frame: np.ndarray) -> np.ndarray:
        """添加隐形水印（在角落添加随机像素变化）"""
        result = frame.copy()
        
        # 在随机位置添加微小变化
        h, w = frame.shape[:2]
        for _ in range(5):
            x = random.randint(0, w - 1)
            y = random.randint(0, h - 1)
            # 微调像素值
            result[y, x] = np.clip(
                result[y, x].astype(np.int16) + random.randint(-2, 2),
                0, 255
            ).astype(np.uint8)
        
        return result
    
    def process_frame(self, frame: np.ndarray) -> np.ndarray:
        """处理单帧图像 - 综合应用所有防检测技术"""
        if frame is None:
            return None
        
        self.frame_count += 1
        
        # 周期性更新随机参数（避免固定模式被检测）
        if self.frame_count % self.change_interval == 0:
            self._update_random_params()
        
        # 应用所有处理（顺序很重要）
        result = frame.copy()
        
        # 1. 随机噪点 - 改变像素指纹
        if self.enable_noise:
            result = self.add_noise(result)
        
        # 2. 随机位移 - 改变帧结构
        if self.enable_shift:
            result = self.apply_shift(result)
        
        # 3. 随机缩放 - 改变分辨率特征
        if self.enable_scale:
            result = self.apply_scale(result)
        
        # 4. 颜色调整 - 改变色彩分布
        if self.enable_color:
            result = self.apply_color_adjustment(result)
        
        # 5. 锐化处理 - 改变边缘特征
        if self.enable_sharpen:
            result = self.apply_sharpen(result)
        
        # 6. 隐形水印 - 额外的随机变化
        if self.enable_watermark:
            result = self.add_invisible_watermark(result)
        
        return result
    
    def set_strength(self, level: str = "medium"):
        """设置处理强度 - 基于TiktokTools验证有效的参数范围
        
        Args:
            level: "low", "medium", "high"
        """
        if level == "low":
            # 低强度 - 几乎不可见
            self.noise_strength = 1
            self.shift_range = 1
            self.scale_range = 0.01
            self.hue_range = 1
            self.brightness_range = 2
            self.contrast_range = 0.01
            self.saturation_range = 0.01
            self.gamma_range = 0.01
            self.sharpen_strength = 0.1
        elif level == "medium":
            # 中等强度 - 推荐
            self.noise_strength = 3
            self.shift_range = 2
            self.scale_range = 0.02
            self.hue_range = 3
            self.brightness_range = 5
            self.contrast_range = 0.02
            self.saturation_range = 0.03
            self.gamma_range = 0.02
            self.sharpen_strength = 0.3
        elif level == "high":
            # 高强度 - 明显变化
            self.noise_strength = 5
            self.shift_range = 3
            self.scale_range = 0.03
            self.hue_range = 5
            self.brightness_range = 10
            self.contrast_range = 0.03
            self.saturation_range = 0.05
            self.gamma_range = 0.03
            self.sharpen_strength = 0.5
        
        self._update_random_params()
    
    def get_current_params(self) -> dict:
        """获取当前处理参数"""
        return {
            "帧数": self.frame_count,
            "位移": (self.current_shift_x, self.current_shift_y),
            "缩放": f"{self.current_scale:.3f}",
            "色调": self.current_hue,
            "亮度": self.current_brightness,
            "对比度": f"{self.current_contrast:.3f}",
            "饱和度": f"{self.current_saturation:.3f}",
            "伽马": f"{self.current_gamma:.3f}",
            "锐化": f"{self.current_sharpen:.3f}"
        }


class VirtualCameraFilter:
    """虚拟摄像头滤镜 - 用于OBS或其他直播软件"""
    
    def __init__(self):
        self.processor = VideoAntiDetection()
        self.enabled = True
    
    def process(self, frame: np.ndarray) -> np.ndarray:
        """处理帧"""
        if not self.enabled:
            return frame
        return self.processor.process_frame(frame)
    
    def toggle(self):
        """切换开关"""
        self.enabled = not self.enabled
        return self.enabled
    
    def set_strength(self, level: str):
        """设置强度"""
        self.processor.set_strength(level)


# 测试用例
if __name__ == "__main__":
    print("视频防检测处理器测试")
    
    # 创建处理器
    processor = VideoAntiDetection()
    processor.set_strength("medium")
    
    # 创建测试图像
    test_frame = np.ones((480, 640, 3), dtype=np.uint8) * 128
    
    # 处理
    result = processor.process_frame(test_frame)
    
    print(f"原始图像: {test_frame.shape}")
    print(f"处理后图像: {result.shape}")
    print(f"像素差异: {np.mean(np.abs(test_frame.astype(int) - result.astype(int))):.2f}")
    print("测试完成！")
