# -*- coding: utf-8 -*-
import os
import sys
import socket
import threading
import traceback
from datetime import datetime
from http.server import SimpleHTTPRequestHandler, HTTPServer
from functools import partial
from pathlib import Path


def _get_log_file():
    """获取日志文件路径"""
    if getattr(sys, 'frozen', False):
        base = Path(sys.executable).parent
    else:
        base = Path(__file__).parent.parent.parent
    log_dir = base / "logs"
    log_dir.mkdir(exist_ok=True)
    return log_dir / "webserver.log"


def _log(level: str, msg: str):
    """写入日志文件"""
    try:
        timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        line = f"{timestamp} | {level:5} | {msg}\n"
        with open(_get_log_file(), "a", encoding="utf-8") as f:
            f.write(line)
            f.flush()
    except:
        pass


def log_info(msg): _log("INFO", msg)
def log_error(msg): _log("ERROR", msg)
def log_debug(msg): _log("DEBUG", msg)


def is_port_available(port: int) -> bool:
    """检查端口是否可用"""
    try:
        with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as s:
            s.bind(("127.0.0.1", port))
            return True
    except OSError as e:
        log_debug(f"端口 {port} 不可用: {e}")
        return False


def find_available_port(start_port: int = 8080, max_tries: int = 10) -> int:
    """查找可用端口，从 start_port 开始尝试"""
    log_info(f"查找可用端口，起始: {start_port}")
    for i in range(max_tries):
        port = start_port + i
        if is_port_available(port):
            log_info(f"找到可用端口: {port}")
            return port
    log_error(f"未找到可用端口，使用默认: {start_port}")
    return start_port


class LoggingHandler(SimpleHTTPRequestHandler):
    """带日志的静态文件服务 Handler"""
    def log_message(self, format, *args):
        """记录请求日志"""
        log_info(f"REQUEST: {args[0] if args else 'unknown'}")
    
    def do_GET(self):
        """处理 GET 请求"""
        log_info(f"GET {self.path}")
        try:
            super().do_GET()
            log_info(f"GET {self.path} 完成")
        except Exception as e:
            log_error(f"GET {self.path} 异常: {e}")
            raise


class WebServerThread(threading.Thread):
    def __init__(self, port=8080, root_dir="./static", auto_port=True):
        super().__init__()
        self.requested_port = port
        self.port = port  # 实际使用的端口
        self.root_dir = root_dir
        self.auto_port = auto_port  # 是否自动切换端口
        self.httpd = None
        self._running = False
        self._started_event = threading.Event()  # 用于通知启动完成
        self._error_msg = None
        self.daemon = True  # 守护线程，主程序退出时自动退出
        log_info(f"__init__: port={port}, root_dir={root_dir}, auto_port={auto_port}")

    def run(self):
        try:
            log_info("run() 开始执行")
            log_info(f"root_dir={self.root_dir}")
            
            # 确保目录存在
            if not os.path.exists(self.root_dir):
                self._error_msg = f"目录不存在: {self.root_dir}"
                log_error(f"root dir not found: {self.root_dir}")
                self._started_event.set()
                return

            log_info(f"目录存在: {self.root_dir}")
            
            # 列出目录内容
            try:
                contents = os.listdir(self.root_dir)
                log_info(f"目录内容: {contents[:10]}")
            except Exception as e:
                log_error(f"无法列出目录: {e}")
            
            # 创建Handler，指定根目录
            log_info("创建 Handler...")
            handler = partial(LoggingHandler, directory=self.root_dir)
            
            # 允许重用端口
            HTTPServer.allow_reuse_address = True
            
            # 如果启用自动端口，查找可用端口
            if self.auto_port:
                self.port = find_available_port(self.requested_port)
                if self.port != self.requested_port:
                    log_info(f"端口 {self.requested_port} 被占用，使用 {self.port}")
            
            log_info(f"创建 HTTPServer, 绑定 127.0.0.1:{self.port}...")
            self.httpd = HTTPServer(("127.0.0.1", self.port), handler)
            self._running = True
            self._started_event.set()  # 通知启动完成
            log_info(f"✓ 服务器启动成功: http://127.0.0.1:{self.port}")
            
            log_info("开始 serve_forever()...")
            try:
                self.httpd.serve_forever()
            except Exception as serve_error:
                log_error(f"serve_forever 异常退出: {serve_error}")
                log_error(traceback.format_exc())
            finally:
                log_info("serve_forever() 已退出")
                self._running = False
        except Exception as e:
            self._error_msg = str(e)
            log_error(f"✗ 启动异常: {e}")
            log_error(traceback.format_exc())
            self._running = False
            self._started_event.set()

    def wait_for_start(self, timeout: float = 2.0) -> bool:
        """等待服务器启动完成，返回是否成功"""
        log_debug(f"wait_for_start(timeout={timeout})...")
        self._started_event.wait(timeout)
        log_info(f"wait_for_start 完成, _running={self._running}")
        return self._running

    def get_actual_port(self) -> int:
        """获取实际使用的端口"""
        return self.port
    
    def get_error(self) -> str:
        """获取错误信息"""
        return self._error_msg

    def stop(self):
        if self.httpd:
            log_info("停止服务器...")
            self.httpd.shutdown()
            self.httpd.server_close()
            self._running = False
            log_info("服务器已停止")
