#include "crc64nvme.h"

uint64_t crc64nvme_bit(uint64_t crc, void const *mem, size_t len) {
    uint8_t const *data = mem;
    if (data == NULL)
        return 0;
    crc = ~crc;
    for (size_t i = 0; i < len; i++) {
        crc ^= data[i];
        for (unsigned k = 0; k < 8; k++) {
            crc = crc & 1 ? (crc >> 1) ^ 0x9a6c9329ac4bc9b5 : crc >> 1;
        }
    }
    crc = ~crc;
    return crc;
}

uint64_t crc64nvme_rem(uint64_t crc, unsigned val, unsigned bits) {
    crc = ~crc;
    val &= (1U << bits) - 1;
    crc ^= val;
    for (unsigned i = 0; i < bits; i++) {
        crc = crc & 1 ? (crc >> 1) ^ 0x9a6c9329ac4bc9b5 : crc >> 1;
    }
    crc = ~crc;
    return crc;
}

#define table_byte table_word[0]

static uint64_t const table_word[][256] = {
   {0xd5da5047efec8728, 0xaab4a08fdfd90e51, 0x2b07b1d78f8795da, 0x5469411fbfb21ca3,
    0x1cb8b53477ad31a7, 0x63d645fc4798b8de, 0xe26554a417c62355, 0x9d0ba46c27f3aa2c,
    0x73c6bcf387f8795d, 0x0ca84c3bb7cdf024, 0x8d1b5d63e7936baf, 0xf275adabd7a6e2d6,
    0xbaa459801fb9cfd2, 0xc5caa9482f8c46ab, 0x4479b8107fd2dd20, 0x3b1748d84fe75459,
    0xad3aaf7c6752e8a9, 0xd2545fb4576761d0, 0x53e74eec0739fa5b, 0x2c89be24370c7322,
    0x64584a0fff135e26, 0x1b36bac7cf26d75f, 0x9a85ab9f9f784cd4, 0xe5eb5b57af4dc5ad,
    0x0b2643c80f4616dc, 0x7448b3003f739fa5, 0xf5fba2586f2d042e, 0x8a9552905f188d57,
    0xc244a6bb9707a053, 0xbd2a5673a732292a, 0x3c99472bf76cb2a1, 0x43f7b7e3c7593bd8,
    0x241bae30fe90582a, 0x5b755ef8cea5d153, 0xdac64fa09efb4ad8, 0xa5a8bf68aecec3a1,
    0xed794b4366d1eea5, 0x9217bb8b56e467dc, 0x13a4aad306bafc57, 0x6cca5a1b368f752e,
    0x820742849684a65f, 0xfd69b24ca6b12f26, 0x7cdaa314f6efb4ad, 0x03b453dcc6da3dd4,
    0x4b65a7f70ec510d0, 0x340b573f3ef099a9, 0xb5b846676eae0222, 0xcad6b6af5e9b8b5b,
    0x5cfb510b762e37ab, 0x2395a1c3461bbed2, 0xa226b09b16452559, 0xdd4840532670ac20,
    0x9599b478ee6f8124, 0xeaf744b0de5a085d, 0x6b4455e88e0493d6, 0x142aa520be311aaf,
    0xfae7bdbf1e3ac9de, 0x85894d772e0f40a7, 0x043a5c2f7e51db2c, 0x7b54ace74e645255,
    0x338558cc867b7f51, 0x4ceba804b64ef628, 0xcd58b95ce6106da3, 0xb2364994d625e4da,
    0x02808afa9582aa47, 0x7dee7a32a5b7233e, 0xfc5d6b6af5e9b8b5, 0x83339ba2c5dc31cc,
    0xcbe26f890dc31cc8, 0xb48c9f413df695b1, 0x353f8e196da80e3a, 0x4a517ed15d9d8743,
    0xa49c664efd965432, 0xdbf29686cda3dd4b, 0x5a4187de9dfd46c0, 0x252f7716adc8cfb9,
    0x6dfe833d65d7e2bd, 0x129073f555e26bc4, 0x932362ad05bcf04f, 0xec4d926535897936,
    0x7a6075c11d3cc5c6, 0x050e85092d094cbf, 0x84bd94517d57d734, 0xfbd364994d625e4d,
    0xb30290b2857d7349, 0xcc6c607ab548fa30, 0x4ddf7122e51661bb, 0x32b181ead523e8c2,
    0xdc7c997575283bb3, 0xa31269bd451db2ca, 0x22a178e515432941, 0x5dcf882d2576a038,
    0x151e7c06ed698d3c, 0x6a708ccedd5c0445, 0xebc39d968d029fce, 0x94ad6d5ebd3716b7,
    0xf341748d84fe7545, 0x8c2f8445b4cbfc3c, 0x0d9c951de49567b7, 0x72f265d5d4a0eece,
    0x3a2391fe1cbfc3ca, 0x454d61362c8a4ab3, 0xc4fe706e7cd4d138, 0xbb9080a64ce15841,
    0x555d9839ecea8b30, 0x2a3368f1dcdf0249, 0xab8079a98c8199c2, 0xd4ee8961bcb410bb,
    0x9c3f7d4a74ab3dbf, 0xe3518d82449eb4c6, 0x62e29cda14c02f4d, 0x1d8c6c1224f5a634,
    0x8ba18bb60c401ac4, 0xf4cf7b7e3c7593bd, 0x757c6a266c2b0836, 0x0a129aee5c1e814f,
    0x42c36ec59401ac4b, 0x3dad9e0da4342532, 0xbc1e8f55f46abeb9, 0xc3707f9dc45f37c0,
    0x2dbd67026454e4b1, 0x52d397ca54616dc8, 0xd3608692043ff643, 0xac0e765a340a7f3a,
    0xe4df8271fc15523e, 0x9bb172b9cc20db47, 0x1a0263e19c7e40cc, 0x656c9329ac4bc9b5,
    0x4fb6c36e43a74e9d, 0x30d833a67392c7e4, 0xb16b22fe23cc5c6f, 0xce05d23613f9d516,
    0x86d4261ddbe6f812, 0xf9bad6d5ebd3716b, 0x7809c78dbb8deae0, 0x076737458bb86399,
    0xe9aa2fda2bb3b0e8, 0x96c4df121b863991, 0x1777ce4a4bd8a21a, 0x68193e827bed2b63,
    0x20c8caa9b3f20667, 0x5fa63a6183c78f1e, 0xde152b39d3991495, 0xa17bdbf1e3ac9dec,
    0x37563c55cb19211c, 0x4838cc9dfb2ca865, 0xc98bddc5ab7233ee, 0xb6e52d0d9b47ba97,
    0xfe34d92653589793, 0x815a29ee636d1eea, 0x00e938b633338561, 0x7f87c87e03060c18,
    0x914ad0e1a30ddf69, 0xee24202993385610, 0x6f973171c366cd9b, 0x10f9c1b9f35344e2,
    0x582835923b4c69e6, 0x2746c55a0b79e09f, 0xa6f5d4025b277b14, 0xd99b24ca6b12f26d,
    0xbe773d1952db919f, 0xc119cdd162ee18e6, 0x40aadc8932b0836d, 0x3fc42c4102850a14,
    0x7715d86aca9a2710, 0x087b28a2faafae69, 0x89c839faaaf135e2, 0xf6a6c9329ac4bc9b,
    0x186bd1ad3acf6fea, 0x670521650afae693, 0xe6b6303d5aa47d18, 0x99d8c0f56a91f461,
    0xd10934dea28ed965, 0xae67c41692bb501c, 0x2fd4d54ec2e5cb97, 0x50ba2586f2d042ee,
    0xc697c222da65fe1e, 0xb9f932eaea507767, 0x384a23b2ba0eecec, 0x4724d37a8a3b6595,
    0x0ff5275142244891, 0x709bd7997211c1e8, 0xf128c6c1224f5a63, 0x8e463609127ad31a,
    0x608b2e96b271006b, 0x1fe5de5e82448912, 0x9e56cf06d21a1299, 0xe1383fcee22f9be0,
    0xa9e9cbe52a30b6e4, 0xd6873b2d1a053f9d, 0x57342a754a5ba416, 0x285adabd7a6e2d6f,
    0x98ec19d339c963f2, 0xe782e91b09fcea8b, 0x6631f84359a27100, 0x195f088b6997f879,
    0x518efca0a188d57d, 0x2ee00c6891bd5c04, 0xaf531d30c1e3c78f, 0xd03dedf8f1d64ef6,
    0x3ef0f56751dd9d87, 0x419e05af61e814fe, 0xc02d14f731b68f75, 0xbf43e43f0183060c,
    0xf7921014c99c2b08, 0x88fce0dcf9a9a271, 0x094ff184a9f739fa, 0x7621014c99c2b083,
    0xe00ce6e8b1770c73, 0x9f6216208142850a, 0x1ed10778d11c1e81, 0x61bff7b0e12997f8,
    0x296e039b2936bafc, 0x5600f35319033385, 0xd7b3e20b495da80e, 0xa8dd12c379682177,
    0x46100a5cd963f206, 0x397efa94e9567b7f, 0xb8cdebccb908e0f4, 0xc7a31b04893d698d,
    0x8f72ef2f41224489, 0xf01c1fe77117cdf0, 0x71af0ebf2149567b, 0x0ec1fe77117cdf02,
    0x692de7a428b5bcf0, 0x1643176c18803589, 0x97f0063448deae02, 0xe89ef6fc78eb277b,
    0xa04f02d7b0f40a7f, 0xdf21f21f80c18306, 0x5e92e347d09f188d, 0x21fc138fe0aa91f4,
    0xcf310b1040a14285, 0xb05ffbd87094cbfc, 0x31ecea8020ca5077, 0x4e821a4810ffd90e,
    0x0653ee63d8e0f40a, 0x793d1eabe8d57d73, 0xf88e0ff3b88be6f8, 0x87e0ff3b88be6f81,
    0x11cd189fa00bd371, 0x6ea3e857903e5a08, 0xef10f90fc060c183, 0x907e09c7f05548fa,
    0xd8affdec384a65fe, 0xa7c10d24087fec87, 0x26721c7c5821770c, 0x591cecb46814fe75,
    0xb7d1f42bc81f2d04, 0xc8bf04e3f82aa47d, 0x490c15bba8743ff6, 0x3662e5739841b68f,
    0x7eb31158505e9b8b, 0x01dde190606b12f2, 0x806ef0c830358979, 0xff00000000000000},
   {0x5708c8933e87cdf0, 0xd07e61ee4d3a1299, 0x6d3cbc3a816be049, 0xea4a1547f2d63f20,
    0x236021c0415f9682, 0xa41688bd32e249eb, 0x19545569feb3bb3b, 0x9e22fc148d0e6452,
    0xbfd91a35c1377b14, 0x38afb348b28aa47d, 0x85ed6e9c7edb56ad, 0x029bc7e10d6689c4,
    0xcbb1f366beef2066, 0x4cc75a1bcd52ff0f, 0xf18587cf01030ddf, 0x76f32eb272bed2b6,
    0xb2724b8d99713353, 0x3504e2f0eaccec3a, 0x88463f24269d1eea, 0x0f3096595520c183,
    0xc61aa2dee6a96821, 0x416c0ba39514b748, 0xfc2ed67759454598, 0x7b587f0a2af89af1,
    0x5aa3992b66c185b7, 0xddd53056157c5ade, 0x6097ed82d92da80e, 0xe7e144ffaa907767,
    0x2ecb70781919dec5, 0xa9bdd9056aa401ac, 0x14ff04d1a6f5f37c, 0x9389adacd5482c15,
    0xa924e8fd29fda3dd, 0x2e5241805a407cb4, 0x93109c5496118e64, 0x14663529e5ac510d,
    0xdd4c01ae5625f8af, 0x5a3aa8d3259827c6, 0xe7787507e9c9d516, 0x600edc7a9a740a7f,
    0x41f53a5bd64d1539, 0xc6839326a5f0ca50, 0x7bc14ef269a13880, 0xfcb7e78f1a1ce7e9,
    0x359dd308a9954e4b, 0xb2eb7a75da289122, 0x0fa9a7a1167963f2, 0x88df0edc65c4bc9b,
    0x4c5e6be38e0b5d7e, 0xcb28c29efdb68217, 0x766a1f4a31e770c7, 0xf11cb637425aafae,
    0x383682b0f1d3060c, 0xbf402bcd826ed965, 0x0202f6194e3f2bb5, 0x85745f643d82f4dc,
    0xa48fb94571bbeb9a, 0x23f91038020634f3, 0x9ebbcdecce57c623, 0x19cd6491bdea194a,
    0xd0e750160e63b0e8, 0x5791f96b7dde6f81, 0xead324bfb18f9d51, 0x6da58dc2c2324238,
    0x9f89ae1c48e482c1, 0x18ff07613b595da8, 0xa5bddab5f708af78, 0x22cb73c884b57011,
    0xebe1474f373cd9b3, 0x6c97ee32448106da, 0xd1d533e688d0f40a, 0x56a39a9bfb6d2b63,
    0x77587cbab7543425, 0xf02ed5c7c4e9eb4c, 0x4d6c081308b8199c, 0xca1aa16e7b05c6f5,
    0x033095e9c88c6f57, 0x84463c94bb31b03e, 0x3904e140776042ee, 0xbe72483d04dd9d87,
    0x7af32d02ef127c62, 0xfd85847f9cafa30b, 0x40c759ab50fe51db, 0xc7b1f0d623438eb2,
    0x0e9bc45190ca2710, 0x89ed6d2ce377f879, 0x34afb0f82f260aa9, 0xb3d919855c9bd5c0,
    0x9222ffa410a2ca86, 0x155456d9631f15ef, 0xa8168b0daf4ee73f, 0x2f602270dcf33856,
    0xe64a16f76f7a91f4, 0x613cbf8a1cc74e9d, 0xdc7e625ed096bc4d, 0x5b08cb23a32b6324,
    0x61a58e725f9eecec, 0xe6d3270f2c233385, 0x5b91fadbe072c155, 0xdce753a693cf1e3c,
    0x15cd67212046b79e, 0x92bbce5c53fb68f7, 0x2ff913889faa9a27, 0xa88fbaf5ec17454e,
    0x89745cd4a02e5a08, 0x0e02f5a9d3938561, 0xb340287d1fc277b1, 0x343681006c7fa8d8,
    0xfd1cb587dff6017a, 0x7a6a1cfaac4bde13, 0xc728c12e601a2cc3, 0x405e685313a7f3aa,
    0x84df0d6cf868124f, 0x03a9a4118bd5cd26, 0xbeeb79c547843ff6, 0x399dd0b83439e09f,
    0xf0b7e43f87b0493d, 0x77c14d42f40d9654, 0xca839096385c6484, 0x4df539eb4be1bbed,
    0x6c0edfca07d8a4ab, 0xeb7876b774657bc2, 0x563aab63b8348912, 0xd14c021ecb89567b,
    0x186636997800ffd9, 0x9f109fe40bbd20b0, 0x22524230c7ecd260, 0xa524eb4db4510d09,
    0xf2d323de8ad6c0f9, 0x75a58aa3f96b1f90, 0xc8e75777353aed40, 0x4f91fe0a46873229,
    0x86bbca8df50e9b8b, 0x01cd63f086b344e2, 0xbc8fbe244ae2b632, 0x3bf91759395f695b,
    0x1a02f1787566761d, 0x9d74580506dba974, 0x203685d1ca8a5ba4, 0xa7402cacb93784cd,
    0x6e6a182b0abe2d6f, 0xe91cb1567903f206, 0x545e6c82b55200d6, 0xd328c5ffc6efdfbf,
    0x17a9a0c02d203e5a, 0x90df09bd5e9de133, 0x2d9dd46992cc13e3, 0xaaeb7d14e171cc8a,
    0x63c1499352f86528, 0xe4b7e0ee2145ba41, 0x59f53d3aed144891, 0xde8394479ea997f8,
    0xff787266d29088be, 0x780edb1ba12d57d7, 0xc54c06cf6d7ca507, 0x423aafb21ec17a6e,
    0x8b109b35ad48d3cc, 0x0c663248def50ca5, 0xb124ef9c12a4fe75, 0x365246e16119211c,
    0x0cff03b09dacaed4, 0x8b89aacdee1171bd, 0x36cb77192240836d, 0xb1bdde6451fd5c04,
    0x7897eae3e274f5a6, 0xffe1439e91c92acf, 0x42a39e4a5d98d81f, 0xc5d537372e250776,
    0xe42ed116621c1830, 0x6358786b11a1c759, 0xde1aa5bfddf03589, 0x596c0cc2ae4deae0,
    0x904638451dc44342, 0x173091386e799c2b, 0xaa724ceca2286efb, 0x2d04e591d195b192,
    0xe98580ae3a5a5077, 0x6ef329d349e78f1e, 0xd3b1f40785b67dce, 0x54c75d7af60ba2a7,
    0x9ded69fd45820b05, 0x1a9bc080363fd46c, 0xa7d91d54fa6e26bc, 0x20afb42989d3f9d5,
    0x01545208c5eae693, 0x8622fb75b65739fa, 0x3b6026a17a06cb2a, 0xbc168fdc09bb1443,
    0x753cbb5bba32bde1, 0xf24a1226c98f6288, 0x4f08cff205de9058, 0xc87e668f76634f31,
    0x3a524551fcb58fc8, 0xbd24ec2c8f0850a1, 0x006631f84359a271, 0x8710988530e47d18,
    0x4e3aac02836dd4ba, 0xc94c057ff0d00bd3, 0x740ed8ab3c81f903, 0xf37871d64f3c266a,
    0xd28397f70305392c, 0x55f53e8a70b8e645, 0xe8b7e35ebce91495, 0x6fc14a23cf54cbfc,
    0xa6eb7ea47cdd625e, 0x219dd7d90f60bd37, 0x9cdf0a0dc3314fe7, 0x1ba9a370b08c908e,
    0xdf28c64f5b43716b, 0x585e6f3228feae02, 0xe51cb2e6e4af5cd2, 0x626a1b9b971283bb,
    0xab402f1c249b2a19, 0x2c3686615726f570, 0x91745bb59b7707a0, 0x1602f2c8e8cad8c9,
    0x37f914e9a4f3c78f, 0xb08fbd94d74e18e6, 0x0dcd60401b1fea36, 0x8abbc93d68a2355f,
    0x4391fdbadb2b9cfd, 0xc4e754c7a8964394, 0x79a5891364c7b144, 0xfed3206e177a6e2d,
    0xc47e653febcfe1e5, 0x4308cc4298723e8c, 0xfe4a11965423cc5c, 0x793cb8eb279e1335,
    0xb0168c6c9417ba97, 0x37602511e7aa65fe, 0x8a22f8c52bfb972e, 0x0d5451b858464847,
    0x2cafb799147f5701, 0xabd91ee467c28868, 0x169bc330ab937ab8, 0x91ed6a4dd82ea5d1,
    0x58c75eca6ba70c73, 0xdfb1f7b7181ad31a, 0x62f32a63d44b21ca, 0xe585831ea7f6fea3,
    0x2104e6214c391f46, 0xa6724f5c3f84c02f, 0x1b309288f3d532ff, 0x9c463bf58068ed96,
    0x556c0f7233e14434, 0xd21aa60f405c9b5d, 0x6f587bdb8c0d698d, 0xe82ed2a6ffb0b6e4,
    0xc9d53487b389a9a2, 0x4ea39dfac03476cb, 0xf3e1402e0c65841b, 0x7497e9537fd85b72,
    0xbdbdddd4cc51f2d0, 0x3acb74a9bfec2db9, 0x8789a97d73bddf69, 0x00ff000000000000},
   {0xc4404010dcd9d394, 0x3b2e0e0f9299eb2a, 0x0e45fa7d18ce3183, 0xf12bb462568e093d,
    0x649212980c6184d1, 0x9bfc5c874221bc6f, 0xae97a8f5c87666c6, 0x51f9e6ea86365e78,
    0xb13dc352253eee75, 0x4e538d4d6b7ed6cb, 0x7b38793fe1290c62, 0x84563720af6934dc,
    0x11ef91daf586b930, 0xee81dfc5bbc6818e, 0xdbea2bb731915b27, 0x248465a87fd16399,
    0x2ebb46952f17a856, 0xd1d5088a615790e8, 0xe4befcf8eb004a41, 0x1bd0b2e7a54072ff,
    0x8e69141dffafff13, 0x71075a02b1efc7ad, 0x446cae703bb81d04, 0xbb02e06f75f825ba,
    0x5bc6c5d7d6f095b7, 0xa4a88bc898b0ad09, 0x91c37fba12e777a0, 0x6ead31a55ca74f1e,
    0xfb14975f0648c2f2, 0x047ad9404808fa4c, 0x31112d32c25f20e5, 0xce7f632d8c1f185b,
    0x256f6b4863d2b77b, 0xda0125572d928fc5, 0xef6ad125a7c5556c, 0x10049f3ae9856dd2,
    0x85bd39c0b36ae03e, 0x7ad377dffd2ad880, 0x4fb883ad777d0229, 0xb0d6cdb2393d3a97,
    0x5012e80a9a358a9a, 0xaf7ca615d475b224, 0x9a1752675e22688d, 0x65791c7810625033,
    0xf0c0ba824a8ddddf, 0x0faef49d04cde561, 0x3ac500ef8e9a3fc8, 0xc5ab4ef0c0da0776,
    0xcf946dcd901cccb9, 0x30fa23d2de5cf407, 0x0591d7a0540b2eae, 0xfaff99bf1a4b1610,
    0x6f463f4540a49bfc, 0x9028715a0ee4a342, 0xa543852884b379eb, 0x5a2dcb37caf34155,
    0xbae9ee8f69fbf158, 0x4587a09027bbc9e6, 0x70ec54e2adec134f, 0x8f821afde3ac2bf1,
    0x1a3bbc07b943a61d, 0xe555f218f7039ea3, 0xd03e066a7d54440a, 0x2f50487533147cb4,
    0x32c730f2fa588921, 0xcda97eedb418b19f, 0xf8c28a9f3e4f6b36, 0x07acc480700f5388,
    0x9215627a2ae0de64, 0x6d7b2c6564a0e6da, 0x5810d817eef73c73, 0xa77e9608a0b704cd,
    0x47bab3b003bfb4c0, 0xb8d4fdaf4dff8c7e, 0x8dbf09ddc7a856d7, 0x72d147c289e86e69,
    0xe768e138d307e385, 0x1806af279d47db3b, 0x2d6d5b5517100192, 0xd203154a5950392c,
    0xd83c36770996f2e3, 0x2752786847d6ca5d, 0x12398c1acd8110f4, 0xed57c20583c1284a,
    0x78ee64ffd92ea5a6, 0x87802ae0976e9d18, 0xb2ebde921d3947b1, 0x4d85908d53797f0f,
    0xad41b535f071cf02, 0x522ffb2abe31f7bc, 0x67440f5834662d15, 0x982a41477a2615ab,
    0x0d93e7bd20c99847, 0xf2fda9a26e89a0f9, 0xc7965dd0e4de7a50, 0x38f813cfaa9e42ee,
    0xd3e81baa4553edce, 0x2c8655b50b13d570, 0x19eda1c781440fd9, 0xe683efd8cf043767,
    0x733a492295ebba8b, 0x8c54073ddbab8235, 0xb93ff34f51fc589c, 0x4651bd501fbc6022,
    0xa69598e8bcb4d02f, 0x59fbd6f7f2f4e891, 0x6c90228578a33238, 0x93fe6c9a36e30a86,
    0x0647ca606c0c876a, 0xf929847f224cbfd4, 0xcc42700da81b657d, 0x332c3e12e65b5dc3,
    0x39131d2fb69d960c, 0xc67d5330f8ddaeb2, 0xf316a742728a741b, 0x0c78e95d3cca4ca5,
    0x99c14fa76625c149, 0x66af01b82865f9f7, 0x53c4f5caa232235e, 0xacaabbd5ec721be0,
    0x4c6e9e6d4f7aabed, 0xb300d072013a9353, 0x866b24008b6d49fa, 0x79056a1fc52d7144,
    0xecbccce59fc2fca8, 0x13d282fad182c416, 0x26b976885bd51ebf, 0xd9d7389715952601,
    0x1d978787c94cf595, 0xe2f9c998870ccd2b, 0xd7923dea0d5b1782, 0x28fc73f5431b2f3c,
    0xbd45d50f19f4a2d0, 0x422b9b1057b49a6e, 0x77406f62dde340c7, 0x882e217d93a37879,
    0x68ea04c530abc874, 0x97844ada7eebf0ca, 0xa2efbea8f4bc2a63, 0x5d81f0b7bafc12dd,
    0xc838564de0139f31, 0x37561852ae53a78f, 0x023dec2024047d26, 0xfd53a23f6a444598,
    0xf76c81023a828e57, 0x0802cf1d74c2b6e9, 0x3d693b6ffe956c40, 0xc2077570b0d554fe,
    0x57bed38aea3ad912, 0xa8d09d95a47ae1ac, 0x9dbb69e72e2d3b05, 0x62d527f8606d03bb,
    0x82110240c365b3b6, 0x7d7f4c5f8d258b08, 0x4814b82d077251a1, 0xb77af6324932691f,
    0x22c350c813dde4f3, 0xddad1ed75d9ddc4d, 0xe8c6eaa5d7ca06e4, 0x17a8a4ba998a3e5a,
    0xfcb8acdf7647917a, 0x03d6e2c03807a9c4, 0x36bd16b2b250736d, 0xc9d358adfc104bd3,
    0x5c6afe57a6ffc63f, 0xa304b048e8bffe81, 0x966f443a62e82428, 0x69010a252ca81c96,
    0x89c52f9d8fa0ac9b, 0x76ab6182c1e09425, 0x43c095f04bb74e8c, 0xbcaedbef05f77632,
    0x29177d155f18fbde, 0xd679330a1158c360, 0xe312c7789b0f19c9, 0x1c7c8967d54f2177,
    0x1643aa5a8589eab8, 0xe92de445cbc9d206, 0xdc461037419e08af, 0x23285e280fde3011,
    0xb691f8d25531bdfd, 0x49ffb6cd1b718543, 0x7c9442bf91265fea, 0x83fa0ca0df666754,
    0x633e29187c6ed759, 0x9c506707322eefe7, 0xa93b9375b879354e, 0x5655dd6af6390df0,
    0xc3ec7b90acd6801c, 0x3c82358fe296b8a2, 0x09e9c1fd68c1620b, 0xf6878fe226815ab5,
    0xeb10f765efcdaf20, 0x147eb97aa18d979e, 0x21154d082bda4d37, 0xde7b0317659a7589,
    0x4bc2a5ed3f75f865, 0xb4acebf27135c0db, 0x81c71f80fb621a72, 0x7ea9519fb52222cc,
    0x9e6d7427162a92c1, 0x61033a38586aaa7f, 0x5468ce4ad23d70d6, 0xab0680559c7d4868,
    0x3ebf26afc692c584, 0xc1d168b088d2fd3a, 0xf4ba9cc202852793, 0x0bd4d2dd4cc51f2d,
    0x01ebf1e01c03d4e2, 0xfe85bfff5243ec5c, 0xcbee4b8dd81436f5, 0x3480059296540e4b,
    0xa139a368ccbb83a7, 0x5e57ed7782fbbb19, 0x6b3c190508ac61b0, 0x9452571a46ec590e,
    0x749672a2e5e4e903, 0x8bf83cbdaba4d1bd, 0xbe93c8cf21f30b14, 0x41fd86d06fb333aa,
    0xd444202a355cbe46, 0x2b2a6e357b1c86f8, 0x1e419a47f14b5c51, 0xe12fd458bf0b64ef,
    0x0a3fdc3d50c6cbcf, 0xf55192221e86f371, 0xc03a665094d129d8, 0x3f54284fda911166,
    0xaaed8eb5807e9c8a, 0x5583c0aace3ea434, 0x60e834d844697e9d, 0x9f867ac70a294623,
    0x7f425f7fa921f62e, 0x802c1160e761ce90, 0xb547e5126d361439, 0x4a29ab0d23762c87,
    0xdf900df77999a16b, 0x20fe43e837d999d5, 0x1595b79abd8e437c, 0xeafbf985f3ce7bc2,
    0xe0c4dab8a308b00d, 0x1faa94a7ed4888b3, 0x2ac160d5671f521a, 0xd5af2eca295f6aa4,
    0x4016883073b0e748, 0xbf78c62f3df0dff6, 0x8a13325db7a7055f, 0x757d7c42f9e73de1,
    0x95b959fa5aef8dec, 0x6ad717e514afb552, 0x5fbce3979ef86ffb, 0xa0d2ad88d0b85745,
    0x356b0b728a57daa9, 0xca05456dc417e217, 0xff6eb11f4e4038be, 0x0000ff0000000000},
   {0x2b2ac921ac68c968, 0xa93bdd5d1691aa6e, 0x1bd1c78b810d9c0f, 0x99c0d3f73bf4ff09,
    0x4adcd475f6a263a6, 0xc8cdc0094c5b00a0, 0x7a27dadfdbc736c1, 0xf836cea3613e55c7,
    0xe8c6f38919fd9cf4, 0x6ad7e7f5a304fff2, 0xd83dfd233498c993, 0x5a2ce95f8e61aa95,
    0x8930eedd4337363a, 0x0b21faa1f9ce553c, 0xb9cbe0776e52635d, 0x3bdaf40bd4ab005b,
    0x982b9a239fd5f13b, 0x1a3a8e5f252c923d, 0xa8d09489b2b0a45c, 0x2ac180f50849c75a,
    0xf9dd8777c51f5bf5, 0x7bcc930b7fe638f3, 0xc92689dde87a0e92, 0x4b379da152836d94,
    0x5bc7a08b2a40a4a7, 0xd9d6b4f790b9c7a1, 0x6b3cae210725f1c0, 0xe92dba5dbddc92c6,
    0x3a31bddf708a0e69, 0xb820a9a3ca736d6f, 0x0acab3755def5b0e, 0x88dba709e7163808,
    0x79f1497693852aa5, 0xfbe05d0a297c49a3, 0x490a47dcbee07fc2, 0xcb1b53a004191cc4,
    0x18075422c94f806b, 0x9a16405e73b6e36d, 0x28fc5a88e42ad50c, 0xaaed4ef45ed3b60a,
    0xba1d73de26107f39, 0x380c67a29ce91c3f, 0x8ae67d740b752a5e, 0x08f76908b18c4958,
    0xdbeb6e8a7cdad5f7, 0x59fa7af6c623b6f1, 0xeb10602051bf8090, 0x6901745ceb46e396,
    0xcaf01a74a03812f6, 0x48e10e081ac171f0, 0xfa0b14de8d5d4791, 0x781a00a237a42497,
    0xab060720faf2b838, 0x2917135c400bdb3e, 0x9bfd098ad797ed5f, 0x19ec1df66d6e8e59,
    0x091c20dc15ad476a, 0x8b0d34a0af54246c, 0x39e72e7638c8120d, 0xbbf63a0a8231710b,
    0x68ea3d884f67eda4, 0xeafb29f4f59e8ea2, 0x581133226202b8c3, 0xda00275ed8fbdbc5,
    0x8e9dc98fd3b30ef2, 0x0c8cddf3694a6df4, 0xbe66c725fed65b95, 0x3c77d359442f3893,
    0xef6bd4db8979a43c, 0x6d7ac0a73380c73a, 0xdf90da71a41cf15b, 0x5d81ce0d1ee5925d,
    0x4d71f32766265b6e, 0xcf60e75bdcdf3868, 0x7d8afd8d4b430e09, 0xff9be9f1f1ba6d0f,
    0x2c87ee733cecf1a0, 0xae96fa0f861592a6, 0x1c7ce0d91189a4c7, 0x9e6df4a5ab70c7c1,
    0x3d9c9a8de00e36a1, 0xbf8d8ef15af755a7, 0x0d679427cd6b63c6, 0x8f76805b779200c0,
    0x5c6a87d9bac49c6f, 0xde7b93a5003dff69, 0x6c91897397a1c908, 0xee809d0f2d58aa0e,
    0xfe70a025559b633d, 0x7c61b459ef62003b, 0xce8bae8f78fe365a, 0x4c9abaf3c207555c,
    0x9f86bd710f51c9f3, 0x1d97a90db5a8aaf5, 0xaf7db3db22349c94, 0x2d6ca7a798cdff92,
    0xdc4649d8ec5eed3f, 0x5e575da456a78e39, 0xecbd4772c13bb858, 0x6eac530e7bc2db5e,
    0xbdb0548cb69447f1, 0x3fa140f00c6d24f7, 0x8d4b5a269bf11296, 0x0f5a4e5a21087190,
    0x1faa737059cbb8a3, 0x9dbb670ce332dba5, 0x2f517dda74aeedc4, 0xad4069a6ce578ec2,
    0x7e5c6e240301126d, 0xfc4d7a58b9f8716b, 0x4ea7608e2e64470a, 0xccb674f2949d240c,
    0x6f471adadfe3d56c, 0xed560ea6651ab66a, 0x5fbc1470f286800b, 0xddad000c487fe30d,
    0x0eb1078e85297fa2, 0x8ca013f23fd01ca4, 0x3e4a0924a84c2ac5, 0xbc5b1d5812b549c3,
    0xacab20726a7680f0, 0x2eba340ed08fe3f6, 0x9c502ed84713d597, 0x1e413aa4fdeab691,
    0xcd5d3d2630bc2a3e, 0x4f4c295a8a454938, 0xfda6338c1dd97f59, 0x7fb727f0a7201c5f,
    0x549dee2e0b48d537, 0xd68cfa52b1b1b631, 0x6466e084262d8050, 0xe677f4f89cd4e356,
    0x356bf37a51827ff9, 0xb77ae706eb7b1cff, 0x0590fdd07ce72a9e, 0x8781e9acc61e4998,
    0x9771d486bedd80ab, 0x1560c0fa0424e3ad, 0xa78ada2c93b8d5cc, 0x259bce502941b6ca,
    0xf687c9d2e4172a65, 0x7496ddae5eee4963, 0xc67cc778c9727f02, 0x446dd304738b1c04,
    0xe79cbd2c38f5ed64, 0x658da950820c8e62, 0xd767b3861590b803, 0x5576a7faaf69db05,
    0x866aa078623f47aa, 0x047bb404d8c624ac, 0xb691aed24f5a12cd, 0x3480baaef5a371cb,
    0x247087848d60b8f8, 0xa66193f83799dbfe, 0x148b892ea005ed9f, 0x969a9d521afc8e99,
    0x45869ad0d7aa1236, 0xc7978eac6d537130, 0x757d947afacf4751, 0xf76c800640362457,
    0x06466e7934a536fa, 0x84577a058e5c55fc, 0x36bd60d319c0639d, 0xb4ac74afa339009b,
    0x67b0732d6e6f9c34, 0xe5a16751d496ff32, 0x574b7d87430ac953, 0xd55a69fbf9f3aa55,
    0xc5aa54d181306366, 0x47bb40ad3bc90060, 0xf5515a7bac553601, 0x77404e0716ac5507,
    0xa45c4985dbfac9a8, 0x264d5df96103aaae, 0x94a7472ff69f9ccf, 0x16b653534c66ffc9,
    0xb5473d7b07180ea9, 0x37562907bde16daf, 0x85bc33d12a7d5bce, 0x07ad27ad908438c8,
    0xd4b1202f5dd2a467, 0x56a03453e72bc761, 0xe44a2e8570b7f100, 0x665b3af9ca4e9206,
    0x76ab07d3b28d5b35, 0xf4ba13af08743833, 0x465009799fe80e52, 0xc4411d0525116d54,
    0x175d1a87e847f1fb, 0x954c0efb52be92fd, 0x27a6142dc522a49c, 0xa5b700517fdbc79a,
    0xf12aee80749312ad, 0x733bfafcce6a71ab, 0xc1d1e02a59f647ca, 0x43c0f456e30f24cc,
    0x90dcf3d42e59b863, 0x12cde7a894a0db65, 0xa027fd7e033ced04, 0x2236e902b9c58e02,
    0x32c6d428c1064731, 0xb0d7c0547bff2437, 0x023dda82ec631256, 0x802ccefe569a7150,
    0x5330c97c9bccedff, 0xd121dd0021358ef9, 0x63cbc7d6b6a9b898, 0xe1dad3aa0c50db9e,
    0x422bbd82472e2afe, 0xc03aa9fefdd749f8, 0x72d0b3286a4b7f99, 0xf0c1a754d0b21c9f,
    0x23dda0d61de48030, 0xa1ccb4aaa71de336, 0x1326ae7c3081d557, 0x9137ba008a78b651,
    0x81c7872af2bb7f62, 0x03d6935648421c64, 0xb13c8980dfde2a05, 0x332d9dfc65274903,
    0xe0319a7ea871d5ac, 0x62208e021288b6aa, 0xd0ca94d4851480cb, 0x52db80a83fede3cd,
    0xa3f16ed74b7ef160, 0x21e07aabf1879266, 0x930a607d661ba407, 0x111b7401dce2c701,
    0xc207738311b45bae, 0x401667ffab4d38a8, 0xf2fc7d293cd10ec9, 0x70ed695586286dcf,
    0x601d547ffeeba4fc, 0xe20c40034412c7fa, 0x50e65ad5d38ef19b, 0xd2f74ea96977929d,
    0x01eb492ba4210e32, 0x83fa5d571ed86d34, 0x3110478189445b55, 0xb30153fd33bd3853,
    0x10f03dd578c3c933, 0x92e129a9c23aaa35, 0x200b337f55a69c54, 0xa21a2703ef5fff52,
    0x71062081220963fd, 0xf31734fd98f000fb, 0x41fd2e2b0f6c369a, 0xc3ec3a57b595559c,
    0xd31c077dcd569caf, 0x510d130177afffa9, 0xe3e709d7e033c9c8, 0x61f61dab5acaaace,
    0xb2ea1a29979c3661, 0x30fb0e552d655567, 0x82111483baf96306, 0x000000ff00000000},
   {0x80ace2b722aa64d1, 0xb791f740a63a39cf, 0xeed6c9582b8adeed, 0xd9ebdcafaf1a83f3,
    0x5c58b56930eb10a9, 0x6b65a09eb47b4db7, 0x32229e8639cbaa95, 0x051f8b71bd5bf78b,
    0x0d9d6b585ebf1f4a, 0x3aa07eafda2f4254, 0x63e740b7579fa576, 0x54da5540d30ff868,
    0xd1693c864cfe6b32, 0xe6542971c86e362c, 0xbf13176945ded10e, 0x882e029ec14e8c10,
    0xae16d73a8217008c, 0x992bc2cd06875d92, 0xc06cfcd58b37bab0, 0xf751e9220fa7e7ae,
    0x72e280e4905674f4, 0x45df951314c629ea, 0x1c98ab0b9976cec8, 0x2ba5befc1de693d6,
    0x23275ed5fe027b17, 0x141a4b227a922609, 0x4d5d753af722c12b, 0x7a6060cd73b29c35,
    0xffd3090bec430f6f, 0xc8ee1cfc68d35271, 0x91a922e4e563b553, 0xa694371361f3e84d,
    0xddd889ac63d0ac6b, 0xeae59c5be740f175, 0xb3a2a2436af01657, 0x849fb7b4ee604b49,
    0x012cde727191d813, 0x3611cb85f501850d, 0x6f56f59d78b1622f, 0x586be06afc213f31,
    0x50e900431fc5d7f0, 0x67d415b49b558aee, 0x3e932bac16e56dcc, 0x09ae3e5b927530d2,
    0x8c1d579d0d84a388, 0xbb20426a8914fe96, 0xe2677c7204a419b4, 0xd55a6985803444aa,
    0xf362bc21c36dc836, 0xc45fa9d647fd9528, 0x9d1897ceca4d720a, 0xaa2582394edd2f14,
    0x2f96ebffd12cbc4e, 0x18abfe0855bce150, 0x41ecc010d80c0672, 0x76d1d5e75c9c5b6c,
    0x7e5335cebf78b3ad, 0x496e20393be8eeb3, 0x10291e21b6580991, 0x27140bd632c8548f,
    0xa2a76210ad39c7d5, 0x959a77e729a99acb, 0xccdd49ffa4197de9, 0xfbe05c08208920f7,
    0x3a443481a05ff5a5, 0x0d79217624cfa8bb, 0x543e1f6ea97f4f99, 0x63030a992def1287,
    0xe6b0635fb21e81dd, 0xd18d76a8368edcc3, 0x88ca48b0bb3e3be1, 0xbff75d473fae66ff,
    0xb775bd6edc4a8e3e, 0x8048a89958dad320, 0xd90f9681d56a3402, 0xee32837651fa691c,
    0x6b81eab0ce0bfa46, 0x5cbcff474a9ba758, 0x05fbc15fc72b407a, 0x32c6d4a843bb1d64,
    0x14fe010c00e291f8, 0x23c314fb8472cce6, 0x7a842ae309c22bc4, 0x4db93f148d5276da,
    0xc80a56d212a3e580, 0xff3743259633b89e, 0xa6707d3d1b835fbc, 0x914d68ca9f1302a2,
    0x99cf88e37cf7ea63, 0xaef29d14f867b77d, 0xf7b5a30c75d7505f, 0xc088b6fbf1470d41,
    0x453bdf3d6eb69e1b, 0x7206cacaea26c305, 0x2b41f4d267962427, 0x1c7ce125e3067939,
    0x67305f9ae1253d1f, 0x500d4a6d65b56001, 0x094a7475e8058723, 0x3e7761826c95da3d,
    0xbbc40844f3644967, 0x8cf91db377f41479, 0xd5be23abfa44f35b, 0xe283365c7ed4ae45,
    0xea01d6759d304684, 0xdd3cc38219a01b9a, 0x847bfd9a9410fcb8, 0xb346e86d1080a1a6,
    0x36f581ab8f7132fc, 0x01c8945c0be16fe2, 0x588faa44865188c0, 0x6fb2bfb302c1d5de,
    0x498a6a1741985942, 0x7eb77fe0c508045c, 0x27f041f848b8e37e, 0x10cd540fcc28be60,
    0x957e3dc953d92d3a, 0xa243283ed7497024, 0xfb0416265af99706, 0xcc3903d1de69ca18,
    0xc4bbe3f83d8d22d9, 0xf386f60fb91d7fc7, 0xaac1c81734ad98e5, 0x9dfcdde0b03dc5fb,
    0x184fb4262fcc56a1, 0x2f72a1d1ab5c0bbf, 0x76359fc926ecec9d, 0x41088a3ea27cb183,
    0xc1a468897fd6d552, 0xf6997d7efb46884c, 0xafde436676f66f6e, 0x98e35691f2663270,
    0x1d503f576d97a12a, 0x2a6d2aa0e907fc34, 0x732a14b864b71b16, 0x4417014fe0274608,
    0x4c95e16603c3aec9, 0x7ba8f4918753f3d7, 0x22efca890ae314f5, 0x15d2df7e8e7349eb,
    0x9061b6b81182dab1, 0xa75ca34f951287af, 0xfe1b9d5718a2608d, 0xc92688a09c323d93,
    0xef1e5d04df6bb10f, 0xd82348f35bfbec11, 0x816476ebd64b0b33, 0xb659631c52db562d,
    0x33ea0adacd2ac577, 0x04d71f2d49ba9869, 0x5d902135c40a7f4b, 0x6aad34c2409a2255,
    0x622fd4eba37eca94, 0x5512c11c27ee978a, 0x0c55ff04aa5e70a8, 0x3b68eaf32ece2db6,
    0xbedb8335b13fbeec, 0x89e696c235afe3f2, 0xd0a1a8dab81f04d0, 0xe79cbd2d3c8f59ce,
    0x9cd003923eac1de8, 0xabed1665ba3c40f6, 0xf2aa287d378ca7d4, 0xc5973d8ab31cfaca,
    0x4024544c2ced6990, 0x771941bba87d348e, 0x2e5e7fa325cdd3ac, 0x19636a54a15d8eb2,
    0x11e18a7d42b96673, 0x26dc9f8ac6293b6d, 0x7f9ba1924b99dc4f, 0x48a6b465cf098151,
    0xcd15dda350f8120b, 0xfa28c854d4684f15, 0xa36ff64c59d8a837, 0x9452e3bbdd48f529,
    0xb26a361f9e1179b5, 0x855723e81a8124ab, 0xdc101df09731c389, 0xeb2d080713a19e97,
    0x6e9e61c18c500dcd, 0x59a3743608c050d3, 0x00e44a2e8570b7f1, 0x37d95fd901e0eaef,
    0x3f5bbff0e204022e, 0x0866aa0766945f30, 0x5121941feb24b812, 0x661c81e86fb4e50c,
    0xe3afe82ef0457656, 0xd492fdd974d52b48, 0x8dd5c3c1f965cc6a, 0xbae8d6367df59174,
    0x7b4cbebffd234426, 0x4c71ab4879b31938, 0x15369550f403fe1a, 0x220b80a77093a304,
    0xa7b8e961ef62305e, 0x9085fc966bf26d40, 0xc9c2c28ee6428a62, 0xfeffd77962d2d77c,
    0xf67d375081363fbd, 0xc14022a705a662a3, 0x98071cbf88168581, 0xaf3a09480c86d89f,
    0x2a89608e93774bc5, 0x1db4757917e716db, 0x44f34b619a57f1f9, 0x73ce5e961ec7ace7,
    0x55f68b325d9e207b, 0x62cb9ec5d90e7d65, 0x3b8ca0dd54be9a47, 0x0cb1b52ad02ec759,
    0x8902dcec4fdf5403, 0xbe3fc91bcb4f091d, 0xe778f70346ffee3f, 0xd045e2f4c26fb321,
    0xd8c702dd218b5be0, 0xeffa172aa51b06fe, 0xb6bd293228abe1dc, 0x81803cc5ac3bbcc2,
    0x0433550333ca2f98, 0x330e40f4b75a7286, 0x6a497eec3aea95a4, 0x5d746b1bbe7ac8ba,
    0x2638d5a4bc598c9c, 0x1105c05338c9d182, 0x4842fe4bb57936a0, 0x7f7febbc31e96bbe,
    0xfacc827aae18f8e4, 0xcdf1978d2a88a5fa, 0x94b6a995a73842d8, 0xa38bbc6223a81fc6,
    0xab095c4bc04cf707, 0x9c3449bc44dcaa19, 0xc57377a4c96c4d3b, 0xf24e62534dfc1025,
    0x77fd0b95d20d837f, 0x40c01e62569dde61, 0x1987207adb2d3943, 0x2eba358d5fbd645d,
    0x0882e0291ce4e8c1, 0x3fbff5de9874b5df, 0x66f8cbc615c452fd, 0x51c5de3191540fe3,
    0xd476b7f70ea59cb9, 0xe34ba2008a35c1a7, 0xba0c9c1807852685, 0x8d3189ef83157b9b,
    0x85b369c660f1935a, 0xb28e7c31e461ce44, 0xebc9422969d12966, 0xdcf457deed417478,
    0x59473e1872b0e722, 0x6e7a2beff620ba3c, 0x373d15f77b905d1e, 0x00000000ff000000},
   {0x4a38ea85d98ca846, 0xa34cc0fc36880d92, 0xac0998255f127085, 0x457db25cb016d551,
    0xb28329978c268aab, 0x5bf703ee63222f7f, 0x54b25b370ab85268, 0xbdc6714ee5bcf7bc,
    0x8f964af22a4f7ef7, 0x66e2608bc54bdb23, 0x69a73852acd1a634, 0x80d3122b43d503e0,
    0x772d89e07fe55c1a, 0x9e59a39990e1f9ce, 0x911cfb40f97b84d9, 0x7868d139167f210d,
    0xf5bc8c39669c964f, 0x1cc8a6408998339b, 0x138dfe99e0024e8c, 0xfaf9d4e00f06eb58,
    0x0d074f2b3336b4a2, 0xe4736552dc321176, 0xeb363d8bb5a86c61, 0x024217f25aacc9b5,
    0x30122c4e955f40fe, 0xd96606377a5be52a, 0xd6235eee13c1983d, 0x3f577497fcc53de9,
    0xc8a9ef5cc0f56213, 0x21ddc5252ff1c7c7, 0x2e989dfc466bbad0, 0xc7ecb785a96f1f04,
    0x01e901afff3b473f, 0xe89d2bd6103fe2eb, 0xe7d8730f79a59ffc, 0x0eac597696a13a28,
    0xf952c2bdaa9165d2, 0x1026e8c44595c006, 0x1f63b01d2c0fbd11, 0xf6179a64c30b18c5,
    0xc447a1d80cf8918e, 0x2d338ba1e3fc345a, 0x2276d3788a66494d, 0xcb02f9016562ec99,
    0x3cfc62ca5952b363, 0xd58848b3b65616b7, 0xdacd106adfcc6ba0, 0x33b93a1330c8ce74,
    0xbe6d6713402b7936, 0x57194d6aaf2fdce2, 0x585c15b3c6b5a1f5, 0xb1283fca29b10421,
    0x46d6a40115815bdb, 0xafa28e78fa85fe0f, 0xa0e7d6a1931f8318, 0x4993fcd87c1b26cc,
    0x7bc3c764b3e8af87, 0x92b7ed1d5cec0a53, 0x9df2b5c435767744, 0x74869fbdda72d290,
    0x83780476e6428d6a, 0x6a0c2e0f094628be, 0x654976d660dc55a9, 0x8c3d5caf8fd8f07d,
    0xdd9b3cd194e376b4, 0x34ef16a87be7d360, 0x3baa4e71127dae77, 0xd2de6408fd790ba3,
    0x2520ffc3c1495459, 0xcc54d5ba2e4df18d, 0xc3118d6347d78c9a, 0x2a65a71aa8d3294e,
    0x18359ca66720a005, 0xf141b6df882405d1, 0xfe04ee06e1be78c6, 0x1770c47f0ebadd12,
    0xe08e5fb4328a82e8, 0x09fa75cddd8e273c, 0x06bf2d14b4145a2b, 0xefcb076d5b10ffff,
    0x621f5a6d2bf348bd, 0x8b6b7014c4f7ed69, 0x842e28cdad6d907e, 0x6d5a02b4426935aa,
    0x9aa4997f7e596a50, 0x73d0b306915dcf84, 0x7c95ebdff8c7b293, 0x95e1c1a617c31747,
    0xa7b1fa1ad8309e0c, 0x4ec5d06337343bd8, 0x418088ba5eae46cf, 0xa8f4a2c3b1aae31b,
    0x5f0a39088d9abce1, 0xb67e1371629e1935, 0xb93b4ba80b046422, 0x504f61d1e400c1f6,
    0x964ad7fbb25499cd, 0x7f3efd825d503c19, 0x707ba55b34ca410e, 0x990f8f22dbcee4da,
    0x6ef114e9e7febb20, 0x87853e9008fa1ef4, 0x88c06649616063e3, 0x61b44c308e64c637,
    0x53e4778c41974f7c, 0xba905df5ae93eaa8, 0xb5d5052cc70997bf, 0x5ca12f55280d326b,
    0xab5fb49e143d6d91, 0x422b9ee7fb39c845, 0x4d6ec63e92a3b552, 0xa41aec477da71086,
    0x29ceb1470d44a7c4, 0xc0ba9b3ee2400210, 0xcfffc3e78bda7f07, 0x268be99e64dedad3,
    0xd175725558ee8529, 0x3801582cb7ea20fd, 0x374400f5de705dea, 0xde302a8c3174f83e,
    0xec601130fe877175, 0x05143b491183d4a1, 0x0a5163907819a9b6, 0xe32549e9971d0c62,
    0x14dbd222ab2d5398, 0xfdaff85b4429f64c, 0xf2eaa0822db38b5b, 0x1b9e8afbc2b72e8f,
    0x51a6607e1bc486c9, 0xb8d24a07f4c0231d, 0xb79712de9d5a5e0a, 0x5ee338a7725efbde,
    0xa91da36c4e6ea424, 0x40698915a16a01f0, 0x4f2cd1ccc8f07ce7, 0xa658fbb527f4d933,
    0x9408c009e8075078, 0x7d7cea700703f5ac, 0x7239b2a96e9988bb, 0x9b4d98d0819d2d6f,
    0x6cb3031bbdad7295, 0x85c7296252a9d741, 0x8a8271bb3b33aa56, 0x63f65bc2d4370f82,
    0xee2206c2a4d4b8c0, 0x07562cbb4bd01d14, 0x08137462224a6003, 0xe1675e1bcd4ec5d7,
    0x1699c5d0f17e9a2d, 0xffedefa91e7a3ff9, 0xf0a8b77077e042ee, 0x19dc9d0998e4e73a,
    0x2b8ca6b557176e71, 0xc2f88cccb813cba5, 0xcdbdd415d189b6b2, 0x24c9fe6c3e8d1366,
    0xd33765a702bd4c9c, 0x3a434fdeedb9e948, 0x350617078423945f, 0xdc723d7e6b27318b,
    0x1a778b543d7369b0, 0xf303a12dd277cc64, 0xfc46f9f4bbedb173, 0x1532d38d54e914a7,
    0xe2cc484668d94b5d, 0x0bb8623f87ddee89, 0x04fd3ae6ee47939e, 0xed89109f0143364a,
    0xdfd92b23ceb0bf01, 0x36ad015a21b41ad5, 0x39e85983482e67c2, 0xd09c73faa72ac216,
    0x2762e8319b1a9dec, 0xce16c248741e3838, 0xc1539a911d84452f, 0x2827b0e8f280e0fb,
    0xa5f3ede8826357b9, 0x4c87c7916d67f26d, 0x43c29f4804fd8f7a, 0xaab6b531ebf92aae,
    0x5d482efad7c97554, 0xb43c048338cdd080, 0xbb795c5a5157ad97, 0x520d7623be530843,
    0x605d4d9f71a08108, 0x892967e69ea424dc, 0x866c3f3ff73e59cb, 0x6f181546183afc1f,
    0x98e68e8d240aa3e5, 0x7192a4f4cb0e0631, 0x7ed7fc2da2947b26, 0x97a3d6544d90def2,
    0xc605b62a56ab583b, 0x2f719c53b9affdef, 0x2034c48ad03580f8, 0xc940eef33f31252c,
    0x3ebe753803017ad6, 0xd7ca5f41ec05df02, 0xd88f0798859fa215, 0x31fb2de16a9b07c1,
    0x03ab165da5688e8a, 0xeadf3c244a6c2b5e, 0xe59a64fd23f65649, 0x0cee4e84ccf2f39d,
    0xfb10d54ff0c2ac67, 0x1264ff361fc609b3, 0x1d21a7ef765c74a4, 0xf4558d969958d170,
    0x7981d096e9bb6632, 0x90f5faef06bfc3e6, 0x9fb0a2366f25bef1, 0x76c4884f80211b25,
    0x813a1384bc1144df, 0x684e39fd5315e10b, 0x670b61243a8f9c1c, 0x8e7f4b5dd58b39c8,
    0xbc2f70e11a78b083, 0x555b5a98f57c1557, 0x5a1e02419ce66840, 0xb36a283873e2cd94,
    0x4494b3f34fd2926e, 0xade0998aa0d637ba, 0xa2a5c153c94c4aad, 0x4bd1eb2a2648ef79,
    0x8dd45d00701cb742, 0x64a077799f181296, 0x6be52fa0f6826f81, 0x829105d91986ca55,
    0x756f9e1225b695af, 0x9c1bb46bcab2307b, 0x935eecb2a3284d6c, 0x7a2ac6cb4c2ce8b8,
    0x487afd7783df61f3, 0xa10ed70e6cdbc427, 0xae4b8fd70541b930, 0x473fa5aeea451ce4,
    0xb0c13e65d675431e, 0x59b5141c3971e6ca, 0x56f04cc550eb9bdd, 0xbf8466bcbfef3e09,
    0x32503bbccf0c894b, 0xdb2411c520082c9f, 0xd461491c49925188, 0x3d156365a696f45c,
    0xcaebf8ae9aa6aba6, 0x239fd2d775a20e72, 0x2cda8a0e1c387365, 0xc5aea077f33cd6b1,
    0xf7fe9bcb3ccf5ffa, 0x1e8ab1b2d3cbfa2e, 0x11cfe96bba518739, 0xf8bbc312555522ed,
    0x0f4558d969657d17, 0xe63172a08661d8c3, 0xe9742a79effba5d4, 0x0000000000ff0000},
   {0xe0afe6b4079d05ba, 0x1cf2c142b8a83ccb, 0x2ccc8f0a2160e433, 0xd091a8fc9e55dd42,
    0x4cb0139b12f155c3, 0xb0ed346dadc46cb2, 0x80d37a25340cb44a, 0x7c8e5dd38b398d3b,
    0x8c492ab975d23623, 0x70140d4fcae70f52, 0x402a4307532fd7aa, 0xbc7764f1ec1aeedb,
    0x2056df9660be665a, 0xdc0bf860df8b5f2b, 0xec35b628464387d3, 0x106891def976bea2,
    0x39627eaee3036288, 0xc53f59585c365bf9, 0xf5011710c5fe8301, 0x095c30e67acbba70,
    0x957d8b81f66f32f1, 0x6920ac77495a0b80, 0x591ee23fd092d378, 0xa543c5c96fa7ea09,
    0x5584b2a3914c5111, 0xa9d995552e796860, 0x99e7db1db7b1b098, 0x65bafceb088489e9,
    0xf99b478c84200168, 0x05c6607a3b153819, 0x35f82e32a2dde0e1, 0xc9a509c41de8d990,
    0x67edf0d2963658b5, 0x9bb0d724290361c4, 0xab8e996cb0cbb93c, 0x57d3be9a0ffe804d,
    0xcbf205fd835a08cc, 0x37af220b3c6f31bd, 0x07916c43a5a7e945, 0xfbcc4bb51a92d034,
    0x0b0b3cdfe4796b2c, 0xf7561b295b4c525d, 0xc7685561c2848aa5, 0x3b3572977db1b3d4,
    0xa714c9f0f1153b55, 0x5b49ee064e200224, 0x6b77a04ed7e8dadc, 0x972a87b868dde3ad,
    0xbe2068c872a83f87, 0x427d4f3ecd9d06f6, 0x724301765455de0e, 0x8e1e2680eb60e77f,
    0x123f9de767c46ffe, 0xee62ba11d8f1568f, 0xde5cf45941398e77, 0x2201d3affe0cb706,
    0xd2c6a4c500e70c1e, 0x2e9b8333bfd2356f, 0x1ea5cd7b261aed97, 0xe2f8ea8d992fd4e6,
    0x7ed951ea158b5c67, 0x8284761caabe6516, 0xb2ba38543376bdee, 0x4ee71fa28c43849f,
    0xdaf2ec2a7c5c2ccf, 0x26afcbdcc36915be, 0x169185945aa1cd46, 0xeacca262e594f437,
    0x76ed190569307cb6, 0x8ab03ef3d60545c7, 0xba8e70bb4fcd9d3f, 0x46d3574df0f8a44e,
    0xb61420270e131f56, 0x4a4907d1b1262627, 0x7a77499928eefedf, 0x862a6e6f97dbc7ae,
    0x1a0bd5081b7f4f2f, 0xe656f2fea44a765e, 0xd668bcb63d82aea6, 0x2a359b4082b797d7,
    0x033f743098c24bfd, 0xff6253c627f7728c, 0xcf5c1d8ebe3faa74, 0x33013a78010a9305,
    0xaf20811f8dae1b84, 0x537da6e9329b22f5, 0x6343e8a1ab53fa0d, 0x9f1ecf571466c37c,
    0x6fd9b83dea8d7864, 0x93849fcb55b84115, 0xa3bad183cc7099ed, 0x5fe7f6757345a09c,
    0xc3c64d12ffe1281d, 0x3f9b6ae440d4116c, 0x0fa524acd91cc994, 0xf3f8035a6629f0e5,
    0x5db0fa4cedf771c0, 0xa1edddba52c248b1, 0x91d393f2cb0a9049, 0x6d8eb404743fa938,
    0xf1af0f63f89b21b9, 0x0df2289547ae18c8, 0x3dcc66ddde66c030, 0xc191412b6153f941,
    0x315636419fb84259, 0xcd0b11b7208d7b28, 0xfd355fffb945a3d0, 0x0168780906709aa1,
    0x9d49c36e8ad41220, 0x6114e49835e12b51, 0x512aaad0ac29f3a9, 0xad778d26131ccad8,
    0x847d6256096916f2, 0x782045a0b65c2f83, 0x481e0be82f94f77b, 0xb4432c1e90a1ce0a,
    0x286297791c05468b, 0xd43fb08fa3307ffa, 0xe401fec73af8a702, 0x185cd93185cd9e73,
    0xe89bae5b7b26256b, 0x14c689adc4131c1a, 0x24f8c7e55ddbc4e2, 0xd8a5e013e2eefd93,
    0x44845b746e4a7512, 0xb8d97c82d17f4c63, 0x88e732ca48b7949b, 0x74ba153cf782adea,
    0x9415f388f01f5750, 0x6848d47e4f2a6e21, 0x58769a36d6e2b6d9, 0xa42bbdc069d78fa8,
    0x380a06a7e5730729, 0xc45721515a463e58, 0xf4696f19c38ee6a0, 0x083448ef7cbbdfd1,
    0xf8f33f85825064c9, 0x04ae18733d655db8, 0x3490563ba4ad8540, 0xc8cd71cd1b98bc31,
    0x54eccaaa973c34b0, 0xa8b1ed5c28090dc1, 0x988fa314b1c1d539, 0x64d284e20ef4ec48,
    0x4dd86b9214813062, 0xb1854c64abb40913, 0x81bb022c327cd1eb, 0x7de625da8d49e89a,
    0xe1c79ebd01ed601b, 0x1d9ab94bbed8596a, 0x2da4f70327108192, 0xd1f9d0f59825b8e3,
    0x213ea79f66ce03fb, 0xdd638069d9fb3a8a, 0xed5dce214033e272, 0x1100e9d7ff06db03,
    0x8d2152b073a25382, 0x717c7546cc976af3, 0x41423b0e555fb20b, 0xbd1f1cf8ea6a8b7a,
    0x1357e5ee61b40a5f, 0xef0ac218de81332e, 0xdf348c504749ebd6, 0x2369aba6f87cd2a7,
    0xbf4810c174d85a26, 0x43153737cbed6357, 0x732b797f5225bbaf, 0x8f765e89ed1082de,
    0x7fb129e313fb39c6, 0x83ec0e15acce00b7, 0xb3d2405d3506d84f, 0x4f8f67ab8a33e13e,
    0xd3aedccc069769bf, 0x2ff3fb3ab9a250ce, 0x1fcdb572206a8836, 0xe39092849f5fb147,
    0xca9a7df4852a6d6d, 0x36c75a023a1f541c, 0x06f9144aa3d78ce4, 0xfaa433bc1ce2b595,
    0x668588db90463d14, 0x9ad8af2d2f730465, 0xaae6e165b6bbdc9d, 0x56bbc693098ee5ec,
    0xa67cb1f9f7655ef4, 0x5a21960f48506785, 0x6a1fd847d198bf7d, 0x9642ffb16ead860c,
    0x0a6344d6e2090e8d, 0xf63e63205d3c37fc, 0xc6002d68c4f4ef04, 0x3a5d0a9e7bc1d675,
    0xae48f9168bde7e25, 0x5215dee034eb4754, 0x622b90a8ad239fac, 0x9e76b75e1216a6dd,
    0x02570c399eb22e5c, 0xfe0a2bcf2187172d, 0xce346587b84fcfd5, 0x32694271077af6a4,
    0xc2ae351bf9914dbc, 0x3ef312ed46a474cd, 0x0ecd5ca5df6cac35, 0xf2907b5360599544,
    0x6eb1c034ecfd1dc5, 0x92ece7c253c824b4, 0xa2d2a98aca00fc4c, 0x5e8f8e7c7535c53d,
    0x7785610c6f401917, 0x8bd846fad0752066, 0xbbe608b249bdf89e, 0x47bb2f44f688c1ef,
    0xdb9a94237a2c496e, 0x27c7b3d5c519701f, 0x17f9fd9d5cd1a8e7, 0xeba4da6be3e49196,
    0x1b63ad011d0f2a8e, 0xe73e8af7a23a13ff, 0xd700c4bf3bf2cb07, 0x2b5de34984c7f276,
    0xb77c582e08637af7, 0x4b217fd8b7564386, 0x7b1f31902e9e9b7e, 0x8742166691aba20f,
    0x290aef701a75232a, 0xd557c886a5401a5b, 0xe56986ce3c88c2a3, 0x1934a13883bdfbd2,
    0x85151a5f0f197353, 0x79483da9b02c4a22, 0x497673e129e492da, 0xb52b541796d1abab,
    0x45ec237d683a10b3, 0xb9b1048bd70f29c2, 0x898f4ac34ec7f13a, 0x75d26d35f1f2c84b,
    0xe9f3d6527d5640ca, 0x15aef1a4c26379bb, 0x2590bfec5baba143, 0xd9cd981ae49e9832,
    0xf0c7776afeeb4418, 0x0c9a509c41de7d69, 0x3ca41ed4d816a591, 0xc0f9392267239ce0,
    0x5cd88245eb871461, 0xa085a5b354b22d10, 0x90bbebfbcd7af5e8, 0x6ce6cc0d724fcc99,
    0x9c21bb678ca47781, 0x607c9c9133914ef0, 0x5042d2d9aa599608, 0xac1ff52f156caf79,
    0x303e4e4899c827f8, 0xcc6369be26fd1e89, 0xfc5d27f6bf35c671, 0x000000000000ff00},
   {0x4b6c30a789f108b4, 0x6a8546b9acd72918, 0x08bedc9bc3bd4bec, 0x2957aa85e69b6a40,
    0xccc9e8df1d698e04, 0xed209ec1384fafa8, 0x8f1b04e35725cd5c, 0xaef272fd7203ecf0,
    0x70fea605f85796bf, 0x5117d01bdd71b713, 0x332c4a39b21bd5e7, 0x12c53c27973df44b,
    0xf75b7e7d6ccf100f, 0xd6b2086349e931a3, 0xb489924126835357, 0x9560e45f03a572fb,
    0x3c491de36abc34a2, 0x1da06bfd4f9a150e, 0x7f9bf1df20f077fa, 0x5e7287c105d65656,
    0xbbecc59bfe24b212, 0x9a05b385db0293be, 0xf83e29a7b468f14a, 0xd9d75fb9914ed0e6,
    0x07db8b411b1aaaa9, 0x2632fd5f3e3c8b05, 0x4409677d5156e9f1, 0x65e011637470c85d,
    0x807e53398f822c19, 0xa1972527aaa40db5, 0xc3acbf05c5ce6f41, 0xe245c91be0e84eed,
    0xa5266a2e4f6b7098, 0x84cf1c306a4d5134, 0xe6f48612052733c0, 0xc71df00c2001126c,
    0x2283b256dbf3f628, 0x036ac448fed5d784, 0x61515e6a91bfb570, 0x40b82874b49994dc,
    0x9eb4fc8c3ecdee93, 0xbf5d8a921bebcf3f, 0xdd6610b07481adcb, 0xfc8f66ae51a78c67,
    0x191124f4aa556823, 0x38f852ea8f73498f, 0x5ac3c8c8e0192b7b, 0x7b2abed6c53f0ad7,
    0xd203476aac264c8e, 0xf3ea317489006d22, 0x91d1ab56e66a0fd6, 0xb038dd48c34c2e7a,
    0x55a69f1238beca3e, 0x744fe90c1d98eb92, 0x1674732e72f28966, 0x379d053057d4a8ca,
    0xe991d1c8dd80d285, 0xc878a7d6f8a6f329, 0xaa433df497cc91dd, 0x8baa4beab2eab071,
    0x6e3409b049185435, 0x4fdd7fae6c3e7599, 0x2de6e58c0354176d, 0x0c0f9392267236c1,
    0xa321a3e75c526b87, 0x82c8d5f979744a2b, 0xe0f34fdb161e28df, 0xc11a39c533380973,
    0x24847b9fc8caed37, 0x056d0d81edeccc9b, 0x675697a38286ae6f, 0x46bfe1bda7a08fc3,
    0x98b335452df4f58c, 0xb95a435b08d2d420, 0xdb61d97967b8b6d4, 0xfa88af67429e9778,
    0x1f16ed3db96c733c, 0x3eff9b239c4a5290, 0x5cc40101f3203064, 0x7d2d771fd60611c8,
    0xd4048ea3bf1f5791, 0xf5edf8bd9a39763d, 0x97d6629ff55314c9, 0xb63f1481d0753565,
    0x53a156db2b87d121, 0x724820c50ea1f08d, 0x1073bae761cb9279, 0x319accf944edb3d5,
    0xef961801ceb9c99a, 0xce7f6e1feb9fe836, 0xac44f43d84f58ac2, 0x8dad8223a1d3ab6e,
    0x6833c0795a214f2a, 0x49dab6677f076e86, 0x2be12c45106d0c72, 0x0a085a5b354b2dde,
    0x4d6bf96e9ac813ab, 0x6c828f70bfee3207, 0x0eb91552d08450f3, 0x2f50634cf5a2715f,
    0xcace21160e50951b, 0xeb2757082b76b4b7, 0x891ccd2a441cd643, 0xa8f5bb34613af7ef,
    0x76f96fcceb6e8da0, 0x571019d2ce48ac0c, 0x352b83f0a122cef8, 0x14c2f5ee8404ef54,
    0xf15cb7b47ff60b10, 0xd0b5c1aa5ad02abc, 0xb28e5b8835ba4848, 0x93672d96109c69e4,
    0x3a4ed42a79852fbd, 0x1ba7a2345ca30e11, 0x799c381633c96ce5, 0x58754e0816ef4d49,
    0xbdeb0c52ed1da90d, 0x9c027a4cc83b88a1, 0xfe39e06ea751ea55, 0xdfd096708277cbf9,
    0x01dc42880823b1b6, 0x203534962d05901a, 0x420eaeb4426ff2ee, 0x63e7d8aa6749d342,
    0x86799af09cbb3706, 0xa790eceeb99d16aa, 0xc5ab76ccd6f7745e, 0xe44200d2f3d155f2,
    0xaf2e30757a205db9, 0x8ec7466b5f067c15, 0xecfcdc49306c1ee1, 0xcd15aa57154a3f4d,
    0x288be80deeb8db09, 0x09629e13cb9efaa5, 0x6b590431a4f49851, 0x4ab0722f81d2b9fd,
    0x94bca6d70b86c3b2, 0xb555d0c92ea0e21e, 0xd76e4aeb41ca80ea, 0xf6873cf564eca146,
    0x13197eaf9f1e4502, 0x32f008b1ba3864ae, 0x50cb9293d552065a, 0x7122e48df07427f6,
    0xd80b1d31996d61af, 0xf9e26b2fbc4b4003, 0x9bd9f10dd32122f7, 0xba308713f607035b,
    0x5faec5490df5e71f, 0x7e47b35728d3c6b3, 0x1c7c297547b9a447, 0x3d955f6b629f85eb,
    0xe3998b93e8cbffa4, 0xc270fd8dcdedde08, 0xa04b67afa287bcfc, 0x81a211b187a19d50,
    0x643c53eb7c537914, 0x45d525f5597558b8, 0x27eebfd7361f3a4c, 0x0607c9c913391be0,
    0x41646afcbcba2595, 0x608d1ce2999c0439, 0x02b686c0f6f666cd, 0x235ff0ded3d04761,
    0xc6c1b2842822a325, 0xe728c49a0d048289, 0x85135eb8626ee07d, 0xa4fa28a64748c1d1,
    0x7af6fc5ecd1cbb9e, 0x5b1f8a40e83a9a32, 0x392410628750f8c6, 0x18cd667ca276d96a,
    0xfd53242659843d2e, 0xdcba52387ca21c82, 0xbe81c81a13c87e76, 0x9f68be0436ee5fda,
    0x364147b85ff71983, 0x17a831a67ad1382f, 0x7593ab8415bb5adb, 0x547add9a309d7b77,
    0xb1e49fc0cb6f9f33, 0x900de9deee49be9f, 0xf23673fc8123dc6b, 0xd3df05e2a405fdc7,
    0x0dd3d11a2e518788, 0x2c3aa7040b77a624, 0x4e013d26641dc4d0, 0x6fe84b38413be57c,
    0x8a760962bac90138, 0xab9f7f7c9fef2094, 0xc9a4e55ef0854260, 0xe84d9340d5a363cc,
    0x4763a335af833e8a, 0x668ad52b8aa51f26, 0x04b14f09e5cf7dd2, 0x25583917c0e95c7e,
    0xc0c67b4d3b1bb83a, 0xe12f0d531e3d9996, 0x831497717157fb62, 0xa2fde16f5471dace,
    0x7cf13597de25a081, 0x5d184389fb03812d, 0x3f23d9ab9469e3d9, 0x1ecaafb5b14fc275,
    0xfb54edef4abd2631, 0xdabd9bf16f9b079d, 0xb88601d300f16569, 0x996f77cd25d744c5,
    0x30468e714cce029c, 0x11aff86f69e82330, 0x7394624d068241c4, 0x527d145323a46068,
    0xb7e35609d856842c, 0x960a2017fd70a580, 0xf431ba35921ac774, 0xd5d8cc2bb73ce6d8,
    0x0bd418d33d689c97, 0x2a3d6ecd184ebd3b, 0x4806f4ef7724dfcf, 0x69ef82f15202fe63,
    0x8c71c0aba9f01a27, 0xad98b6b58cd63b8b, 0xcfa32c97e3bc597f, 0xee4a5a89c69a78d3,
    0xa929f9bc691946a6, 0x88c08fa24c3f670a, 0xeafb1580235505fe, 0xcb12639e06732452,
    0x2e8c21c4fd81c016, 0x0f6557dad8a7e1ba, 0x6d5ecdf8b7cd834e, 0x4cb7bbe692eba2e2,
    0x92bb6f1e18bfd8ad, 0xb35219003d99f901, 0xd169832252f39bf5, 0xf080f53c77d5ba59,
    0x151eb7668c275e1d, 0x34f7c178a9017fb1, 0x56cc5b5ac66b1d45, 0x77252d44e34d3ce9,
    0xde0cd4f88a547ab0, 0xffe5a2e6af725b1c, 0x9dde38c4c01839e8, 0xbc374edae53e1844,
    0x59a90c801eccfc00, 0x78407a9e3beaddac, 0x1a7be0bc5480bf58, 0x3b9296a271a69ef4,
    0xe59e425afbf2e4bb, 0xc4773444ded4c517, 0xa64cae66b1bea7e3, 0x87a5d8789498864f,
    0x623b9a226f6a620b, 0x43d2ec3c4a4c43a7, 0x21e9761e25262153, 0x00000000000000ff}
};

uint64_t crc64nvme_byte(uint64_t crc, void const *mem, size_t len) {
    uint8_t const *data = mem;
    if (data == NULL)
        return 0;
    for (size_t i = 0; i < len; i++) {
        crc = (crc >> 8) ^
              table_byte[(crc ^ data[i]) & 0xff];
    }
    return crc;
}

// This code assumes that integers are stored little-endian.

uint64_t crc64nvme_word(uint64_t crc, void const *mem, size_t len) {
    uint8_t const *data = mem;
    if (data == NULL)
        return 0;
    while (len && ((ptrdiff_t)data & 0x7)) {
        len--;
        crc = (crc >> 8) ^
              table_byte[(crc ^ *data++) & 0xff];
    }
    size_t n = len >> 3;
    for (size_t i = 0; i < n; i++) {
        uint64_t word = crc ^ ((uint64_t const *)data)[i];
        crc = table_word[7][word & 0xff] ^
              table_word[6][(word >> 8) & 0xff] ^
              table_word[5][(word >> 16) & 0xff] ^
              table_word[4][(word >> 24) & 0xff] ^
              table_word[3][(word >> 32) & 0xff] ^
              table_word[2][(word >> 40) & 0xff] ^
              table_word[1][(word >> 48) & 0xff] ^
              table_word[0][word >> 56];
    }
    data += n << 3;
    len &= 7;
    while (len) {
        len--;
        crc = (crc >> 8) ^
              table_byte[(crc ^ *data++) & 0xff];
    }
    return crc;
}

static uint64_t multmodp(uint64_t a, uint64_t b) {
    uint64_t prod = 0;
    for (;;) {
        if (a & 0x8000000000000000) {
            prod ^= b;
            if ((a & 0x7fffffffffffffff) == 0)
                break;
        }
        a <<= 1;
        b = b & 1 ? (b >> 1) ^ 0x9a6c9329ac4bc9b5 : b >> 1;
    }
    return prod;
}

static uint64_t const table_comb[] = {
    0x4000000000000000, 0x2000000000000000, 0x0800000000000000, 0x0080000000000000,
    0x0000800000000000, 0x0000000080000000, 0x9a6c9329ac4bc9b5, 0x10f4bb0f129310d6,
    0x70f05dcea2ebd226, 0x311211205672822d, 0x2fc297db0f46c96e, 0xca4d536fabf7da84,
    0xfb4cdc3b379ee6ed, 0xea261148df25140a, 0x59ccb2c07aa6c9b4, 0x20b3674a839af27a,
    0x2d8e1986da94d583, 0x42cdf4c20337635d, 0x1d78724bf0f26839, 0xb96c84e0afb34bd5,
    0x5d2e1fcd2df0a3ea, 0xcd9506572332be42, 0x23bda2427f7d690f, 0x347a953232374f07,
    0x1c2a807ac2a8ceea, 0x9b92ad0e14fe1460, 0x2574114889f670b2, 0x4a84a6c45e3bf520,
    0x915bbac21cd1c7ff, 0xb0290ec579f291f5, 0xcf2548505c624e6e, 0xb154f27bf08a8207,
    0xce4e92344baf7d35, 0x51da8d7e057c5eb3, 0x9fb10823f5be15df, 0x73b825b3ff1f71cf,
    0x5db436c5406ebb74, 0xfa7ed8f3ec3f2bca, 0xc4d58efdc61b9ef6, 0xa7e39e61e855bd45,
    0x97ad46f9dd1bf2f1, 0x1a0abb01f853ee6b, 0x3f0827c3348f8215, 0x4eb68c4506134607,
    0x4a46f6de5df34e0a, 0x2d855d6a1c57a8dd, 0x8688da58e1115812, 0x5232f417fc7c7300,
    0xa4080fb2e767d8da, 0xd515a7e17693e562, 0x1181f7c862e94226, 0x9e23cd058204ca91,
    0x9b8992c57a0aed82, 0xb2c0afb84609b6ff, 0x2f7160553a5ea018, 0x3cd378b5c99f2722,
    0x814054ad61a3b058, 0xbf766189fce806d8, 0x85a5e898ac49f86f, 0x34830d11bc84f346,
    0x9644d95b173c8c1c, 0x150401ac9ac759b1, 0xebe1f7f46fb00eba, 0x8ee4ce0c2e2bd662
};

static uint64_t x8nmodp(uintmax_t n) {
    uint64_t xp = 0x8000000000000000;
    int k = 3;
    for (;;) {
        if (n & 1)
            xp = multmodp(table_comb[k], xp);
        n >>= 1;
        if (n == 0)
            break;
        if (++k == 64)
            k = 0;
    }
    return xp;
}

uint64_t crc64nvme_comb(uint64_t crc1, uint64_t crc2,
        uintmax_t len2) {
    return multmodp(x8nmodp(len2), crc1) ^ crc2;
}
