/**
 * Author......: See docs/credits.txt
 * License.....: MIT
 */

#include "common.h"
#include "types.h"
#include "modules.h"
#include "bitops.h"
#include "convert.h"
#include "shared.h"
#include "memory.h"

static const u32   ATTACK_EXEC    = ATTACK_EXEC_OUTSIDE_KERNEL;
static const u32   DGST_POS0      = 0;
static const u32   DGST_POS1      = 1;
static const u32   DGST_POS2      = 2;
static const u32   DGST_POS3      = 3;
static const u32   DGST_SIZE      = DGST_SIZE_4_16;
static const u32   HASH_CATEGORY  = HASH_CATEGORY_FDE;
static const char *HASH_NAME      = "LUKS v1 SHA-1 + Serpent";
static const u64   KERN_TYPE      = 14612; // old kernel used here
static const u32   OPTI_TYPE      = OPTI_TYPE_ZERO_BYTE
                                  | OPTI_TYPE_SLOW_HASH_SIMD_LOOP;
static const u64   OPTS_TYPE      = OPTS_TYPE_STOCK_MODULE
                                  | OPTS_TYPE_PT_GENERATE_LE;
static const u32   SALT_TYPE      = SALT_TYPE_EMBEDDED;
static const char *ST_PASS        = "hashcat";
static const char *ST_HASH        = "$luks$1$sha1$serpent$cbc-plain64$256$152380$3530e7a181e2ef9ccb053e85380148087313218cac9cca053a772a8f583a23b0$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$67bd07c62be978781f70c96255a66dab83952ceda291dc56672db4dbbde87749f0b936b7a072bafc190a9052cab558013647db5d84aada532880f0df03c0ced09984af94c355e03dfafbeb4a21746ac98e7ce2253fd3718fb5b3983882688b53ccfcd329455ccd58696a206e23601ae6165b80d3a15a3bb92febfbd6441ce202ad421de324a2df1d47b7816526e93e546b43a0049efd7cf33da6adbb1aaa8c845fdb01f8feb0da92fe775cac6159610ca8f0af185decc137ffc38a6aadf2d36e3d96a060920cd0260b43a4dfea8a1557c0758fa6527ccc6f622092789e87652db861da0c2909da1b8f5ecbf712125ee43bc30850f0e25926ee00e9e64c765c61d151250ba12f7bf7544301c2a8ca548d2ff7f6f5a15b6330b1e6b22eeaffeeb2cc344d8fa585360b16c56ce4cf655e5b86ee207671b67d51397e9d6621d77dcac560d9b770e318f8b06ae09fec5d0bc3a0cfda80e4e7b7b3ab5fe8a69da8fe827fc6dff4f5d72ad811777f692d535af3af66458165950980ec1c639566d1564f5323c59222278e1677ffe6831397842e73506fd243f2b63c2c80f478afd9c781ca56b017a557659011d97c0b7cff991f93a0842a5bbb5f595058a231ecf48ed704bf4981f4603fb8b1d21e93330349860485a835b61c242802c3ea8c8b44f5eb592c29304c1dfece9f7366eaa07aef8a054f960a16e7b1e45f8a57f62db78952";

u32         module_attack_exec    (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ATTACK_EXEC;     }
u32         module_dgst_pos0      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS0;       }
u32         module_dgst_pos1      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS1;       }
u32         module_dgst_pos2      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS2;       }
u32         module_dgst_pos3      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS3;       }
u32         module_dgst_size      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_SIZE;       }
u32         module_hash_category  (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return HASH_CATEGORY;   }
const char *module_hash_name      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return HASH_NAME;       }
u64         module_kern_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return KERN_TYPE;       }
u32         module_opti_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return OPTI_TYPE;       }
u64         module_opts_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return OPTS_TYPE;       }
u32         module_salt_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return SALT_TYPE;       }
const char *module_st_hash        (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ST_HASH;         }
const char *module_st_pass        (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ST_PASS;         }

const char *module_usage_notice (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  return USAGE_NOTICE_LUKS;
}

#define LUKS_STRIPES        (                                   4000)
#define LUKS_SALT_LEN       (                                     32)
#define LUKS_SALT_HEX_LEN   (                      LUKS_SALT_LEN * 2)
#define LUKS_CT_LEN         (                                    512)
#define LUKS_CT_HEX_LEN     (                        LUKS_CT_LEN * 2)
#define LUKS_AF_MIN_LEN     (HC_LUKS_KEY_SIZE_128 / 8 * LUKS_STRIPES)
#define LUKS_AF_MIN_HEX_LEN (                    LUKS_AF_MIN_LEN * 2)
#define LUKS_AF_MAX_LEN     (HC_LUKS_KEY_SIZE_512 / 8 * LUKS_STRIPES)
#define LUKS_AF_MAX_HEX_LEN (                    LUKS_AF_MAX_LEN * 2)

typedef enum hc_luks_hash_type
{
  HC_LUKS_HASH_TYPE_SHA1      = 1,
  HC_LUKS_HASH_TYPE_SHA256    = 2,
  HC_LUKS_HASH_TYPE_SHA512    = 3,
  HC_LUKS_HASH_TYPE_RIPEMD160 = 4,
  HC_LUKS_HASH_TYPE_WHIRLPOOL = 5,

} hc_luks_hash_type_t;

typedef enum hc_luks_key_size
{
  HC_LUKS_KEY_SIZE_128 = 128,
  HC_LUKS_KEY_SIZE_256 = 256,
  HC_LUKS_KEY_SIZE_512 = 512,

} hc_luks_key_size_t;

typedef enum hc_luks_cipher_type
{
  HC_LUKS_CIPHER_TYPE_AES     = 1,
  HC_LUKS_CIPHER_TYPE_SERPENT = 2,
  HC_LUKS_CIPHER_TYPE_TWOFISH = 3,

} hc_luks_cipher_type_t;

typedef enum hc_luks_cipher_mode
{
  HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256 = 1,
  HC_LUKS_CIPHER_MODE_CBC_PLAIN        = 2,
  HC_LUKS_CIPHER_MODE_CBC_PLAIN64      = 3,
  HC_LUKS_CIPHER_MODE_XTS_PLAIN        = 4,
  HC_LUKS_CIPHER_MODE_XTS_PLAIN64      = 5,

} hc_luks_cipher_mode_t;

typedef struct luks
{
  int hash_type;   // hc_luks_hash_type_t
  int key_size;    // hc_luks_key_size_t
  int cipher_type; // hc_luks_cipher_type_t
  int cipher_mode; // hc_luks_cipher_mode_t

  u32 ct_buf[LUKS_CT_LEN / 4];

  u32 af_buf[LUKS_AF_MAX_LEN / 4];
  u32 af_len;

} luks_t;

typedef struct luks_tmp
{
  u32 ipad32[8];
  u64 ipad64[8];

  u32 opad32[8];
  u64 opad64[8];

  u32 dgst32[32];
  u64 dgst64[16];

  u32 out32[32];
  u64 out64[16];

} luks_tmp_t;

static const char *SIGNATURE_LUKS = "$luks$1$sha1$serpent$";

bool module_unstable_warning (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra, MAYBE_UNUSED const hc_device_param_t *device_param)
{
  if ((device_param->opencl_platform_vendor_id == VENDOR_ID_APPLE) && (device_param->opencl_device_type & CL_DEVICE_TYPE_GPU))
  {
    if (device_param->is_metal == true)
    {
      if (strncmp (device_param->device_name, "Intel", 5) == 0)
      {
        // Intel Iris Graphics, Metal Version 244.303: failed to create 'm14612_init' pipeline, timeout reached
        return true;
      }
    }
  }

  return false;
}

u64 module_esalt_size (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  const u64 esalt_size = (const u64) sizeof (luks_t);

  return esalt_size;
}

u64 module_tmp_size (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  const u64 tmp_size = (const u64) sizeof (luks_tmp_t);

  return tmp_size;
}

int module_hash_decode (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED void *digest_buf, MAYBE_UNUSED salt_t *salt, MAYBE_UNUSED void *esalt_buf, MAYBE_UNUSED void *hook_salt_buf, MAYBE_UNUSED hashinfo_t *hash_info, const char *line_buf, MAYBE_UNUSED const int line_len)
{
  luks_t *luks = (luks_t *) esalt_buf;

  hc_token_t token;

  memset (&token, 0, sizeof (hc_token_t));

  token.token_cnt  = 7;

  token.signatures_cnt    = 1;
  token.signatures_buf[0] = SIGNATURE_LUKS;

  // signature with hash and cipher type
  token.len[0]     = 21;
  token.attr[0]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_SIGNATURE;

  // cipher mode
  token.sep[1]     = '$';
  token.len_min[1] = 9;
  token.len_max[1] = 16;
  token.attr[1]    = TOKEN_ATTR_VERIFY_LENGTH;

  // key size
  token.sep[2]     = '$';
  token.len[2]     = 3;
  token.attr[2]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_DIGIT;

  // iter
  token.sep[3]     = '$';
  token.len_min[3] = 1;
  token.len_max[3] = 10;
  token.attr[3]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_DIGIT;

  // salt
  token.sep[4]     = '$';
  token.len[4]     = LUKS_SALT_HEX_LEN;
  token.attr[4]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  // af
  token.sep[5]     = '$';
  token.len_min[5] = LUKS_AF_MIN_HEX_LEN;
  token.len_max[5] = LUKS_AF_MAX_HEX_LEN;
  token.attr[5]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  // ct
  token.sep[6]     = '$';
  token.len[6]     = LUKS_CT_HEX_LEN;
  token.attr[6]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  const int rc_tokenizer = input_tokenizer ((const u8 *) line_buf, line_len, &token);

  if (rc_tokenizer != PARSER_OK) return (rc_tokenizer);

  // hash type

  luks->hash_type = HC_LUKS_HASH_TYPE_SHA1;

  // cipher type

  luks->cipher_type = HC_LUKS_CIPHER_TYPE_SERPENT;

  // cipher mode

  const u8 *cipher_mode_pos = token.buf[1];
  const u32 cipher_mode_len = token.len[1];

  if ((strncmp ((const char *) cipher_mode_pos, "cbc-essiv:sha256", 16) == 0) && (cipher_mode_len == 16))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "cbc-plain",    9) == 0) && (cipher_mode_len ==  9))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_PLAIN;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "cbc-plain64", 11) == 0) && (cipher_mode_len == 11))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_PLAIN64;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "xts-plain",    9) == 0) && (cipher_mode_len ==  9))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_XTS_PLAIN;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "xts-plain64", 11) == 0) && (cipher_mode_len == 11))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_XTS_PLAIN64;
  }
  else
  {
    return (PARSER_LUKS_CIPHER_MODE);
  }

  // key size

  const u8 *key_size_pos = token.buf[2];

  const u32 key_size = hc_strtoul ((const char *) key_size_pos, NULL, 10);

  if (key_size == 128)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_128;
  }
  else if (key_size == 256)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_256;
  }
  else if (key_size == 512)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_512;
  }
  else
  {
    return (PARSER_LUKS_KEY_SIZE);
  }

  // salt

  const u8 *salt_pos = token.buf[4];

  salt->salt_len = hex_decode (salt_pos, LUKS_SALT_HEX_LEN, (u8 *) salt->salt_buf);

  // iter

  const u8 *iter_pos = token.buf[3];

  const u32 iter = hc_strtoul ((const char *) iter_pos, NULL, 10);

  salt->salt_iter = iter - 1;

  // af

  u32 af_len = token.len[5] / 2;

  if (af_len != (key_size / 8) * LUKS_STRIPES) return (PARSER_LUKS_FILE_SIZE);

  const u8 *af_pos = token.buf[5];

  luks->af_len = hex_decode (af_pos, token.len[5], (u8 *) luks->af_buf);

  // ct

  u32 ct_len = token.len[6] / 2;

  if (ct_len != LUKS_CT_LEN) return (PARSER_LUKS_FILE_SIZE);

  const u8 *ct_pos = token.buf[6];

  hex_decode (ct_pos, LUKS_CT_HEX_LEN, (u8 *) luks->ct_buf);

  return (PARSER_OK);
}

int module_hash_encode (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const void *digest_buf, MAYBE_UNUSED const salt_t *salt, MAYBE_UNUSED const void *esalt_buf, MAYBE_UNUSED const void *hook_salt_buf, MAYBE_UNUSED const hashinfo_t *hash_info, char *line_buf, MAYBE_UNUSED const int line_size)
{
  const luks_t *luks = (const luks_t *) esalt_buf;

  // cipher mode

  const char *cipher_mode = "";

  switch (luks->cipher_mode)
  {
    case HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256: cipher_mode = "cbc-essiv:sha256"; break;
    case HC_LUKS_CIPHER_MODE_CBC_PLAIN:        cipher_mode = "cbc-plain";        break;
    case HC_LUKS_CIPHER_MODE_CBC_PLAIN64:      cipher_mode = "cbc-plain64";      break;
    case HC_LUKS_CIPHER_MODE_XTS_PLAIN:        cipher_mode = "xts-plain";        break;
    case HC_LUKS_CIPHER_MODE_XTS_PLAIN64:      cipher_mode = "xts-plain64";      break;
    default:                                                                     return 0;
  }

  // key size

  u32 key_size = 0;

  switch (luks->key_size)
  {
    case HC_LUKS_KEY_SIZE_128: key_size = 128; break;
    case HC_LUKS_KEY_SIZE_256: key_size = 256; break;
    case HC_LUKS_KEY_SIZE_512: key_size = 512; break;
    default:                                   return 0;
  }

  // iterations

  u32 iterations = salt->salt_iter + 1;

  // salt

  char *salt_buf = (char *) hcmalloc (LUKS_SALT_HEX_LEN + 1);

  hex_encode ((const u8 *) salt->salt_buf, LUKS_SALT_LEN, (u8 *) salt_buf);

  // af

  char *af_buf = (char *) hcmalloc (LUKS_AF_MAX_HEX_LEN + 1);

  hex_encode ((const u8 *) luks->af_buf, luks->af_len, (u8 *) af_buf);

  // ct

  char *ct_buf = (char *) hcmalloc (LUKS_CT_HEX_LEN + 1);

  hex_encode ((const u8 *) luks->ct_buf, LUKS_CT_LEN, (u8 *) ct_buf);

  // output

  int line_len = snprintf (line_buf, line_size, "%s%s$%u$%u$%s$%s$%s",
    SIGNATURE_LUKS,
    cipher_mode,
    key_size,
    iterations,
    salt_buf,
    af_buf,
    ct_buf);

  hcfree (salt_buf);
  hcfree (af_buf);
  hcfree (ct_buf);

  return line_len;
}

void module_init (module_ctx_t *module_ctx)
{
  module_ctx->module_context_size             = MODULE_CONTEXT_SIZE_CURRENT;
  module_ctx->module_interface_version        = MODULE_INTERFACE_VERSION_CURRENT;

  module_ctx->module_attack_exec              = module_attack_exec;
  module_ctx->module_benchmark_esalt          = MODULE_DEFAULT;
  module_ctx->module_benchmark_hook_salt      = MODULE_DEFAULT;
  module_ctx->module_benchmark_mask           = MODULE_DEFAULT;
  module_ctx->module_benchmark_charset        = MODULE_DEFAULT;
  module_ctx->module_benchmark_salt           = MODULE_DEFAULT;
  module_ctx->module_bridge_name              = MODULE_DEFAULT;
  module_ctx->module_bridge_type              = MODULE_DEFAULT;
  module_ctx->module_build_plain_postprocess  = MODULE_DEFAULT;
  module_ctx->module_deep_comp_kernel         = MODULE_DEFAULT;
  module_ctx->module_deprecated_notice        = MODULE_DEFAULT;
  module_ctx->module_usage_notice             = module_usage_notice;
  module_ctx->module_dgst_pos0                = module_dgst_pos0;
  module_ctx->module_dgst_pos1                = module_dgst_pos1;
  module_ctx->module_dgst_pos2                = module_dgst_pos2;
  module_ctx->module_dgst_pos3                = module_dgst_pos3;
  module_ctx->module_dgst_size                = module_dgst_size;
  module_ctx->module_dictstat_disable         = MODULE_DEFAULT;
  module_ctx->module_esalt_size               = module_esalt_size;
  module_ctx->module_extra_buffer_size        = MODULE_DEFAULT;
  module_ctx->module_extra_tmp_size           = MODULE_DEFAULT;
  module_ctx->module_extra_tuningdb_block     = MODULE_DEFAULT;
  module_ctx->module_forced_outfile_format    = MODULE_DEFAULT;
  module_ctx->module_hash_binary_count        = MODULE_DEFAULT;
  module_ctx->module_hash_binary_parse        = MODULE_DEFAULT;
  module_ctx->module_hash_binary_save         = MODULE_DEFAULT;
  module_ctx->module_hash_decode_postprocess  = MODULE_DEFAULT;
  module_ctx->module_hash_decode_potfile      = MODULE_DEFAULT;
  module_ctx->module_hash_decode_zero_hash    = MODULE_DEFAULT;
  module_ctx->module_hash_decode              = module_hash_decode;
  module_ctx->module_hash_encode_status       = MODULE_DEFAULT;
  module_ctx->module_hash_encode_potfile      = MODULE_DEFAULT;
  module_ctx->module_hash_encode              = module_hash_encode;
  module_ctx->module_hash_init_selftest       = MODULE_DEFAULT;
  module_ctx->module_hash_mode                = MODULE_DEFAULT;
  module_ctx->module_hash_category            = module_hash_category;
  module_ctx->module_hash_name                = module_hash_name;
  module_ctx->module_hashes_count_min         = MODULE_DEFAULT;
  module_ctx->module_hashes_count_max         = MODULE_DEFAULT;
  module_ctx->module_hlfmt_disable            = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_size    = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_init    = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_term    = MODULE_DEFAULT;
  module_ctx->module_hook12                   = MODULE_DEFAULT;
  module_ctx->module_hook23                   = MODULE_DEFAULT;
  module_ctx->module_hook_salt_size           = MODULE_DEFAULT;
  module_ctx->module_hook_size                = MODULE_DEFAULT;
  module_ctx->module_jit_build_options        = MODULE_DEFAULT;
  module_ctx->module_jit_cache_disable        = MODULE_DEFAULT;
  module_ctx->module_kernel_accel_max         = MODULE_DEFAULT;
  module_ctx->module_kernel_accel_min         = MODULE_DEFAULT;
  module_ctx->module_kernel_loops_max         = MODULE_DEFAULT;
  module_ctx->module_kernel_loops_min         = MODULE_DEFAULT;
  module_ctx->module_kernel_threads_max       = MODULE_DEFAULT;
  module_ctx->module_kernel_threads_min       = MODULE_DEFAULT;
  module_ctx->module_kern_type                = module_kern_type;
  module_ctx->module_kern_type_dynamic        = MODULE_DEFAULT;
  module_ctx->module_opti_type                = module_opti_type;
  module_ctx->module_opts_type                = module_opts_type;
  module_ctx->module_outfile_check_disable    = MODULE_DEFAULT;
  module_ctx->module_outfile_check_nocomp     = MODULE_DEFAULT;
  module_ctx->module_potfile_custom_check     = MODULE_DEFAULT;
  module_ctx->module_potfile_disable          = MODULE_DEFAULT;
  module_ctx->module_potfile_keep_all_hashes  = MODULE_DEFAULT;
  module_ctx->module_pwdump_column            = MODULE_DEFAULT;
  module_ctx->module_pw_max                   = MODULE_DEFAULT;
  module_ctx->module_pw_min                   = MODULE_DEFAULT;
  module_ctx->module_salt_max                 = MODULE_DEFAULT;
  module_ctx->module_salt_min                 = MODULE_DEFAULT;
  module_ctx->module_salt_type                = module_salt_type;
  module_ctx->module_separator                = MODULE_DEFAULT;
  module_ctx->module_st_hash                  = module_st_hash;
  module_ctx->module_st_pass                  = module_st_pass;
  module_ctx->module_tmp_size                 = module_tmp_size;
  module_ctx->module_unstable_warning         = module_unstable_warning;
  module_ctx->module_warmup_disable           = MODULE_DEFAULT;
}
