/*
 * Decompiled with CFR 0.152.
 */
package ghidra.formats.gfilesystem;

import ghidra.formats.gfilesystem.FSRL;
import ghidra.formats.gfilesystem.FSRLRoot;
import ghidra.formats.gfilesystem.FSUtilities;
import ghidra.formats.gfilesystem.GFile;
import ghidra.formats.gfilesystem.GFileImpl;
import ghidra.formats.gfilesystem.GFileSystem;
import ghidra.util.Msg;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

public class FileSystemIndexHelper<METADATATYPE> {
    private static final int MAX_SYMLINK_RECURSE_DEPTH = 10;
    private FileData<METADATATYPE> rootDir;
    protected Map<GFile, FileData<METADATATYPE>> fileToEntryMap = new HashMap<GFile, FileData<METADATATYPE>>();
    protected Map<Long, FileData<METADATATYPE>> fileIndexToEntryMap = new HashMap<Long, FileData<METADATATYPE>>();
    protected Map<GFile, Map<String, FileData<METADATATYPE>>> directoryToListing = new HashMap<GFile, Map<String, FileData<METADATATYPE>>>();

    public FileSystemIndexHelper(GFileSystem fs, FSRLRoot fsFSRL) {
        GFileImpl rootGFile = GFileImpl.fromFSRL(fs, null, fsFSRL.withPath("/"), true, -1L);
        this.rootDir = new FileData<Object>(rootGFile, null, -1L);
        this.fileToEntryMap.put(this.rootDir.file, this.rootDir);
        this.directoryToListing.put(this.rootDir.file, new HashMap());
    }

    public GFile getRootDir() {
        return this.rootDir.file;
    }

    public synchronized void clear() {
        this.fileToEntryMap.clear();
        this.directoryToListing.clear();
        this.fileIndexToEntryMap.clear();
    }

    public synchronized int getFileCount() {
        return this.fileToEntryMap.size();
    }

    public synchronized METADATATYPE getMetadata(GFile f) {
        FileData<METADATATYPE> fileData = this.fileToEntryMap.get(f);
        return fileData != null ? (METADATATYPE)fileData.metaData : null;
    }

    public synchronized void setMetadata(GFile f, METADATATYPE metaData) throws IOException {
        FileData<METADATATYPE> fileData = this.getFileData(f);
        fileData.metaData = metaData;
    }

    public synchronized GFile getFileByIndex(long fileIndex) {
        FileData<METADATATYPE> fileData = this.fileIndexToEntryMap.get(fileIndex);
        return fileData != null ? fileData.file : null;
    }

    public synchronized List<GFile> getListing(GFile directory) {
        Map<String, FileData<METADATATYPE>> dirListing = this.getDirectoryContents(directory, false);
        if (dirListing == null) {
            return List.of();
        }
        return dirListing.values().stream().map(fd -> fd.file).toList();
    }

    public synchronized GFile lookup(String path) {
        return this.lookup(null, path, null);
    }

    public synchronized GFile lookup(GFile baseDir, String path, Comparator<String> nameComp) {
        try {
            FileData<METADATATYPE> baseDirData = this.getFileData(baseDir);
            FileData<METADATATYPE> fileData = this.lookup(baseDirData, FSUtilities.splitPath(path), -1, false, nameComp);
            return fileData != null ? fileData.file : null;
        }
        catch (IOException iOException) {
            return null;
        }
    }

    protected FileData<METADATATYPE> lookup(FileData<METADATATYPE> baseDir, String[] nameparts, int maxpart, boolean createIfMissing, Comparator<String> nameComp) {
        maxpart = maxpart < 0 ? nameparts.length : maxpart;
        FileData<METADATATYPE> currentFile = Objects.requireNonNullElse(baseDir, this.rootDir);
        for (int i = 0; i < maxpart && currentFile != null; ++i) {
            String name = nameparts[i];
            if (name.isEmpty()) continue;
            Map<String, FileData<METADATATYPE>> currentDirContents = this.getDirectoryContents(currentFile.file, createIfMissing);
            FileData<METADATATYPE> next = this.lookupFileInDir(currentDirContents, name, nameComp);
            if (next == null && createIfMissing) {
                next = this.doStoreMissingDir(name, currentFile.file);
            }
            currentFile = next;
        }
        return currentFile;
    }

    protected FileData<METADATATYPE> resolveSymlinkPath(FileData<METADATATYPE> baseDir, String path, int depth, StringBuilder symlinkPathDebug, Comparator<String> nameComp) throws IOException {
        symlinkPathDebug = Objects.requireNonNullElseGet(symlinkPathDebug, StringBuilder::new);
        if (depth > 10) {
            throw new IOException("Too many symlinks: %s, %s".formatted(symlinkPathDebug, path));
        }
        symlinkPathDebug.append("[");
        FileData<METADATATYPE> currentFile = Objects.requireNonNullElse(baseDir, this.rootDir);
        String[] pathparts = FSUtilities.splitPath(path);
        for (int i = 0; i < pathparts.length && currentFile != null; ++i) {
            String name = pathparts[i];
            symlinkPathDebug.append(i != 0 ? "," : "").append(name);
            if (i == 0 && name.isEmpty()) {
                currentFile = this.rootDir;
                continue;
            }
            if (name.isEmpty() || ".".equals(name)) continue;
            if ("..".equals(name)) {
                currentFile = this.getParentFileData(currentFile);
                continue;
            }
            Map<String, FileData<METADATATYPE>> currentDirContents = this.getDirectoryContents(currentFile.file, false);
            FileData<METADATATYPE> next = this.lookupFileInDir(currentDirContents, name, nameComp);
            if (next != null && next.symlinkPath != null) {
                next = this.resolveSymlinkPath(currentFile, next.symlinkPath, depth + 1, symlinkPathDebug, nameComp);
            }
            currentFile = next;
        }
        symlinkPathDebug.append("]");
        return currentFile;
    }

    public synchronized GFile resolveSymlinks(GFile file) throws IOException {
        FileData<METADATATYPE> fd = this.getFileData(file);
        if (fd.symlinkPath != null) {
            fd = this.resolveSymlinkPath(this.getParentFileData(fd), fd.symlinkPath, 0, null, null);
        }
        return fd != null ? fd.file : null;
    }

    public synchronized String getSymlinkPath(GFile file) {
        FileData<METADATATYPE> fd = file == null ? this.rootDir : this.fileToEntryMap.get(file);
        return fd != null ? fd.symlinkPath : null;
    }

    private FileData<METADATATYPE> getFileData(GFile f) throws IOException {
        if (f == null) {
            return this.rootDir;
        }
        FileData<METADATATYPE> fd = this.fileToEntryMap.get(f);
        if (fd == null) {
            throw new IOException("Unknown file: %s".formatted(f));
        }
        return fd;
    }

    private FileData<METADATATYPE> getParentFileData(FileData<METADATATYPE> file) {
        GFile parentGFile = file.file.getParentFile();
        return parentGFile != null ? this.fileToEntryMap.get(parentGFile) : null;
    }

    public synchronized GFile storeFile(String path, long fileIndex, boolean isDirectory, long length, METADATATYPE metadata) {
        String[] nameparts = FSUtilities.splitPath(path);
        if (nameparts.length == 0) {
            return this.rootDir.file;
        }
        GFile parent = this.lookupParent(nameparts, null);
        String lastpart = nameparts[nameparts.length - 1];
        FileData<METADATATYPE> fileData = this.doStoreFile(lastpart, parent, fileIndex, isDirectory, length, null, metadata);
        return fileData.file;
    }

    public synchronized GFile storeFileWithParent(String filename, GFile parent, long fileIndex, boolean isDirectory, long length, METADATATYPE metadata) {
        FileData<METADATATYPE> fileData = this.doStoreFile(filename, parent, fileIndex, isDirectory, length, null, metadata);
        return fileData.file;
    }

    public synchronized GFile storeSymlink(String path, long fileIndex, String symlinkPath, long length, METADATATYPE metadata) {
        String[] nameparts = FSUtilities.splitPath(path);
        if (nameparts.length == 0) {
            Msg.warn((Object)this, (Object)"Unable to create invalid symlink file [%s] -> [%s]".formatted(path, symlinkPath));
            return this.rootDir.file;
        }
        length = length != 0L ? length : (long)symlinkPath.length();
        GFile parent = this.lookupParent(nameparts, null);
        String lastpart = nameparts[nameparts.length - 1];
        FileData<METADATATYPE> fileData = this.doStoreFile(lastpart, parent, fileIndex, false, length, symlinkPath, metadata);
        return fileData.file;
    }

    public synchronized GFile storeSymlinkWithParent(String filename, GFile parent, long fileIndex, String symlinkPath, long length, METADATATYPE metadata) {
        length = length != 0L ? length : (long)symlinkPath.length();
        FileData<METADATATYPE> fileData = this.doStoreFile(filename, parent, fileIndex, false, length, symlinkPath, metadata);
        return fileData.file;
    }

    private FileData<METADATATYPE> doStoreMissingDir(String filename, GFile parent) {
        parent = parent == null ? this.rootDir.file : parent;
        Map<String, FileData<METADATATYPE>> dirContents = this.getDirectoryContents(parent, true);
        GFileImpl file = this.createNewFile(parent, filename, true, -1L, null);
        FileData<Object> fileData = new FileData<Object>(file, null, -1L);
        this.fileToEntryMap.put(file, fileData);
        dirContents.put(filename, fileData);
        this.getDirectoryContents(file, true);
        return fileData;
    }

    private FileData<METADATATYPE> doStoreFile(String filename, GFile parent, long fileIndex, boolean isDirectory, long length, String symlinkPath, METADATATYPE metadata) {
        Map<String, FileData<METADATATYPE>> dirContents;
        long fileNum;
        parent = parent == null ? this.rootDir.file : parent;
        long l = fileNum = fileIndex != -1L ? fileIndex : (long)this.fileToEntryMap.size();
        if (this.fileIndexToEntryMap.containsKey(fileNum)) {
            Msg.warn((Object)this, (Object)"Duplicate fileNum %d for file %s/%s".formatted(fileNum, parent.getPath(), filename));
        }
        String uniqueName = this.makeUniqueFilename((dirContents = this.getDirectoryContents(parent, true)).containsKey(filename) && !isDirectory, filename, fileNum);
        GFileImpl file = this.createNewFile(parent, uniqueName, isDirectory, length, metadata);
        FileData<METADATATYPE> fileData = new FileData<METADATATYPE>(file, metadata, fileNum, symlinkPath);
        this.fileToEntryMap.put(file, fileData);
        this.fileIndexToEntryMap.put(fileNum, fileData);
        dirContents.put(uniqueName, fileData);
        if (isDirectory) {
            this.getDirectoryContents(file, true);
        }
        return fileData;
    }

    private String makeUniqueFilename(boolean wasNameCollision, String filename, long fileIndex) {
        return wasNameCollision ? filename + "[%d]".formatted(fileIndex) : filename;
    }

    private Map<String, FileData<METADATATYPE>> getDirectoryContents(GFile directoryFile, boolean createIfMissing) {
        Map<String, FileData<METADATATYPE>> dirContents = this.directoryToListing.get(directoryFile = directoryFile != null ? directoryFile : this.rootDir.file);
        if (dirContents == null && createIfMissing) {
            dirContents = new HashMap<String, FileData<METADATATYPE>>();
            this.directoryToListing.put(directoryFile, dirContents);
        }
        return dirContents;
    }

    protected GFile lookupParent(String[] nameparts, Comparator<String> nameComp) {
        FileData<METADATATYPE> parent = this.lookup(this.rootDir, nameparts, nameparts.length - 1, true, nameComp);
        return parent.file;
    }

    protected FileData<METADATATYPE> lookupFileInDir(Map<String, FileData<METADATATYPE>> dirContents, String filename, Comparator<String> nameComp) {
        if (dirContents == null) {
            return null;
        }
        if (nameComp == null) {
            return dirContents.get(filename);
        }
        ArrayList<FileData<METADATATYPE>> candidateFiles = new ArrayList<FileData<METADATATYPE>>();
        for (FileData<METADATATYPE> fd : dirContents.values()) {
            if (nameComp.compare(filename, fd.file.getName()) != 0) continue;
            if (fd.file.getName().equals(filename)) {
                return fd;
            }
            candidateFiles.add(fd);
        }
        Collections.sort(candidateFiles, (f1, f2) -> f1.file.getName().compareTo(f2.file.getName()));
        return !candidateFiles.isEmpty() ? (FileData)candidateFiles.get(0) : null;
    }

    protected GFileImpl createNewFile(GFile parentFile, String name, boolean isDirectory, long size, METADATATYPE metadata) {
        FSRL newFileFSRL = parentFile.getFSRL().appendPath(name);
        return GFileImpl.fromFSRL(this.rootDir.file.getFilesystem(), parentFile, newFileFSRL, isDirectory, size);
    }

    public synchronized void updateFSRL(GFile file, FSRL newFSRL) {
        Map<String, FileData<METADATATYPE>> dirListing;
        GFileImpl newFile = GFileImpl.fromFSRL(this.rootDir.file.getFilesystem(), file.getParentFile(), newFSRL, file.isDirectory(), file.getLength());
        FileData<METADATATYPE> fileData = this.fileToEntryMap.get(file);
        if (fileData != null) {
            this.fileToEntryMap.remove(file);
            this.fileIndexToEntryMap.remove(fileData.fileIndex);
            fileData.file = newFile;
            this.fileToEntryMap.put(newFile, fileData);
            if (fileData.fileIndex != -1L) {
                this.fileIndexToEntryMap.put(fileData.fileIndex, fileData);
            }
        }
        if ((dirListing = this.directoryToListing.get(file)) != null) {
            this.directoryToListing.remove(file);
            this.directoryToListing.put(newFile, dirListing);
        }
    }

    public String toString() {
        return "FileSystemIndexHelper for " + String.valueOf(this.rootDir.file.getFilesystem());
    }

    static class FileData<METADATATYPE> {
        GFile file;
        METADATATYPE metaData;
        final long fileIndex;
        final String symlinkPath;

        FileData(GFile file, METADATATYPE metaData, long fileIndex) {
            this(file, metaData, fileIndex, null);
        }

        FileData(GFile file, METADATATYPE metaData, long fileIndex, String symlinkPath) {
            this.file = file;
            this.metaData = metaData;
            this.fileIndex = fileIndex;
            this.symlinkPath = symlinkPath;
        }
    }
}

