/*
 * Decompiled with CFR 0.152.
 */
package ghidra.program.model.data;

import ghidra.program.model.lang.Endian;
import ghidra.util.charset.CharsetInfoManager;
import ghidra.util.exception.UsrException;
import java.nio.BufferOverflowException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.CoderResult;
import java.nio.charset.MalformedInputException;
import java.nio.charset.UnmappableCharacterException;
import java.util.Collections;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

public class StringRenderParser {
    private static String hexDigits = "0123456789ABCDEFabcdef";
    private final char quoteChar;
    private final Endian endian;
    private final String charsetName;
    private final boolean includeBOM;
    private final CharBuffer oneCodePointBuffer = CharBuffer.allocate(2);
    private int pos;
    private State state;
    private Charset charset;
    private CharsetEncoder encoder;
    private int val;
    private int codeDigits;

    public StringRenderParser(char quoteChar, Endian endian, String charsetName, boolean includeBOM) {
        this.quoteChar = Objects.requireNonNull(Character.valueOf(quoteChar)).charValue();
        this.endian = Objects.requireNonNull(endian);
        this.charsetName = charsetName;
        this.includeBOM = includeBOM;
        this.reset();
    }

    public void reset() {
        this.pos = 0;
        this.state = State.INIT;
        this.val = 0;
        this.charset = null;
        this.encoder = null;
    }

    protected void initCharset(ByteBuffer out, String reprCharsetName) {
        Object charsetName = this.charsetName != null ? this.charsetName : reprCharsetName;
        int charSize = CharsetInfoManager.getInstance().getCharsetCharSize((String)charsetName);
        if (CharsetInfoManager.isBOMCharset((String)charsetName)) {
            charsetName = (String)charsetName + (this.endian.isBigEndian() ? "BE" : "LE");
        }
        this.charset = Charset.forName((String)charsetName);
        if (this.includeBOM) {
            if (charSize == 2) {
                out.putShort((short)-257);
            } else if (charSize == 4) {
                out.putInt(65279);
            }
        }
        this.encoder = this.charset.newEncoder();
    }

    protected int valHexDigit(char c) {
        if ('0' <= c && c <= '9') {
            return c - 48 + 0;
        }
        if ('A' <= c && c <= 'F') {
            return c - 65 + 10;
        }
        if ('a' <= c && c <= 'f') {
            return c - 97 + 10;
        }
        throw new AssertionError();
    }

    protected State parseCharInit(ByteBuffer out, char c) {
        if (c == 'u') {
            return State.PREFIX;
        }
        if (c == 'U') {
            this.initCharset(out, "UTF-32");
            return State.UNIT;
        }
        this.initCharset(out, "US-ASCII");
        return this.parseCharUnit(out, c);
    }

    protected State parseCharPrefix(ByteBuffer out, char c) {
        if (c == '8') {
            this.initCharset(out, "UTF-8");
            return State.UNIT;
        }
        this.initCharset(out, "UTF-16");
        return this.parseCharUnit(out, c);
    }

    protected State parseCharUnit(ByteBuffer out, char c) {
        if ('0' <= c && c <= '9' || 'A' <= c && c <= 'F' || 'a' <= c && c <= 'f') {
            this.val = this.valHexDigit(c);
            return State.BYTE;
        }
        if (c == this.quoteChar) {
            return State.STR;
        }
        throw new AssertionError();
    }

    protected void encodeCodePoint(ByteBuffer out, int cp) throws MalformedInputException, UnmappableCharacterException {
        if (!Character.isSupplementaryCodePoint(cp)) {
            this.encodeChar(out, (char)cp);
            return;
        }
        this.oneCodePointBuffer.clear();
        this.oneCodePointBuffer.put(0, Character.highSurrogate(cp));
        this.oneCodePointBuffer.put(1, Character.lowSurrogate(cp));
        this.encodeBufferedCodePoint(out);
    }

    protected void encodeChar(ByteBuffer out, char c) throws MalformedInputException, UnmappableCharacterException {
        this.oneCodePointBuffer.clear();
        this.oneCodePointBuffer.put(0, c);
        this.oneCodePointBuffer.limit(1);
        this.encodeBufferedCodePoint(out);
    }

    protected void encodeBufferedCodePoint(ByteBuffer out) throws MalformedInputException, UnmappableCharacterException {
        CoderResult cr = this.encoder.encode(this.oneCodePointBuffer, out, false);
        if (cr.isOverflow()) {
            throw new BufferOverflowException();
        }
        if (cr.isMalformed()) {
            throw new MalformedInputException(1);
        }
        if (cr.isUnmappable()) {
            throw new UnmappableCharacterException(1);
        }
    }

    protected State parseCharStr(ByteBuffer out, char c) throws MalformedInputException, UnmappableCharacterException {
        if (c == this.quoteChar) {
            return State.COMMA;
        }
        if (c == '\\') {
            return State.ESCAPE;
        }
        this.encodeChar(out, c);
        return State.STR;
    }

    protected State parseCharByte(ByteBuffer out, char c) {
        this.val = (this.val << 4) + this.valHexDigit(c);
        return State.BYTE_SUFFIX;
    }

    protected State parseCharByteSuffix(ByteBuffer out, char c) {
        if (c == 'h') {
            out.put((byte)this.val);
            this.val = 0;
            return State.COMMA;
        }
        throw new AssertionError();
    }

    protected State parseCharComma(ByteBuffer out, char c) {
        if (c == ',') {
            return State.UNIT;
        }
        throw new AssertionError();
    }

    protected State parseCharEscape(ByteBuffer out, char c) throws MalformedInputException, UnmappableCharacterException {
        switch (c) {
            case '0': {
                this.encodeChar(out, '\u0000');
                return State.STR;
            }
            case 'a': {
                this.encodeChar(out, '\u0007');
                return State.STR;
            }
            case 'b': {
                this.encodeChar(out, '\b');
                return State.STR;
            }
            case 't': {
                this.encodeChar(out, '\t');
                return State.STR;
            }
            case 'n': {
                this.encodeChar(out, '\n');
                return State.STR;
            }
            case 'v': {
                this.encodeChar(out, '\u000b');
                return State.STR;
            }
            case 'f': {
                this.encodeChar(out, '\f');
                return State.STR;
            }
            case 'r': {
                this.encodeChar(out, '\r');
                return State.STR;
            }
            case '\\': {
                this.encodeChar(out, '\\');
                return State.STR;
            }
            case '\"': {
                this.encodeChar(out, '\"');
                return State.STR;
            }
            case '\'': {
                this.encodeChar(out, '\'');
                return State.STR;
            }
            case 'x': {
                this.codeDigits = 2;
                return State.CODE_POINT;
            }
            case 'u': {
                this.codeDigits = 4;
                return State.CODE_POINT;
            }
            case 'U': {
                this.codeDigits = 8;
                return State.CODE_POINT;
            }
        }
        throw new AssertionError();
    }

    protected State parseCharCodePoint(ByteBuffer out, char c) throws MalformedInputException, UnmappableCharacterException {
        assert (this.codeDigits > 0);
        this.val = (this.val << 4) + this.valHexDigit(c);
        if (0 == --this.codeDigits) {
            this.encodeCodePoint(out, this.val);
            return State.STR;
        }
        return State.CODE_POINT;
    }

    protected State parseChar(ByteBuffer out, char c) throws MalformedInputException, UnmappableCharacterException {
        switch (this.state.ordinal()) {
            case 0: {
                return this.parseCharInit(out, c);
            }
            case 1: {
                return this.parseCharPrefix(out, c);
            }
            case 2: {
                return this.parseCharUnit(out, c);
            }
            case 3: {
                return this.parseCharStr(out, c);
            }
            case 4: {
                return this.parseCharByte(out, c);
            }
            case 5: {
                return this.parseCharByteSuffix(out, c);
            }
            case 6: {
                return this.parseCharComma(out, c);
            }
            case 7: {
                return this.parseCharEscape(out, c);
            }
            case 8: {
                return this.parseCharCodePoint(out, c);
            }
        }
        throw new AssertionError();
    }

    public ByteBuffer parse(CharBuffer in) throws StringParseException, MalformedInputException, UnmappableCharacterException {
        int bufSize = in.remaining() * 2;
        while (true) {
            ByteBuffer out = ByteBuffer.allocate(bufSize).order(this.endian == Endian.BIG ? ByteOrder.BIG_ENDIAN : ByteOrder.LITTLE_ENDIAN);
            int pos = in.position();
            try {
                this.parse(out, in);
                this.finish(out);
                out.flip();
                return out;
            }
            catch (BufferOverflowException e) {
                bufSize <<= 1;
                in.position(pos);
                this.reset();
                continue;
            }
            break;
        }
    }

    public void parse(ByteBuffer out, CharBuffer in) throws StringParseException, MalformedInputException, UnmappableCharacterException {
        try {
            while (in.hasRemaining()) {
                char c = in.get();
                this.state.checkAccepts(this.pos, c);
                this.state = this.parseChar(out, c);
                ++this.pos;
            }
        }
        catch (Throwable e) {
            in.position(in.position() - 1);
            throw e;
        }
    }

    public void finish(ByteBuffer out) throws StringParseException {
        this.state.checkFinal(this.pos);
    }

    private static enum State {
        INIT(true, "uU'\"" + hexDigits),
        PREFIX(false, "8'\"" + hexDigits),
        UNIT(true, "'\"" + hexDigits),
        STR(false),
        BYTE(false, hexDigits),
        BYTE_SUFFIX(false, "h"),
        COMMA(true, ","),
        ESCAPE(false, "0abtnvfr\\\"'xuU"),
        CODE_POINT(false, hexDigits);

        private final boolean isFinal;
        private final Set<Character> accepts;

        private State(boolean isFinal) {
            this.isFinal = isFinal;
            this.accepts = null;
        }

        private State(boolean isFinal, String accepts) {
            this(isFinal, accepts.chars().mapToObj(i -> Character.valueOf((char)i)).collect(Collectors.toSet()));
        }

        private State(boolean isFinal, Set<Character> accepts) {
            this.isFinal = isFinal;
            this.accepts = Collections.unmodifiableSet(accepts);
        }

        private void checkAccepts(int pos, char c) throws StringParseException {
            if (this.accepts == null) {
                return;
            }
            if (this.accepts.contains(Character.valueOf(c))) {
                return;
            }
            throw new StringParseException(pos, this.accepts, c);
        }

        private void checkFinal(int pos) throws StringParseException {
            if (this.isFinal) {
                return;
            }
            throw new StringParseException(pos);
        }
    }

    public static class StringParseException
    extends UsrException {
        public StringParseException(int pos, Set<Character> expected, char got) {
            super("Error parsing string representation at position " + pos + ". Expected one of " + String.valueOf(Objects.requireNonNull(expected)) + " but got " + got);
        }

        public StringParseException(int pos) {
            super("Unexpected end of string representation at position " + pos + ".");
        }
    }
}

