/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ghidra.pcode.emu.jit.gen.op;

import static ghidra.lifecycle.Unfinished.TODO;

import java.util.Comparator;

import ghidra.pcode.emu.jit.analysis.JitControlFlowModel.JitBlock;
import ghidra.pcode.emu.jit.analysis.JitType;
import ghidra.pcode.emu.jit.analysis.JitType.*;
import ghidra.pcode.emu.jit.gen.JitCodeGenerator;
import ghidra.pcode.emu.jit.gen.opnd.Opnd.Ext;
import ghidra.pcode.emu.jit.gen.tgt.JitCompiledPassage;
import ghidra.pcode.emu.jit.gen.tgt.JitCompiledPassage.EntryPoint;
import ghidra.pcode.emu.jit.gen.util.*;
import ghidra.pcode.emu.jit.gen.util.Emitter.*;
import ghidra.pcode.emu.jit.gen.util.Lbl.LblEm;
import ghidra.pcode.emu.jit.gen.util.Methods.RetReq;
import ghidra.pcode.emu.jit.gen.util.Types.*;
import ghidra.pcode.emu.jit.op.JitFloatTestOp;

/**
 * An extension for float comparison operators
 * 
 * @param <T> the class of p-code op node in the use-def graph
 */
public interface FloatCompareBinOpGen<T extends JitFloatTestOp> extends BinOpGen<T> {

	@Override
	default boolean isSigned() {
		return false;
	}

	/**
	 * Emit the JVM bytecode to perform the comparison with float operands on the stack.
	 * <p>
	 * The result should be as defined by {@link Comparator#compare(Object, Object)}.
	 * 
	 * @param <N2> the tail of the incoming stack
	 * @param <N1> the tail of the incoming stack including the right operand
	 * @param <N0> the incoming stack with the right and left operands on top
	 * @param em the emitter typed with the incoming stack
	 * @return the emitter typed with the resulting stack, i.e., the tail with the result pushed
	 */
	<N2 extends Next, N1 extends Ent<N2, TFloat>, N0 extends Ent<N1, TFloat>>
			Emitter<Ent<N2, TInt>> opForFloatCmp(Emitter<N0> em);

	/**
	 * Emit the JVM bytecode to perform the comparison with double operands on the stack.
	 * <p>
	 * The result should be as defined by {@link Comparator#compare(Object, Object)}.
	 * 
	 * @param <N2> the tail of the incoming stack
	 * @param <N1> the tail of the incoming stack including the right operand
	 * @param <N0> the incoming stack with the right and left operands on top
	 * @param em the emitter typed with the incoming stack
	 * @return the emitter typed with the resulting stack, i.e., the tail with the result pushed
	 */
	<N2 extends Next, N1 extends Ent<N2, TDouble>, N0 extends Ent<N1, TDouble>>
			Emitter<Ent<N2, TInt>> opForDoubleCmp(Emitter<N0> em);

	/**
	 * Emit the JVM opcode to perform the conditional jump.
	 * <p>
	 * The condition should correspond to the true case of the p-code operator.
	 * 
	 * @param <N1> the tail of the incoming stack
	 * @param <N0> the incoming stack including the comparison result
	 * @param em the emitter typed with the incoming stack
	 * @return the target label and emitter typed with the resulting stack, i.e., with the
	 *         comparison result popped
	 */
	<N1 extends Next, N0 extends Ent<N1, TInt>> LblEm<N1, N1> opForCondJump(Emitter<N0> em);

	/**
	 * {@inheritDoc}
	 * 
	 * <p>
	 * This implementation reduces the need to just a few opcodes: 1) the opcode for comparing in
	 * case of JVM {@code float}, 2) the opcode for comparing in the case of JVM {@code double}, and
	 * 3) the conditional jump on the result of that comparison. First, the comparison opcode is
	 * emitted. It should result in and int &lt;0, ==0, or &gt;0 on the stack, depending on whether
	 * L&lt;R, L==R, or L&gt;R, respectively. Then the conditional jump is emitted. We place labels
	 * in an if-else pattern to place either a 1 (true) or 0 (false) value of the appropriate p-code
	 * type on the stack.
	 * 
	 * @implNote This template is consistently generated by the Java compiler (Adoptium OpenJDK 21),
	 *           despite there being possible branchless implementations. That could indicate one of
	 *           a few things: 1) the HotSpot JIT knows how to optimize this pattern, perhaps using
	 *           branchless native instructions, 2) branchless optimizations don't yield the speedup
	 *           here we might expect, or 3) they didn't care to optimize. <b>TODO</b>: Investigate
	 *           in case it's thing 3. We might like to see if branchless JVM bytecodes can improve
	 *           performance.
	 */
	@Override
	default <THIS extends JitCompiledPassage> OpResult genRun(Emitter<Bot> em,
			Local<TRef<THIS>> localThis, Local<TInt> localCtxmod, RetReq<TRef<EntryPoint>> retReq,
			JitCodeGenerator<THIS> gen, T op, JitBlock block, Scope scope) {
		JitType lType = gen.resolveType(op.l(), op.lType());
		JitType rType = gen.resolveType(op.r(), op.rType());
		assert rType == lType;
		Emitter<Ent<Bot, TInt>> emCmp = switch (lType) {
			case FloatJitType t -> em
					.emit(gen::genReadToStack, localThis, op.l(), t, ext())
					.emit(gen::genReadToStack, localThis, op.r(), t, rExt())
					.emit(this::opForFloatCmp);
			case DoubleJitType t -> em
					.emit(gen::genReadToStack, localThis, op.l(), t, Ext.ZERO)
					.emit(gen::genReadToStack, localThis, op.r(), t, Ext.ZERO)
					.emit(this::opForDoubleCmp);
			case MpFloatJitType t -> TODO("MpFloat");
			default -> throw new AssertionError();
		};
		// LATER: Given the type conversion during write, we may not need a jump at all.
		var lblTrue = emCmp
				.emit(this::opForCondJump);
		var lblDone = lblTrue.em()
				.emit(Op::ldc__i, 0)
				.emit(Op::goto_);
		return new LiveOpResult(lblDone.em()
				.emit(Lbl::placeDead, lblTrue.lbl())
				.emit(Op::ldc__i, 1)
				.emit(Lbl::place, lblDone.lbl())
				.emit(gen::genWriteFromStack, localThis, op.out(), IntJitType.I4, Ext.ZERO, scope));
	}
}
