/*
 * Decompiled with CFR 0.152.
 */
package net.dongliu.commons.retry;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.function.Predicate;
import java.util.function.Supplier;
import net.dongliu.commons.collection.Lists;
import net.dongliu.commons.exception.MaxRetryTimesExceedException;
import net.dongliu.commons.exception.RetryInterruptedException;
import net.dongliu.commons.retry.RetryBackOff;
import net.dongliu.commons.retry.RetryBackOffs;
import net.dongliu.commons.retry.RetryListener;

public class Retry {
    private final int maxRetryTimes;
    private final Predicate<Throwable> exceptionPredicate;
    private final RetryBackOff retryBackOff;
    private final List<RetryListener> retryListeners;

    private Retry(Builder builder) {
        this.maxRetryTimes = builder.maxRetryTimes;
        this.exceptionPredicate = builder.exceptionPredicate;
        this.retryListeners = Lists.copyOf(builder.retryListeners);
        this.retryBackOff = builder.retryBackOff;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public Builder toBuilder() {
        Builder builder = new Builder();
        builder.maxRetryTimes = this.maxRetryTimes;
        builder.exceptionPredicate = this.exceptionPredicate;
        builder.retryListeners = new ArrayList<RetryListener>(this.retryListeners);
        builder.retryBackOff = this.retryBackOff;
        return builder;
    }

    public void run(Runnable runnable) {
        this.call(() -> {
            runnable.run();
            return null;
        }, v -> false);
    }

    public <T> T call(Supplier<T> supplier) {
        return (T)this.call(supplier, v -> false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public <T> T call(Supplier<T> supplier, Predicate<? super T> returnValuePredicate) {
        long lastInterval = 0L;
        long lastSecondInterval = 0L;
        for (int i = 0; i <= this.maxRetryTimes; ++i) {
            this.notifyRetryBegin(i);
            try {
                T v = supplier.get();
                if (!returnValuePredicate.test(v)) {
                    T t = v;
                    return t;
                }
            }
            catch (Throwable e) {
                if (!this.exceptionPredicate.test(e)) {
                    throw e;
                }
                if (i == this.maxRetryTimes) {
                    this.notifyMaxRetryTimesExceeded();
                    throw e;
                }
            }
            finally {
                this.notifyRetryEnd(i);
            }
            long interval = this.retryBackOff.nextIntervalOf(i + 1, lastInterval, lastSecondInterval);
            lastSecondInterval = lastInterval;
            lastInterval = interval;
            if (interval < 0L) continue;
            try {
                Thread.sleep(interval);
                continue;
            }
            catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
                throw new RetryInterruptedException();
            }
        }
        this.notifyMaxRetryTimesExceeded();
        throw new MaxRetryTimesExceedException();
    }

    private void notifyRetryBegin(int retryTimes) {
        if (retryTimes > 0) {
            for (RetryListener retryListener : this.retryListeners) {
                try {
                    retryListener.onRetryBegin(retryTimes);
                }
                catch (Throwable e) {
                    e.printStackTrace();
                }
            }
        }
    }

    private void notifyRetryEnd(int retryTimes) {
        if (retryTimes > 0) {
            for (RetryListener retryListener : this.retryListeners) {
                try {
                    retryListener.onRetryEnd(retryTimes);
                }
                catch (Throwable e) {
                    e.printStackTrace();
                }
            }
        }
    }

    private void notifyMaxRetryTimesExceeded() {
        for (RetryListener retryListener : this.retryListeners) {
            try {
                retryListener.onMaxRetryTimesExceeded();
            }
            catch (Throwable e) {
                e.printStackTrace();
            }
        }
    }

    public static final class Builder {
        private int maxRetryTimes;
        private Predicate<Throwable> exceptionPredicate = e -> true;
        private List<RetryListener> retryListeners = new ArrayList<RetryListener>();
        private RetryBackOff retryBackOff;

        private Builder() {
        }

        public Builder maxRetryTimes(int times) {
            this.maxRetryTimes = Builder.checkTimes(times);
            return this;
        }

        public Builder retryIfExceptionMatch(Predicate<Throwable> predicate) {
            this.exceptionPredicate = Objects.requireNonNull(predicate);
            return this;
        }

        public Builder retryIfExceptionIs(Class<? extends Throwable> cls) {
            return this.retryIfExceptionMatch(cls::isInstance);
        }

        public Builder addRetryListener(RetryListener listener) {
            this.retryListeners.add(Objects.requireNonNull(listener));
            return this;
        }

        public Builder retryBackOff(RetryBackOff backOff) {
            this.retryBackOff = Objects.requireNonNull(backOff);
            return this;
        }

        public Builder waitBeforeRetry(long interval) {
            return this.retryBackOff(RetryBackOffs.newFixBackOff(interval));
        }

        public Retry build() {
            if (this.maxRetryTimes < 0) {
                throw new RuntimeException("max retry times not set");
            }
            if (this.retryBackOff == null) {
                throw new RuntimeException("backOff not set");
            }
            return new Retry(this);
        }

        private static int checkTimes(int times) {
            if (times < 0) {
                throw new IllegalArgumentException("invalid retry times");
            }
            return times;
        }
    }
}

