import { AnimatePresence, motion } from 'motion/react'
import { Children, cloneElement, useState, useId } from 'react'
import clsx from 'clsx'

const AnimatedBackground = ({ children, defaultValue, onValueChange, className, transition, enableHover = false }) => {
  // 初始化时使用 defaultValue，避免在 effect 中同步 setState 导致多次渲染
  const [activeId, setActiveId] = useState(() => (defaultValue !== undefined ? defaultValue : null))
  const uniqueId = useId()

  const handleSetActiveId = (id) => {
    setActiveId(id)

    if (onValueChange) {
      onValueChange(id)
    }
  }

  return Children.map(children, (child, index) => {
    const id = child.props['data-id']

    const interactionProps = enableHover
      ? {
          onMouseEnter: () => handleSetActiveId(id),
          onMouseLeave: () => handleSetActiveId(null),
        }
      : {
          onClick: () => handleSetActiveId(id),
        }

    return cloneElement(
      child,
      {
        key: index,
        className: clsx('relative inline-flex', child.props.className),
        'data-checked': activeId === id ? 'true' : 'false',
        ...interactionProps,
      },
      <>
        <AnimatePresence initial={false}>
          {activeId === id && (
            <motion.div
              layoutId={`background-${uniqueId}`}
              className={clsx('absolute inset-0', className)}
              transition={transition}
              initial={{ opacity: defaultValue ? 1 : 0 }}
              animate={{
                opacity: 1,
              }}
              exit={{
                opacity: 0,
              }}
            />
          )}
        </AnimatePresence>
        <div className="z-10">{child.props.children}</div>
      </>
    )
  })
}

export default AnimatedBackground
