---
order: 30
title: OpenAPI
group:
  title: Network Request
nav:
  title: Documents
  path: /docs
  order: 1
---

In specific development, joint debugging is always troublesome, especially after the front-end and back-end are separated, the back-end generally needs to maintain a document to tell us what specific API functions, specific field information, and the information Maintenance costs are still quite high.

## Install plugin

In Pro, we introduced an openAPI plug-in. In the scaffolding, we have this feature. If you are using the unofficial version of v5, you can install the plug-in with the following command.

```bash
 yarn add @umijs/plugin-openapi

 // npm
 npm i @umijs/plugin-openapi --save
```

Then configure the relevant configuration of openAPI in `config/config.ts`.

```tsx | pure
openAPI: {
   requestLibPath: "import {request} from'umi'",
   // Or use the online version
   // schemaPath: "https://gw.alipayobjects.com/os/antfincdn/M%24jrzTTYJN/oneapi.json",
   schemaPath: join(__dirname,'oneapi.json'),
   mock: false,
 }
```

You also need to add a command to the scripts of package.json.

```ts
"openapi": "umi openapi",
```

Finally, we can execute `npm run openapi` to generate related interfaces and documents.

## 如何使用

openAPI has some workload for the backend, but the workload is far less than the cost of maintaining a document. If you maintain a document, you need to edit the document every time you update the code. With openAPI, you only need to access swagger and do some configuration to generate an interface. If you are using python or java, then access will become easy. For detailed access steps, please see the official document of [swagger](https://swagger.io/). Here mainly introduces how to use the front end.

After the back-end access to swagger is completed, we can access the documents generated by swagger, which are generally `http://localhost:8080/swagger-ui.html`, and we can get an openapi specification file by visiting the page.

![swagger-ui](https://gw.alipayobjects.com/zos/antfincdn/c0uvca5Mx4/184D8866-6067-43DC-8395-0DC031D1A873.png)

We need to copy the url of swagger to the configuration of openapi. Taking the openapi of pro as an example, let's configure it:

```tsx | pure
openAPI: {
   requestLibPath: "import {request} from'umi'",
   // use the url of copy here
   schemaPath: "https://gw.alipayobjects.com/os/antfincdn/M%24jrzTTYJN/oneapi.json",
   mock: false,
 }
```

There are two configurations `requestLibPath` and `mock` that need to be noted.

### requestLibPath

How can `requestLibPath` use `request`? Generally speaking, we recommend using umi's request directly, but sometimes you need to customize it and you can modify the configuration of `requestLibPath`. For example, to use the request in utils, we can configure it like this:

```tsx | pure
openAPI: {
   schemaPath: "import request from'@utils/request",
   // use the url of copy here
   schemaPath: "https://gw.alipayobjects.com/os/antfincdn/M%24jrzTTYJN/oneapi.json",
   mock: false,
 }
```

Of course, you need to ensure that the `schemaPath` configuration introduces the request, otherwise the generated code may not be executed. The generated code is as follows:

```tsx | pure
// configuration of requestLibPath
import { request } from 'umi';

/** Get the list of rules GET /api/rule */
export async function rule(params: API.PageParams, options?: { [key: string]: any }) {
  return request<API.RuleList>('/api/rule', {
    method: 'GET',
    params: {
      ...params,
    },
    ...(options || {}),
  });
}
```

The comments will also be loaded automatically, saving us the trouble of checking the documentation. At the same time, we will also generate the `typings.d.ts` file in the serves, which contains all the definitions in the openapi. `API.RuleList` is the description of the data that the backend needs to return. Examples are as follows:

```tsx | pure
declare namespace API {
  type RuleListItem = {
    key?: number;
    disabled?: boolean;
    href?: string;
    avatar?: string;
    name?: string;
    owner?: string;
    desc?: string;
    callNo?: number;
    status?: number;
    updatedAt?: string;
    createdAt?: string;
    progress?: number;
  };

  type RuleList = {
    data?: RuleListItem[];
    /** The total content of the list */
    total?: number;
    success?: boolean;
  };
}
```

In this way, we can cooperate with ProTable to quickly make a CRUD, the code is easy.

```tsx | pure
import { rule } from '@/services/ant-design-pro/rule';

// Two generics, the first is the type definition of the list item, and the second is the definition of the query parameter.
// 🥳 A table has been generated
<ProTable<API.RuleListItem, API.PageParams> request={rule} columns={columns} />;
```

### mock

`mock` is relatively easy. After setting it to true, some mock files will be automatically generated. Although the quality is not as good as ours, it is no problem to use it in development.

The generated mock file is in the mock file under the project root path. The generated mock data is different every time. If you want to debug, you can modify it at will. Only by executing `npm run openapi` will it be modified.

```tsx | pure
import { Request, Response } from 'express';

export default {
  'GET /api/rule': (req: Request, res: Response) => {
    res.status(200).send({
      data: [
        {
          key: 86,
          disabled: false,
          href: 'https://ant.design',
          avatar: 'https://gw.alipayobjects.com/zos/rmsportal/KDpgvguMpGfqaHPjicRK.svg',
          name: 'Luo Xiulan',
          owner: 'Garcia',
          desc:
            'Sida kind of rectification and construction is difficult, but wait for it to come to light. ',
          callNo: 96,
          status: 89,
          updatedAt: 'PpVmJ50',
          createdAt: 'FbRG',
          progress: 100,
        },
      ],
      total: 98,
      success: false,
    });
  },
};
```

### Documentation

In development, we can't just look at the code, but also the documentation. Swagger-ui is also integrated by default in Pro, which provides an interface to read the openapi configuration in the current project. We can find a shortcut in the lower right corner of the Layout:

![options](https://gw.alipayobjects.com/zos/antfincdn/htZYO3ojsm/957C1E97-466B-48aa-B107-FA7C4EFF6971.png)

This operation is only valid in the development environment. If it is a lower version, you can visit `/umi/plugin/openapi` to view, the final effect should be like this:

![doc](https://gw.alipayobjects.com/zos/antfincdn/htZYO3ojsm/957C1E97-466B-48aa-B107-FA7C4EFF6971.png)
