#!/bin/bash
# 全面功能验证测试

cd "$(dirname "$0")"

echo "========================================"
echo "Codelin 全面功能验证"
echo "========================================"
echo ""

# 颜色定义
GREEN='\033[0;32m'
RED='\033[0;31m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

test_count=0
pass_count=0
fail_count=0

run_test() {
    local test_name="$1"
    local test_cmd="$2"
    local expected_pattern="$3"
    
    test_count=$((test_count + 1))
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
    echo "测试 #$test_count: $test_name"
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
    
    # 执行测试
    local output=$(eval "$test_cmd" 2>&1)
    local exit_code=$?
    
    # 检查结果
    if [[ -z "$expected_pattern" ]]; then
        # 只检查退出码
        if [ $exit_code -eq 0 ]; then
            echo -e "${GREEN}✅ PASS${NC}"
            pass_count=$((pass_count + 1))
        else
            echo -e "${RED}❌ FAIL${NC} (exit code: $exit_code)"
            fail_count=$((fail_count + 1))
        fi
    else
        # 检查输出模式
        if echo "$output" | grep -q "$expected_pattern"; then
            echo -e "${GREEN}✅ PASS${NC}"
            pass_count=$((pass_count + 1))
        else
            echo -e "${RED}❌ FAIL${NC}"
            echo "期望: $expected_pattern"
            echo "输出: $(echo "$output" | head -5)"
            fail_count=$((fail_count + 1))
        fi
    fi
    
    echo ""
}

echo "开始测试..."
echo ""

# ============================================
# 1. 编译和环境检查
# ============================================
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo "第一部分：编译和环境检查"
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo ""

echo "1.1 检查源代码修复"
if grep -q "if (len <= 0)" src/io/raw_input_utils_unix.cj; then
    echo -e "${GREEN}✅ 源代码已修复${NC}"
else
    echo -e "${RED}❌ 源代码未修复${NC}"
    exit 1
fi

echo ""
echo "1.2 检查 FFI 库"
if [ -f "ffi/librawinput.dylib" ]; then
    ffi_time=$(stat -f "%Sm" -t "%Y-%m-%d %H:%M" ffi/librawinput.dylib)
    echo -e "${GREEN}✅ FFI 库存在${NC} (编译时间: $ffi_time)"
else
    echo -e "${RED}❌ FFI 库不存在${NC}"
    exit 1
fi

echo ""
echo "1.3 检查可执行文件"
if [ -f "target/release/bin/cli" ]; then
    bin_time=$(stat -f "%Sm" -t "%Y-%m-%d %H:%M" target/release/bin/cli)
    echo -e "${GREEN}✅ 可执行文件存在${NC} (编译时间: $bin_time)"
else
    echo -e "${RED}❌ 可执行文件不存在${NC}"
    echo "运行 cjpm build 进行编译"
    exit 1
fi

echo ""
echo ""

# ============================================
# 2. 基础功能测试
# ============================================
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo "第二部分：基础功能测试"
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo ""

run_test "2.1 程序启动测试" \
    "echo 'exit' | cjpm run --name cli" \
    "CODELIN"

run_test "2.2 EOF 处理测试（空输入）" \
    "echo '' | cjpm run --name cli" \
    ""

run_test "2.3 EOF 处理测试（/dev/null）" \
    "cjpm run --name cli < /dev/null" \
    ""

echo "2.4 异常检测测试"
if echo '' | cjpm run --name cli 2>&1 | grep -q "Exception.*Read bytes 4294967295"; then
    echo -e "${RED}❌ FAIL - 仍然抛出异常${NC}"
    fail_count=$((fail_count + 1))
else
    echo -e "${GREEN}✅ PASS - 无异常${NC}"
    pass_count=$((pass_count + 1))
fi
test_count=$((test_count + 1))

echo ""
echo ""

# ============================================
# 3. 命令功能测试
# ============================================
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo "第三部分：命令功能测试"
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo ""

run_test "3.1 exit 命令测试" \
    "echo 'exit' | cjpm run --name cli" \
    "Thanks for using"

run_test "3.2 /help 命令测试" \
    "echo -e '/help\nexit' | cjpm run --name cli" \
    ""

run_test "3.3 /sessions 命令测试" \
    "echo -e '/sessions\nexit' | cjpm run --name cli" \
    ""

echo ""
echo ""

# ============================================
# 4. 交互式测试提示
# ============================================
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo "第四部分：交互式功能测试"
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo ""

echo -e "${YELLOW}⚠️  以下测试需要手动交互${NC}"
echo ""
echo "请手动测试以下功能："
echo ""
echo "4.1 Ctrl+D 退出测试"
echo "    命令: cjpm run --name cli"
echo "    操作: 在提示符处按 Ctrl+D"
echo "    期望: 程序优雅退出，显示 'Thanks for using'"
echo ""
echo "4.2 Ctrl+C 中断测试"
echo "    命令: cjpm run --name cli"
echo "    操作: 输入一个问题，等待响应时按 Ctrl+C"
echo "    期望: 取消当前操作，返回提示符"
echo ""
echo "4.3 正常对话测试"
echo "    命令: cjpm run --name cli"
echo "    操作: 输入 '你好'"
echo "    期望: AI 正常响应"
echo ""
echo "4.4 文件引用测试"
echo "    命令: cjpm run --name cli"
echo "    操作: 输入 '@README.md'"
echo "    期望: 引用文件内容"
echo ""
echo "4.5 Shell 执行测试"
echo "    命令: cjpm run --name cli"
echo "    操作: 输入 '!ls -la'"
echo "    期望: 执行 shell 命令"
echo ""

read -p "按 Enter 运行交互式测试，或 Ctrl+C 跳过..."

echo ""
echo "启动交互式测试..."
echo ""
echo -e "${YELLOW}提示：${NC}"
echo "  1. 程序启动后，尝试按 Ctrl+D"
echo "  2. 观察是否正常退出（无异常）"
echo "  3. 可以尝试输入各种命令"
echo ""

# 启动交互式测试
cjpm run --name cli

echo ""
echo ""

# ============================================
# 5. 测试总结
# ============================================
echo "========================================"
echo "测试总结"
echo "========================================"
echo ""
echo "总测试数: $test_count"
echo -e "${GREEN}通过: $pass_count${NC}"
echo -e "${RED}失败: $fail_count${NC}"
echo ""

if [ $fail_count -eq 0 ]; then
    echo -e "${GREEN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -e "${GREEN}✅ 所有自动化测试通过！${NC}"
    echo -e "${GREEN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo ""
    echo "🎉 EOF 异常修复验证成功！"
    echo ""
    echo "所有功能正常，可以安心使用 Codelin！"
    exit 0
else
    echo -e "${RED}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -e "${RED}❌ 有测试失败${NC}"
    echo -e "${RED}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo ""
    echo "请检查失败的测试项"
    exit 1
fi

