# CODELIN Configuration

## Project Information

**Project Name**: CodeLin
**Project Type**: AI-powered code assistant and development tool
**Primary Language**: Cangjie (.cj)
**Framework**: CangjieMagic ecosystem
**Architecture**: Multi-layered agent-based system with LSP integration

### Project Structure

```
codelin/
├── src/                    # Core source code
│   ├── main.cj             # Entry point
│   ├── core/               # Core modules
│   │   ├── agents/          # AI agent implementations
│   │   ├── context/         # Context management
│   │   ├── tools/           # Tool implementations
│   │   ├── config/          # Configuration management
│   │   ├── api/             # API interfaces
│   │   └── services/        # Service implementations
│   ├── app/                # Application layer
│   ├── io/                 # Input/output handling
│   ├── lsp/                # Language Server Protocol
│   └── utils/              # Utility functions
├── cjpm.toml              # Project configuration
├── build.cj               # Build configuration
├── codebin/                # Compiled binaries
├── scripts/                # Build and utility scripts
├── docs/                   # Documentation
├── codebuff/               # Subproject - Node.js tool
└── tui/                    # Terminal UI subproject
```

## Coding Standards

### Cangjie Language Conventions

1. **File Naming**: Use snake_case for files (e.g., `file_utils.cj`)
2. **Class Naming**: Use PascalCase for classes (e.g., `FSToolset`)
3. **Function Naming**: Use camelCase for functions (e.g., `readFile`, `createCliApp`)
4. **Constant Naming**: Use UPPER_SNAKE_CASE for constants
5. **Package Structure**: Follow hierarchical naming (e.g., `codelin.core.tools`)

### Code Organization

1. **Module Organization**: Group related functionality in packages
2. **File Size**: Keep files under 1000 lines when possible
3. **Function Length**: Prefer functions under 50 lines
4. **Class Design**: Single responsibility principle
5. **Error Handling**: Use Result<T> types for error propagation

### Documentation Standards

1. **Public APIs**: Must have comprehensive documentation
2. **Complex Logic**: Add inline comments for non-obvious code
3. **Code Examples**: Provide usage examples in docstrings
4. **API Contracts**: Document all input/output types

## Development Guide

### Build System

```bash
# Build the project
cjpm build

# Run tests
cjpm test

# Build release version
cjpm build --release

# Install dependencies
cjpm install
```

### Testing Strategy

1. **Unit Tests**: Test individual functions and classes
2. **Integration Tests**: Test component interactions
3. **LSP Tests**: Verify language server functionality
4. **Agent Tests**: Test AI agent behavior

### Core Components

#### 1. Agent System
- **Location**: `src/core/agents/`
- **Purpose**: AI-powered code analysis and generation
- **Key Classes**: `CodeLinAgent`, `GeneralCodeAgent`
- **Subagents**: Specialized agents for different tasks

#### 2. Context Engine
- **Location**: `src/core/context/`
- **Purpose**: File context management and caching
- **Key Classes**: `ContextEngine`, `FileCacheManager`
- **Features**: BM25 scoring, dependency analysis

#### 3. Tool System
- **Location**: `src/core/tools/`
- **Purpose**: File system operations and language services
- **Key Classes**: `FSToolset`, `LSPToolset`, `CangjieToolset`
- **Features**: Parallel processing, caching, error recovery

#### 4. LSP Integration
- **Location**: `src/lsp/`
- **Purpose**: Language Server Protocol implementation
- **Key Classes**: `LspClient`, `SemanticInfo`
- **Features**: Symbol analysis, code completion

#### 5. Configuration System
- **Location**: `src/core/config/`
- **Purpose**: Project and runtime configuration
- **Key Classes**: `CodeLinConfigManager`, `MemoryConfig`
- **Features**: Multi-level config, model token limits

### Performance Considerations

1. **Memory Management**: Use efficient caching strategies
2. **Parallel Processing**: Leverage concurrent execution where possible
3. **File I/O**: Use lazy loading and partial reads
4. **Token Optimization**: Optimize for LLM token usage

### Error Handling

1. **Custom Error Types**: Define specific error types for different domains
2. **Recovery Strategies**: Implement retry logic with exponential backoff
3. **User Feedback**: Provide clear error messages and recovery suggestions
4. **Logging**: Use structured logging for debugging

## AI Assistant Guidance

### When Working with CodeLin Code

1. **Use LSP Tools First**: Before reading files, use `getFileSymbols`, `getSymbolsByKind`, `findSymbolByName` to understand structure
2. **Leverage Context Engine**: Use the context system to understand file relationships
3. **Follow Agent Patterns**: Understand the multi-agent architecture before making changes
4. **Respect Configuration**: Honor the multi-level configuration system

### Common Development Patterns

1. **Tool Development**: Follow the established tool pattern in `src/core/tools/`
2. **Agent Creation**: Use the agent framework for new AI capabilities
3. **Configuration**: Use `CodeLinConfigManager` for all configuration needs
4. **Error Handling**: Follow the established error handling patterns

### Code Analysis Workflow

1. **Discover**: Use `globSearch` to find relevant files
2. **Analyze**: Use LSP tools to understand code structure
3. **Context**: Use context engine to understand relationships
4. **Modify**: Apply changes following established patterns
5. **Test**: Verify changes work with existing test suite

### Performance Optimization

1. **Caching**: Leverage the file cache manager for repeated operations
2. **Parallel Execution**: Use parallel tools for batch operations
3. **Memory Management**: Monitor memory usage in long-running operations
4. **Token Efficiency**: Optimize prompts and context for LLM interactions

### Integration Points

1. **File System**: Use `FSToolset` for all file operations
2. **Language Features**: Use `LSPToolset` for code-aware operations
3. **AI Capabilities**: Use agent system for intelligent code operations
4. **Configuration**: Respect user and project configuration

### Best Practices

1. **Modular Design**: Keep components loosely coupled
2. **Testing First**: Write tests before implementing features
3. **Documentation**: Maintain comprehensive documentation
4. **Performance**: Consider performance implications of changes
5. **Compatibility**: Maintain backward compatibility when possible

### Testing Commands

```bash
# Run specific test file
cjpm test src/core/tools/fs_toolset_test.cj

# Run all tests in a package
cjpm test src/core/tools/

# Run with verbose output
cjpm test --verbose
```

### Debugging

1. **Logging**: Use the built-in logging system
2. **Performance**: Use telemetry system for performance analysis
3. **Error Recovery**: Leverage the error recovery system
4. **Context Analysis**: Use context engine for debugging file relationships

### Tool Usage Guidelines

1. **File Operations**: Prefer `FSToolset.readFile` with ranges for large files
2. **Symbol Analysis**: Use LSP tools for code understanding
3. **Batch Operations**: Use parallel tools for multiple file operations
4. **Context Management**: Let the context engine manage file relationships

This configuration serves as a comprehensive guide for working with the CodeLin project, emphasizing its AI-powered nature, Cangjie language specifics, and sophisticated tool ecosystem.