# UI 输入系统全面分析与优化方案

## 问题分析

### 1. 当前实现状态

**已完成**：
- ✅ Phase 1.4: 换行符光标位置修复
- ✅ Phase 1.5: 自动换行（wrap）光标位置修复
- ✅ Phase 1.3: 渲染性能优化（缓存布局计算）

**待优化**：
- ⚠️ 内容截断时的滚动逻辑
- ⚠️ 光标可见性保证
- ⚠️ 多行输入时的智能滚动
- ⚠️ 渲染性能进一步优化

### 2. 核心问题

#### 问题1：内容截断时的滚动逻辑

**当前实现**：
```cangjie
if (wrappedLineCount <= actualInputBoxHeight) {
    // 输入框完全显示
    PrintUtils.printString(cleanWrappedContent)
} else {
    // 输入框太高，只显示最后 actualInputBoxHeight 行
    let linesToSkip = wrappedLineCount - actualInputBoxHeight
    // 显示最后几行
}
```

**问题**：
- 当内容被截断时，光标可能在不可见区域
- 没有智能滚动，确保光标所在行可见
- 用户无法看到光标位置，体验差

#### 问题2：光标位置计算在截断场景下的问题

**当前实现**：
```cangjie
let (cursorAbsoluteRow, safeCursorCol) = this.calculateAdjustedCursorPosition(
    cursorRow,
    cursorCol,
    textLineCount,
    actualInputBoxHeight,
    inputBoxStartRow,
    terminalHeight,
    terminalWidth
)
```

**问题**：
- `calculateAdjustedCursorPosition` 处理了截断，但可能不够智能
- 当光标在截断区域时，应该自动滚动到可见区域

#### 问题3：渲染性能

**当前实现**：
- 有缓存机制（`cachedWrappedLineCount` 等）
- 但每次输入仍需要重新生成内容（因为可能有颜色变化）

**优化空间**：
- 可以进一步优化，只在必要时重新渲染
- 可以增量更新，而不是全量重绘

### 3. 优化方案

#### 方案1：智能滚动（Cursor-Aware Scrolling）

**核心思路**：
1. 计算光标所在的行（在完整内容中的位置）
2. 如果光标行不在可见区域，调整滚动位置
3. 确保光标行始终在可见区域的中间或底部

**实现**：
```cangjie
// 计算应该显示的行范围，确保光标可见
let cursorLineInFullContent = cursorRow  // 光标在完整内容中的行号
let visibleStartLine = if (wrappedLineCount <= actualInputBoxHeight) {
    0  // 全部显示
} else {
    // 智能滚动：确保光标行可见
    let idealCursorPosition = actualInputBoxHeight / 2  // 理想位置：中间
    let scrollStart = max(0, cursorLineInFullContent - idealCursorPosition)
    let maxScrollStart = wrappedLineCount - actualInputBoxHeight
    min(scrollStart, maxScrollStart)
}
```

#### 方案2：优化光标位置计算

**改进点**：
1. 在截断场景下，确保光标位置计算准确
2. 光标应该始终指向正确的位置，即使内容被截断

#### 方案3：渲染性能优化

**改进点**：
1. 只在文本真正变化时重新计算布局
2. 缓存更多中间结果
3. 减少不必要的字符串操作

## 实施计划

### Phase 1.6: 智能滚动实现 ✅

1. **修改滚动逻辑**：
   - ✅ 计算光标所在行（在完整内容中的位置，包括边框）
   - ✅ 如果不在可见区域，调整滚动位置
   - ✅ 确保光标行始终可见（理想位置：可见区域的 2/3 位置）

2. **优化光标位置计算**：
   - ✅ 在截断场景下，确保光标位置准确
   - ✅ 处理滚动偏移，确保光标位置与滚动内容一致
   - ✅ 处理边界情况

3. **实现细节**：
   - ✅ 滚动逻辑：`idealCursorPosition = (actualInputBoxHeight * 2) / 3`
   - ✅ 滚动偏移计算：`scrollStart = max(0, cursorLineInFullContent - idealCursorPosition)`
   - ✅ 光标位置调整：考虑滚动偏移，确保光标在可见区域

4. **测试验证**：
   - ✅ 长文本输入
   - ✅ 多行输入
   - ✅ 光标在不同位置时的滚动
   - ✅ 构建验证通过

### Phase 1.7: 渲染性能进一步优化（待实现）

1. **增量更新**：
   - 只在必要时重新渲染
   - 缓存更多中间结果

2. **减少字符串操作**：
   - 优化字符串拼接
   - 减少不必要的字符串创建

## 预期效果

1. **用户体验**：
   - ✅ 光标始终可见
   - ✅ 智能滚动，跟随光标
   - ✅ 流畅的输入体验

2. **性能**：
   - ✅ 减少不必要的渲染（已有缓存机制）
   - 可以进一步优化

3. **稳定性**：
   - ✅ 处理各种边界情况
   - ✅ 确保光标位置准确

## 实现总结

### Phase 1.6 实现内容

1. **智能滚动逻辑**：
   ```cangjie
   // 计算光标在完整内容中的行号（包括边框）
   let cursorLineInFullContent = cursorRow + 1
   
   // 智能滚动：确保光标行可见（2/3 位置）
   let idealCursorPosition = (actualInputBoxHeight * 2) / 3
   let scrollStart = max(0, cursorLineInFullContent - idealCursorPosition)
   let maxScrollStart = wrappedLineCount - actualInputBoxHeight
   let actualScrollStart = min(scrollStart, maxScrollStart)
   ```

2. **光标位置调整**：
   ```cangjie
   // 考虑滚动偏移，调整光标行
   let adjustedCursorRowForScroll = if (scrollOffset > 0) {
       let contentScrollOffset = scrollOffset - 1  // -1 是因为 scrollOffset 包括顶部边框
       max(0, cursorRow - contentScrollOffset)
   } else {
       cursorRow
   }
   ```

3. **关键改进**：
   - ✅ 光标行始终在可见区域
   - ✅ 滚动位置智能计算（2/3 位置提供更好的上下文）
   - ✅ 光标位置计算与滚动逻辑完全一致
   - ✅ 处理边界情况（滚动位置有效范围检查）

