# CodeLin 22：完整包架构与平台化改造方案

> **基于 codelin21.md 的全面总结与深化**  
> 面向仓颉生态的下一代函数式多代理平台：完整包架构设计、整体系统架构、模块化扩展策略

---

## 📋 文档索引

本文档是 `codelin21.md` 的完整总结与架构深化版本，包含：

1. **完整包架构设计**：基于仓颉多模块化特性的包结构
2. **整体架构图**：分层架构与数据流
3. **模块化扩展策略**：开源/闭源、插件、跨领域
4. **实施路线图**：详细的分阶段改造计划

---

## 1. 执行摘要

### 1.1 改造目标

将 CodeLin 从单一 CLI 应用升级为**仓颉函数式多代理平台**，实现：

- ✅ **函数式内核**：充分利用仓颉不可变、纯函数、模式匹配特性
- ✅ **模块化包架构**：基于仓颉包管理（`cjpm.toml`）的清晰分层
- ✅ **平台化扩展**：支持插件、开源/闭源双轨、跨领域应用
- ✅ **重复率治理**：从当前 23-84% 降至 <15%
- ✅ **对标顶级系统**：Claude Code、CodeBuff、Cursor Augment Code、Claude Agent SDK

### 1.2 核心原则

1. **高内聚低耦合**：每个包单一职责，通过接口契约交互
2. **函数式优先**：不可变状态 + 纯函数 + IOEffect 边界
3. **可插拔平台**：统一 Plugin/Agent/Tool 元数据接口
4. **跨领域扩展**：知识域抽象，不限于编程

---

## 2. 完整包架构设计

### 2.1 包结构总览

基于仓颉包管理（`cjpm.toml`）和 CodeBuff 的 monorepo 经验，设计如下包结构：

```
codelin/
├── cjpm.toml                    # 根包配置
├── packages/                     # 多包工作区（类似 CodeBuff）
│   ├── core/                     # 核心运行时包
│   │   ├── cjpm.toml
│   │   └── src/
│   │       ├── runtime/           # 函数式运行时
│   │       ├── state/            # 状态管理
│   │       ├── effects/          # IOEffect 系统
│   │       └── functional/      # 函数式工具库
│   │
│   ├── context/                  # 上下文平台包
│   │   ├── cjpm.toml
│   │   └── src/
│   │       ├── kernel/           # ContextKernel 2.1
│   │       ├── conversation/      # 对话管理
│   │       ├── knowledge/        # 知识域抽象
│   │       └── cache/            # 缓存层
│   │
│   ├── agents/                   # 代理编排包
│   │   ├── cjpm.toml
│   │   └── src/
│   │       ├── mesh/             # 多代理调度
│   │       ├── definitions/      # AgentDefinition Schema
│   │       ├── orchestration/    # 任务图编排
│   │       └── registry/         # 代理注册表
│   │
│   ├── tools/                    # 工具总线包
│   │   ├── cjpm.toml
│   │   └── src/
│   │       ├── bus/              # ToolBus 核心
│   │       ├── abi/              # Tool ABI Compiler
│   │       ├── plugins/          # 工具插件接口
│   │       └── adapters/         # MCP/LSP/外部适配
│   │
│   ├── surfaces/                 # 体验层包
│   │   ├── cjpm.toml
│   │   └── src/
│   │       ├── cli/              # CLI Surface
│   │       ├── tui/              # TUI Surface
│   │       ├── ide/              # IDE 插件桥接
│   │       └── api/              # API Gateway
│   │
│   ├── plugins/                  # 插件系统包
│   │   ├── cjpm.toml
│   │   └── src/
│   │       ├── loader/           # 插件加载器
│   │       ├── registry/          # 插件注册表
│   │       ├── schema/           # Plugin.toml Schema
│   │       └── hooks/            # Hook 系统
│   │
│   ├── knowledge-packs/          # 知识域包
│   │   ├── code/                 # 编程域
│   │   ├── docs/                 # 文档域
│   │   ├── data/                 # 数据域
│   │   └── creative/             # 创意域
│   │
│   ├── oss/                      # 开源版打包
│   │   └── cjpm.toml            # 公开依赖配置
│   │
│   └── enterprise/               # 企业版包（闭源）
│       └── cjpm.toml            # 私有依赖配置
│
└── src/                          # 当前单体代码（逐步迁移）
    ├── main.cj                   # 入口（保留）
    └── ...                       # 逐步迁移到 packages/
```

### 2.2 核心包详细设计

#### 2.2.1 `packages/core` - 函数式运行时核心

**职责**：提供不可变状态、IOEffect、函数组合等基础设施

**包结构**：
```cangjie
package codelin.core

// runtime/state_manager.cj
public struct StateManager<T> {
    private var state: T
    private let lock: ReadWriteLock
    
    public func applyTransition(
        transition: (T) -> Result<(T, Array<IOEffect>), Error>
    ): Result<Array<IOEffect>, Error>
}

// effects/io_effect.cj
public enum IOEffect {
    | Log(message: String, level: LogLevel)
    | WriteFile(path: Path, content: String)
    | ReadFile(path: Path)
    | NetworkRequest(url: String, method: String)
    | ExecuteTool(toolName: String, params: HashMap<String, String>)
}

// functional/pipeline.cj
public func pipe<T, R>(value: T, ...functions: Array<(T) -> R>): R
public func compose<T, U, V>(f: (U) -> V, g: (T) -> U): (T) -> V
```

**依赖关系**：
- 无外部依赖（纯仓颉标准库）
- 为所有上层包提供基础

#### 2.2.2 `packages/context` - 上下文平台

**职责**：对话感知上下文、文件图、知识域抽象

**包结构**：
```cangjie
package codelin.context

// kernel/context_kernel.cj
public struct ContextState {
    public let fileCache: HashMap<Path, FileContext>
    public let tokenCount: Int64
    public let maxTokens: Int64
    public let roundSnapshots: Array<RoundContextSnapshot>
    public let memoryDAG: MemoryDAG
}

public func evolveContext(
    state: ContextState,
    event: ContextEvent
): Result<(ContextState, Array<IOEffect>), ContextError>

// conversation/conversation_state.cj
public struct ConversationState {
    public let conversation: Conversation
    public let sessionId: String
    public let history: Array<SessionMetadata>
    public let topicEvolution: TopicEvolution
}

// knowledge/knowledge_pack.cj
public interface KnowledgePack {
    func extractContext(state: ContextState): Array<FileContext>
    func scoreRelevance(query: String, file: FileContext): Float64
    func renderSummary(context: Array<FileContext>): String
}
```

**依赖关系**：
- `codelin.core` - 状态管理、IOEffect

#### 2.2.3 `packages/agents` - 代理编排

**职责**：多代理调度、AgentDefinition Schema、任务图编排

**包结构**：
```cangjie
package codelin.agents

// definitions/agent_definition.cj
public struct AgentDefinition {
    public let id: String
    public let displayName: String
    public let model: String
    public let systemPrompt: String
    public let toolNames: Array<String>
    public let spawnableAgents: Array<String>
    public let instructionsPrompt: Option<String>
    // 兼容 Claude Agent SDK Schema
}

// mesh/agent_mesh.cj
public class AgentMesh {
    public func spawnAgent(
        definition: AgentDefinition,
        context: AgentContext
    ): Result<AgentHandle, AgentError>
    
    public func orchestrate(
        mission: MissionGraph
    ): Result<MissionResult, OrchestrationError>
}

// orchestration/mission_graph.cj
public enum MissionNodeStatus {
    | Pending
    | Running
    | Blocked
    | Done
    | Failed
}

public struct MissionGraph {
    public let nodes: Array<MissionNode>
    public let edges: Array<MissionEdge>
}
```

**依赖关系**：
- `codelin.core` - 状态管理
- `codelin.context` - 上下文提供
- `codelin.tools` - 工具调用

#### 2.2.4 `packages/tools` - 工具总线

**职责**：ToolBus、ABI Compiler、工具插件接口

**包结构**：
```cangjie
package codelin.tools

// bus/tool_bus.cj
public class ToolBus {
    public func dispatch(
        request: ToolRequest
    ): Result<ToolResponse, ToolError>
    
    public func registerPlugin(plugin: ToolPlugin): Unit
    public func loadMCP(server: MCPServer): Result<Unit, MCPError>
}

// abi/tool_abi_compiler.cj
public class ToolABICompiler {
    // 解析 @toolset/@tool 注释 → 生成 ABI JSON
    public func compileTool(
        sourceFile: Path
    ): Result<ToolABI, CompileError>
    
    public func validateABI(abi: ToolABI): Result<Unit, ValidationError>
}

// plugins/tool_plugin.cj
public interface ToolPlugin {
    func getName(): String
    func getSchema(): ToolSchema
    func execute(
        params: HashMap<String, String>
    ): Result<String, ToolError>
}
```

**依赖关系**：
- `codelin.core` - IOEffect、状态管理
- `codelin.plugins` - 插件接口

#### 2.2.5 `packages/surfaces` - 体验层

**职责**：CLI/TUI/IDE/API 适配层

**包结构**：
```cangjie
package codelin.surfaces

// cli/cli_surface.cj
public class CLISurface implements SurfacePlugin {
    public func render(state: AppState): Unit
    public func handleInput(input: String): Result<Command, ParseError>
}

// tui/tui_surface.cj
public class TUISurface implements SurfacePlugin {
    // Cursor Augment Code 风格面板
    public func renderAugmentPanel(
        selection: CodeSelection,
        suggestions: Array<Suggestion>
    ): Unit
}

// api/api_gateway.cj
public class APIGateway {
    public func handleMission(request: MissionRequest): Result<MissionResponse, APIError>
    public func getRound(id: String): Result<RoundSnapshot, APIError>
}
```

**依赖关系**：
- `codelin.core` - 状态管理
- `codelin.agents` - 代理调用
- `codelin.context` - 上下文获取

#### 2.2.6 `packages/plugins` - 插件系统

**职责**：插件加载、注册、Hook 系统

**包结构**：
```cangjie
package codelin.plugins

// loader/plugin_loader.cj
public class PluginLoader {
    public func loadPlugin(
        pluginPath: Path
    ): Result<Plugin, LoadError>
    
    public func validatePlugin(plugin: Plugin): Result<Unit, ValidationError>
}

// registry/plugin_registry.cj
public class PluginRegistry {
    public func register(plugin: Plugin): Unit
    public func getPlugin(name: String): Option<Plugin>
    public func getHooks(hookType: HookType): Array<Hook>
}

// schema/plugin_schema.cj
public struct PluginManifest {
    public let name: String
    public let version: String
    public let dependencies: Array<String>
    public let tools: Array<ToolDefinition>
    public let agents: Array<AgentDefinition>
    public let surfaces: Array<SurfaceDefinition>
    public let hooks: Array<HookDefinition>
}
```

**依赖关系**：
- `codelin.core` - 基础类型
- 所有其他包（作为扩展点）

### 2.3 知识域包设计

#### 2.3.1 `packages/knowledge-packs/code` - 编程域

```cangjie
package codelin.knowledge.code

public class CodeKnowledgePack implements KnowledgePack {
    public func extractContext(state: ContextState): Array<FileContext> {
        // 基于 LSP、依赖分析提取代码上下文
    }
    
    public func scoreRelevance(query: String, file: FileContext): Float64 {
        // BM25 + 语义相似度
    }
}
```

#### 2.3.2 `packages/knowledge-packs/docs` - 文档域

```cangjie
package codelin.knowledge.docs

public class DocsKnowledgePack implements KnowledgePack {
    // 文档解析、章节提取、交叉引用
}
```

### 2.4 开源/闭源包设计

#### 2.4.1 `packages/oss` - 开源版

**包含包**：
- `core`, `context`, `agents`, `tools`, `surfaces/cli`, `surfaces/tui`
- `knowledge-packs/code`, `knowledge-packs/docs`
- `plugins`（基础插件系统）

**许可证**：Apache 2.0 / MIT

#### 2.4.2 `packages/enterprise` - 企业版（闭源）

**额外包**：
- `enterprise/audit` - 审计日志
- `enterprise/collaboration` - 团队协作
- `enterprise/cloud` - 云存储集成
- `enterprise/compliance` - 合规检查
- `enterprise/analytics` - 高级分析

**通过 LicenseHook 启用**：
```cangjie
package codelin.enterprise

public class LicenseHook {
    public func checkFeature(feature: String): Bool
    public func enforceQuota(quota: Quota): Result<Unit, QuotaError>
}
```

---

## 3. 整体架构设计

### 3.1 分层架构图

```
┌─────────────────────────────────────────────────────────────────────┐
│ Experience Layer (Surfaces)                                         │
│ ┌──────────┐  ┌──────────┐  ┌──────────┐  ┌──────────┐           │
│ │   CLI    │  │   TUI    │  │   IDE    │  │   API    │           │
│ │ Surface  │  │ Surface   │  │  Bridge   │  │ Gateway  │           │
│ └────┬─────┘  └────┬─────┘  └────┬─────┘  └────┬─────┘           │
│      └──────────────┴──────────────┴──────────────┘                │
│                    │ IInteractionPort + StreamingChannel            │
└────────────────────┼────────────────────────────────────────────────┘
                     │
┌────────────────────▼────────────────────────────────────────────────┐
│ Service Layer (Functional Services)                                 │
│ ┌──────────────────────┐  ┌──────────────────────┐                 │
│ │ ConversationPipeline │  │ ContextPipeline      │                 │
│ │ - Round Snapshot     │  │ - Budget Management  │                 │
│ │ - Topic Evolution    │  │ - File Graph         │                 │
│ │ - Feedback Digest    │  │ - Knowledge Packs    │                 │
│ └──────────────────────┘  └──────────────────────┘                 │
│ ┌──────────────────────┐  ┌──────────────────────┐                 │
│ │ MissionGraph         │  │ ToolBusService       │                 │
│ │ Orchestrator         │  │ - ABI Validation     │                 │
│ │ - Multi-Agent Flow   │  │ - MCP Bridge         │                 │
│ │ - Task Scheduling    │  │ - Streaming/WS        │                 │
│ └──────────────────────┘  └──────────────────────┘                 │
└────────────────────┬───────────────────────────────────────────────┘
                     │ Kernel Contracts (Result<Option<T>>)
┌────────────────────▼────────────────────────────────────────────────┐
│ Cangjie Kernel (Immutable Core)                                     │
│ ┌──────────────┐  ┌──────────────┐  ┌──────────────┐               │
│ │ ContextKernel│  │ ToolABI     │  │ Functional   │               │
│ │ - Round Snap │  │ Compiler    │  │ Primitives   │               │
│ │ - Memory DAG │  │ - Determinism│ │ - pipe/compose│              │
│ └──────────────┘  └──────────────┘  └──────────────┘               │
│ ┌──────────────┐  ┌──────────────┐  ┌──────────────┐               │
│ │ IO/LSP Ports │  │ Data Fabric  │  │ StateManager │               │
│ │ (Event-driven)│ │ - DedupIndex │  │ - IOEffect   │               │
│ └──────────────┘  └──────────────┘  └──────────────┘               │
└────────────────────┬───────────────────────────────────────────────┘
                     │ Plugin ABI
┌────────────────────▼────────────────────────────────────────────────┐
│ Extension Plane (Plugins & Packs)                                   │
│ ┌──────────────┐  ┌──────────────┐  ┌──────────────┐               │
│ │ Tool Packs   │  │ Agent Packs  │  │ Knowledge    │               │
│ │ - FS/LSP/Git │  │ - CodeBuff   │  │ Packs        │               │
│ │ - MCP Bridge │  │ - Custom     │  │ - Code/Docs  │               │
│ └──────────────┘  └──────────────┘  └──────────────┘               │
│ ┌──────────────┐  ┌──────────────┐                                 │
│ │ Enterprise   │  │ Community    │                                 │
│ │ Packs         │  │ Extensions   │                                 │
│ └──────────────┘  └──────────────┘                                 │
└─────────────────────────────────────────────────────────────────────┘
```

### 3.2 数据流设计

#### 3.2.1 用户请求处理流程

```
User Input (CLI/TUI/IDE)
    ↓
Surface Layer (parse & validate)
    ↓
Service Layer: ConversationPipeline
    ├─→ ContextPipeline (build context from conversation)
    │       ├─→ ContextKernel (Round Snapshot + Memory DAG)
    │       └─→ KnowledgePack (domain-specific extraction)
    │
    ├─→ MissionGraph Orchestrator
    │       ├─→ AgentMesh (spawn agents)
    │       └─→ ToolBus (dispatch tools)
    │
    └─→ StateManager (apply transitions, collect IOEffects)
            ↓
        Execute IOEffects (side effects isolated)
            ↓
        Update State (immutable)
            ↓
        Render Response (Surface)
```

#### 3.2.2 函数式状态演进

```cangjie
// 纯函数状态转换
let (newState, effects) = evolveState(
    currentState: AppState,
    event: UserEvent
): (AppState, Array<IOEffect>)

// 副作用隔离执行
for (effect in effects) {
    executeEffect(effect)  // IO 操作
}

// 状态更新（不可变）
appState = newState
```

### 3.3 插件系统架构

```
Plugin Registry
    ├─→ Tool Plugins
    │       ├─→ FSToolset Plugin
    │       ├─→ LSPToolset Plugin
    │       └─→ MCP Bridge Plugin
    │
    ├─→ Agent Plugins
    │       ├─→ CodeBuff Agent Adapter
    │       ├─→ Claude Agent SDK Adapter
    │       └─→ Custom Agent
    │
    ├─→ Surface Plugins
    │       ├─→ IDE Bridge (VSCode/Cursor)
    │       └─→ Web UI Plugin
    │
    └─→ Knowledge Pack Plugins
            ├─→ Code Pack
            ├─→ Docs Pack
            └─→ Custom Domain Pack
```

---

## 4. 仓颉函数式特性应用

### 4.1 不可变状态

```cangjie
// ✅ 使用 let 声明不可变状态
public struct ContextState {
    public let fileCache: HashMap<Path, FileContext>  // 不可变
    public let tokenCount: Int64
    public let maxTokens: Int64
}

// ✅ 状态转换返回新状态
public func addFile(
    state: ContextState,
    path: Path,
    content: String
): ContextState {
    let newCache = state.fileCache.clone()  // 克隆而非修改
    newCache[path] = FileContext(path, content)
    return ContextState(
        fileCache: newCache,
        tokenCount: state.tokenCount + estimateTokens(content),
        maxTokens: state.maxTokens
    )
}
```

### 4.2 纯函数与组合

```cangjie
// ✅ 纯函数：无副作用
public func rankFilesByRelevance(
    query: String,
    files: Array<FileContext>
): Array<FileContext> {
    return files
        .map({ file => (file, calculateBM25(query, file)) })
        .sort({ (a, b) => compare(b.1, a.1) })
        .map({ (file, _) => file })
}

// ✅ 函数组合
let buildContext = compose(
    { files => formatContext(files) },
    { files => filterByBudget(files, budget) },
    { query => rankFilesByRelevance(query, allFiles) }
)
```

### 4.3 模式匹配与 Option/Result

```cangjie
// ✅ 使用 Option 处理可选值
public func getFile(path: Path): Option<FileContext> {
    return this.fileCache.get(path)
}

match (getFile(path)) {
    case Some(file) => processFile(file)
    case None => LogUtils.warn("File not found: ${path}")
}

// ✅ 使用 Result 处理错误
public func loadPlugin(path: Path): Result<Plugin, LoadError> {
    if (File.exists(path)) {
        return Ok(parsePlugin(path))
    } else {
        return Err(LoadError("Plugin not found: ${path}"))
    }
}
```

### 4.4 高阶函数与集合操作

```cangjie
// ✅ 使用 map/filter/fold 等高阶函数
let relevantFiles = allFiles
    .filter({ file => file.relevanceScore > 0.5 })
    .map({ file => compressFile(file, budget) })
    .fold(ArrayList<FileContext>(), { (acc, file) => 
        acc.add(file)
        acc
    })
```

---

## 5. 重复率治理策略

### 5.1 当前重复率分析

根据 `reports/jscpd/jscpd-report.json`：

| 文件 | 重复率 | 问题 |
| --- | --- | --- |
| `console_exporter.cj` | 80.65% | 序列化逻辑重复 |
| `http_json_exporter.cj` | 23.17% | HTTP 请求逻辑重复 |
| `head.cj` | 31.93% | 文件读取逻辑重复 |

### 5.2 治理方案

#### 5.2.1 Telemetry Exporter 抽象

```cangjie
// packages/core/src/telemetry/exporter_base.cj
public abstract class TelemetryExporter {
    // 公共序列化逻辑
    protected func serialize(event: TelemetryEvent): String {
        // 统一序列化实现
    }
    
    // 公共重试逻辑
    protected func executeWithRetry(
        operation: () -> Result<T, Error>,
        maxRetries: Int64 = 3
    ): Result<T, Error> {
        // 统一重试实现
    }
    
    // 子类实现具体导出逻辑
    public abstract func export(event: TelemetryEvent): Result<Unit, ExportError>
}

// packages/core/src/telemetry/console_exporter.cj
public class ConsoleExporter extends TelemetryExporter {
    public func export(event: TelemetryEvent): Result<Unit, ExportError> {
        let serialized = this.serialize(event)
        PrintUtils.printLine(serialized)
        return Ok(())
    }
}

// packages/core/src/telemetry/http_json_exporter.cj
public class HTTPJsonExporter extends TelemetryExporter {
    public func export(event: TelemetryEvent): Result<Unit, ExportError> {
        let serialized = this.serialize(event)
        return this.executeWithRetry({
            => http.post(this.endpoint, serialized)
        })
    }
}
```

#### 5.2.2 FileOp Trait 抽象

```cangjie
// packages/tools/src/fs/file_op.cj
public interface FileOp<T> {
    func execute(path: Path, params: T): Result<String, FileError>
    func validate(path: Path): Result<Unit, ValidationError>
}

// packages/tools/src/fs/head.cj
public class HeadOp implements FileOp<HeadParams> {
    public func execute(path: Path, params: HeadParams): Result<String, FileError> {
        return this.readFileLines(path, params.lines)
    }
}

// packages/tools/src/fs/tail.cj
public class TailOp implements FileOp<TailParams> {
    public func execute(path: Path, params: TailParams): Result<String, FileError> {
        return this.readFileLinesFromEnd(path, params.lines)
    }
}
```

#### 5.2.3 Functional Template Library (FTL)

```cangjie
// packages/core/src/functional/template.cj
public func withLogging<T>(
    operation: () -> Result<T, Error>,
    context: String
): Result<T, Error> {
    LogUtils.debug("Starting: ${context}")
    match (operation()) {
        case Ok(value) => {
            LogUtils.debug("Success: ${context}")
            Ok(value)
        }
        case Err(error) => {
            LogUtils.error("Failed: ${context}, error: ${error.message}")
            Err(error)
        }
    }
}

public func withRetry<T>(
    operation: () -> Result<T, Error>,
    maxRetries: Int64 = 3
): Result<T, Error> {
    // 统一重试逻辑
}
```

### 5.3 目标指标

- **总体重复率**：从当前平均 40%+ 降至 <15%
- **单文件重复率**：<20%（当前最高 80.65%）
- **CI 阻断阈值**：新增代码重复率 >5% 时阻断合并

---

## 6. 插件系统设计

### 6.1 Plugin Manifest Schema

```toml
# plugin.toml
[plugin]
name = "my-custom-tool"
version = "1.0.0"
description = "Custom tool plugin"
author = "Your Name"
license = "MIT"

[dependencies]
codelin.core = "1.0.0"
codelin.tools = "1.0.0"

[[tools]]
name = "my_tool"
description = "My custom tool"
parameters = [
    { name = "input", type = "String", required = true }
]

[[agents]]
id = "my-agent"
displayName = "My Agent"
model = "anthropic/claude-sonnet-4"
systemPrompt = "You are a helpful assistant."

[[hooks]]
type = "before_tool_execution"
handler = "my_hook_handler"
```

### 6.2 插件加载流程

```
Plugin Discovery
    ↓
Load plugin.toml
    ↓
Validate Schema
    ↓
Load Dependencies
    ↓
Register Tools/Agents/Hooks
    ↓
Activate Plugin
```

### 6.3 插件接口

```cangjie
// packages/plugins/src/interfaces/plugin.cj
public interface Plugin {
    func getName(): String
    func getVersion(): String
    func initialize(context: PluginContext): Result<Unit, PluginError>
    func getTools(): Array<ToolDefinition>
    func getAgents(): Array<AgentDefinition>
    func getHooks(): Array<HookDefinition>
}
```

---

## 7. 跨领域扩展设计

### 7.1 KnowledgePack 接口

```cangjie
// packages/context/src/knowledge/knowledge_pack.cj
public interface KnowledgePack {
    func extractContext(
        state: ContextState,
        query: String
    ): Array<FileContext>
    
    func scoreRelevance(
        query: String,
        file: FileContext
    ): Float64
    
    func renderSummary(
        context: Array<FileContext>
    ): String
    
    func getDomain(): String  // "code", "docs", "data", etc.
}
```

### 7.2 领域包示例

#### 7.2.1 代码域（已有）

```cangjie
package codelin.knowledge.code

public class CodeKnowledgePack implements KnowledgePack {
    public func extractContext(...): Array<FileContext> {
        // LSP 符号分析、依赖图、BM25 评分
    }
}
```

#### 7.2.2 文档域

```cangjie
package codelin.knowledge.docs

public class DocsKnowledgePack implements KnowledgePack {
    public func extractContext(...): Array<FileContext> {
        // Markdown 解析、章节提取、交叉引用
    }
}
```

#### 7.2.3 数据域

```cangjie
package codelin.knowledge.data

public class DataKnowledgePack implements KnowledgePack {
    public func extractContext(...): Array<FileContext> {
        // CSV/JSON 解析、数据模式提取、统计分析
    }
}
```

---

## 8. 实施路线图（14 周）

### Phase 0: 基础准备（Week 1-2）

**目标**：建立包结构、重复率基线、工具链

**任务**：
1. 创建 `packages/` 目录结构
2. 为每个包创建 `cjpm.toml`
3. 实现重复率基线脚本（`scripts/dedup-baseline.cj`）
4. 建立 CI 重复率检查（阈值 5%）
5. 迁移 `src/core/utils/functional_utils.cj` 到 `packages/core`

**交付物**：
- `packages/` 目录结构
- `dup-baseline.json`
- CI 配置更新

### Phase 1: 核心运行时（Week 3-4）

**目标**：实现函数式运行时核心

**任务**：
1. 实现 `packages/core/runtime/state_manager.cj`
2. 实现 `packages/core/effects/io_effect.cj`
3. 实现 `packages/core/functional/pipeline.cj`
4. 迁移现有 `functional_utils.cj` 到新包结构
5. 编写单元测试

**交付物**：
- `packages/core` 完整实现
- 单元测试覆盖率 >80%

### Phase 2: 上下文平台化（Week 5-7）

**目标**：重构 ContextEngine 为函数式平台

**任务**：
1. 设计 `ContextState` 不可变结构
2. 实现 `ContextKernel 2.1`（Round Snapshot + Memory DAG）
3. 迁移 `ConversationManager` 到 `packages/context`
4. 实现 `KnowledgePack` 接口
5. 实现 `CodeKnowledgePack`
6. 重构 `buildContextFromConversation` 为纯函数

**交付物**：
- `packages/context` 完整实现
- ContextEngine 函数式重构完成

### Phase 3: 代理编排（Week 8-9）

**目标**：实现多代理编排系统

**任务**：
1. 设计 `AgentDefinition` Schema（兼容 Claude Agent SDK）
2. 实现 `AgentMesh` 多代理调度
3. 实现 `MissionGraph` 任务图编排
4. 迁移现有 SubAgent 到新架构
5. 实现 CodeBuff Agent 适配器

**交付物**：
- `packages/agents` 完整实现
- AgentDefinition JSON Schema
- CodeBuff 兼容层

### Phase 4: 工具总线（Week 10-11）

**目标**：实现 ToolBus 和 ABI Compiler

**任务**：
1. 实现 `ToolBus` 核心
2. 实现 `ToolABICompiler`（解析 @toolset 注释）
3. 迁移 FSToolset/LSPToolset 到插件架构
4. 实现 MCP Bridge
5. 实现工具重复率治理（FileOp Trait）

**交付物**：
- `packages/tools` 完整实现
- Tool ABI Compiler
- 工具重复率 <15%

### Phase 5: 体验层与插件（Week 12-13）

**目标**：重构 Surface 层，实现插件系统

**任务**：
1. 重构 CLI Surface 使用新架构
2. 实现 TUI Surface（Cursor Augment 风格）
3. 实现 IDE Bridge（VSCode/Cursor）
4. 实现 Plugin Loader 和 Registry
5. 实现 Plugin Manifest Schema
6. 迁移现有功能到插件

**交付物**：
- `packages/surfaces` 完整实现
- `packages/plugins` 完整实现
- 插件示例

### Phase 6: 平台化与优化（Week 14）

**目标**：开源/闭源分层、文档、优化

**任务**：
1. 创建 `packages/oss` 和 `packages/enterprise` 配置
2. 实现 LicenseHook
3. 完成重复率治理（目标 <15%）
4. 编写平台文档
5. 性能优化

**交付物**：
- 开源/闭源包配置
- 完整文档
- 性能报告

---

## 9. 验证矩阵

| 验证项 | 目标 | 工具/测试 |
| --- | --- | --- |
| 包依赖正确性 | 无循环依赖 | `scripts/check-deps.cj` |
| 函数式边界 | Adapter 不依赖 Kernel 实现 | `scripts/check-functional-boundary.cj` |
| Tool ABI Determinism | 任意加载顺序一致 | `tests/tool_abi_determinism.cj` |
| 重复率 | 总体 <15%，单文件 <20% | `scripts/dedup-check.cj` |
| 状态不可变性 | 所有 State 使用 let | `scripts/check-immutability.cj` |
| 插件加载 | 所有插件可加载 | `tests/plugin_loader_spec.cj` |
| 跨领域扩展 | 知识包可替换 | `tests/knowledge_pack_spec.cj` |

---

## 10. 总结

### 10.1 核心改进

1. **包架构**：从单体到多包工作区，清晰分层
2. **函数式**：全面应用仓颉函数式特性
3. **平台化**：插件系统、开源/闭源双轨
4. **重复率**：从 40%+ 降至 <15%
5. **扩展性**：跨领域知识包、插件生态

### 10.2 技术亮点

- **仓颉函数式内核**：不可变状态 + 纯函数 + IOEffect
- **CodeBuff 兼容**：AgentDefinition Schema、多代理编排
- **Claude Code 体验**：对话感知上下文、Round Snapshot
- **Cursor Augment**：增量可视化、IDE 桥接
- **Claude Agent SDK**：Tool Schema、Memory、Workflow Hooks

### 10.3 下一步行动

1. **立即开始**：创建 `packages/` 目录结构
2. **本周完成**：Phase 0 基础准备
3. **两周内**：完成核心运行时（Phase 1）
4. **持续监控**：重复率、函数式边界、包依赖

---

**本方案将 CodeLin 从单一 CLI 应用升级为仓颉函数式多代理平台，为开源社区与商业化奠定坚实基础。**

