# Codelin 整体架构设计文档

**项目**: Codelin - 智能编码助手  
**技术栈**: 仓颉语言 (Cangjie) - 华为自研编程语言  
**版本**: v1.0.0  
**日期**: 2025-11-18

---

## 📋 目录

1. [项目概览](#项目概览)
2. [整体架构](#整体架构)
3. [核心技术栈](#核心技术栈)
4. [模块详解](#模块详解)
5. [数据流](#数据流)
6. [关键设计](#关键设计)

---

## 🎯 项目概览

### 项目定位
Codelin 是一个**命令行智能编码助手**，基于华为仓颉语言开发，集成了大语言模型能力，提供智能代码生成、分析、重构等功能。

### 核心特性
- ✅ **多模型支持** - 支持主模型、快速模型、降级模型
- ✅ **智能工具链** - 10+ 专业工具集（LSP、Git、文件系统等）
- ✅ **会话管理** - Claude Code 风格的会话系统
- ✅ **MCP 集成** - Model Context Protocol 标准支持
- ✅ **多语言支持** - 仓颉语言特化 + 通用语言支持
- ✅ **交互式终端** - 现代化 CLI 用户体验

### 技术亮点
- **华为仓颉语言** - 新一代静态类型编程语言
- **LSP 深度集成** - 语义级代码理解
- **Agent 架构** - 主 Agent + 7 个子 Agent 协作
- **流式处理** - 异步执行和实时反馈

---

## 🏗️ 整体架构

### 五层架构设计

```
┌─────────────────────────────────────────────────────────────────────┐
│                         Presentation Layer                          │
│                           展示层 (UI)                                │
├─────────────────────────────────────────────────────────────────────┤
│  • Terminal UI (src/io/)                                            │
│    - PrintUtils: 输出管理 (状态指示器、Markdown渲染)                   │
│    - InputUtils: 输入管理 (Readline、自动补全)                         │
│    - Colors: 颜色系统 (3套主题、256色支持)                              │
│  • Interactive Controller (Ctrl+C/I/P 交互控制)                      │
└─────────────────────────────────────────────────────────────────────┘
                                  ↓↑
┌─────────────────────────────────────────────────────────────────────┐
│                        Application Layer                            │
│                           应用层                                      │
├─────────────────────────────────────────────────────────────────────┤
│  • CliApp (src/app/)                                                │
│    - 主应用入口、生命周期管理                                           │
│    - Interactive / Non-Interactive 模式                             │
│  • ProcessInput: 输入处理和命令分发                                    │
│  • InteractiveController: 实时交互控制                                │
│  • ModelFallback: 模型降级机制                                        │
└─────────────────────────────────────────────────────────────────────┘
                                  ↓↑
┌─────────────────────────────────────────────────────────────────────┐
│                          Agent Layer                                │
│                           智能体层                                    │
├─────────────────────────────────────────────────────────────────────┤
│  主 Agent:                                                           │
│  • CangjieCodeAgent - 仓颉语言专用Agent                               │
│  • GeneralCodeAgent - 通用语言Agent                                  │
│                                                                      │
│  7个子 Agent (Subagents):                                            │
│  ├─ ExplorerAgent - 代码探索                                         │
│  ├─ PlannerAgent - 任务规划                                          │
│  ├─ EditorAgent - 代码编辑                                           │
│  ├─ ReviewerAgent - 代码审查                                         │
│  ├─ TestGeneratorAgent - 测试生成                                    │
│  ├─ RefactoringAgent - 重构                                          │
│  └─ CodeAnalyzerAgent - 代码分析                                     │
│                                                                      │
│  辅助 Agent:                                                         │
│  • Compactor - 上下文压缩                                             │
│  • SessionAutoNamer - 会话自动命名                                    │
└─────────────────────────────────────────────────────────────────────┘
                                  ↓↑
┌─────────────────────────────────────────────────────────────────────┐
│                          Service Layer                              │
│                           服务层                                      │
├─────────────────────────────────────────────────────────────────────┤
│  工具集 (src/core/tools/):                                           │
│  ├─ LSPToolset - 语言服务器协议 (语义分析、重构)                        │
│  ├─ FSToolset - 文件系统 (读写、搜索、缓存)                             │
│  ├─ GitToolset - Git操作 (提交、分支、diff)                            │
│  ├─ CangjieToolset - 仓颉工具 (编译、测试、文档)                        │
│  ├─ CodeQualityToolset - 代码质量 (检查、格式化)                       │
│  ├─ HarmonyToolset - 鸿蒙开发工具                                     │
│  ├─ SearchToolset - 代码搜索                                          │
│  ├─ ShellToolset - Shell命令执行                                     │
│  ├─ PlanToolset - 任务计划管理                                        │
│  └─ TavilyTool - Web搜索                                            │
│                                                                      │
│  上下文服务 (src/core/context/):                                     │
│  • ContextEngine - 上下文管理引擎 (18个模块)                           │
│                                                                      │
│  会话服务 (src/core/conversation/):                                  │
│  • ConversationManager - 会话管理                                    │
│  • SessionMetadataManager - 会话元数据                                │
│  • SessionSelector - 交互式会话选择器                                  │
│                                                                      │
│  MCP 服务 (src/core/mcp/):                                           │
│  • MCPConfigManager - MCP配置管理                                    │
│  • MCPWrapper - MCP协议封装                                          │
└─────────────────────────────────────────────────────────────────────┘
                                  ↓↑
┌─────────────────────────────────────────────────────────────────────┐
│                      Infrastructure Layer                           │
│                           基础设施层                                  │
├─────────────────────────────────────────────────────────────────────┤
│  外部集成:                                                            │
│  • 华为大模型服务MaaS                                         │
│                                                                      │
│  • LSP Client (src/lsp/) - 语言服务器客户端                           │
│    - JSON-RPC 通信                                                  │
│    - 语义分析能力                                                      │
│                                                                      │
│  配置管理 (src/core/config/):                                        │
│  • CliConfig - 全局配置                                              │
│  • CliSettingManager - 设置管理                                      │
│  • ModelTokenLimits - Token 限制                                    │
│                                                                      │
│  实用工具 (src/utils/):                                              │
│  • GitUtils, FileUtils, ApiClient, TomlParser                      │
│                                                                      │
│  遥测 (src/telemetry/):                                              │
│  • DataCollector - 使用数据收集                                       │
└─────────────────────────────────────────────────────────────────────┘
```

---

## 🔧 核心技术栈

### 华为技术生态

#### 1. 仓颉编程语言 (Cangjie)

**语言特性**:
```cangjie
// 静态类型 + 类型推导
let name: String = "codelin"
let auto = 42  // 自动推导为 Int64

// 泛型支持
class ArrayList<T> { ... }

// 模式匹配
match (value) {
    case Some(x) => print(x)
    case None => print("empty")
}

// 函数式编程
array |> map { x => x * 2 } |> filter { x => x > 10 }

// 并发原语
spawn { ... }  // 协程
Channel<T>     // 通道
Mutex          // 互斥锁
AtomicBool     // 原子操作
```

**与 Codelin 的深度集成**:
- `CangjieCodeAgent` - 专门针对仓颉语言优化的 Agent
- `CangjieToolset` - 仓颉工具链集成（编译、测试、格式化）
- `CangjieFixer` - 编译错误自动修复
- `CangjieDocRetriever` - 仓颉文档检索

#### 2. Magic Framework (华为 AI Agent 框架)

**核心组件**:
```cangjie
import magic.agent.ConversationAgent
import magic.core.message.Conversation
import magic.core.model.ChatModel
import magic.model.ModelManager
import magic.dsl.*  // Agent DSL
```

**功能特性**:
- **Agent 抽象** - 统一的智能体接口
- **工具系统** - Tool / Toolset 标准化
- **对话管理** - Conversation 历史管理
- **模型管理** - 多模型支持和切换

**在 Codelin 中的使用**:
```cangjie
class CangjieCodeAgent <: ConversationAgent {
    init() {
        super.init(
            name: "Cangjie Code Expert",
            instruction: "专业的仓颉语言助手...",
            model: CliConfig.model,
            tools: [
                LSPToolset(),
                CangjieToolset(),
                FSToolset(),
                // ...
            ]
        )
    }
}
```

#### 3. LSP (Language Server Protocol)

**深度集成** (src/lsp/):
- `LspClient` - LSP 客户端实现
- `JsonRpcClient` - JSON-RPC 2.0 通信
- `SemanticInfo` - 语义信息提取

**支持的 LSP 能力**:
```
• textDocument/definition       - 跳转到定义
• textDocument/references       - 查找引用
• textDocument/hover            - 悬停信息
• textDocument/rename           - 重命名
• textDocument/codeAction       - 代码操作
• textDocument/semanticTokens   - 语义高亮
```

**LSPToolset 工具集** (37K 代码):
- `semanticSearch` - 语义级代码搜索
- `findReferences` - 引用查找
- `getTypeHierarchy` - 类型层次
- `extractSymbols` - 符号提取

#### 4. HarmonyOS 生态支持

**HarmonyToolset** (15K 代码):
- 鸿蒙应用开发支持
- ArkTS 语言支持
- DevEco Studio 集成
- 鸿蒙 API 文档检索

---

## 📦 模块详解

### 核心模块划分

```
src/
├── app/                    # 应用层 (5个文件)
│   ├── cli_app.cj         # 主应用类 (240行)
│   ├── process_input.cj   # 输入处理 (350行)
│   ├── interactive_controller.cj  # 交互控制 (209行)
│   ├── model_fallback.cj  # 模型降级
│   └── init_magic_file.cj # 初始化向导
│
├── core/                   # 核心业务层
│   ├── agents/            # Agent 层 (14个文件)
│   │   ├── cangjie_code_agent.cj     # 仓颉 Agent (19K)
│   │   ├── general_code_agent.cj     # 通用 Agent (31K)
│   │   ├── cangjie_fixer.cj          # 编译错误修复
│   │   ├── code_analyzer.cj          # 代码分析器
│   │   ├── code_designer.cj          # 设计规划
│   │   ├── context_compactor.cj      # 上下文压缩
│   │   └── subagents/                # 7个子Agent
│   │       ├── explorer_agent.cj     # 代码探索
│   │       ├── planner_agent.cj      # 任务规划
│   │       ├── editor_agent.cj       # 代码编辑
│   │       ├── reviewer_agent.cj     # 代码审查
│   │       ├── test_generator_agent.cj  # 测试生成
│   │       ├── refactoring_agent.cj  # 重构
│   │       └── (未使用的Agent...)
│   │
│   ├── tools/             # 工具集层 (20个文件)
│   │   ├── lsp_toolset.cj            # LSP工具 (37K) ⭐核心
│   │   ├── fs_toolset.cj             # 文件系统 (26K) ⭐核心
│   │   ├── code_quality_toolset.cj   # 代码质量 (32K)
│   │   ├── git_toolset.cj            # Git工具 (18K)
│   │   ├── cangjie_toolset.cj        # 仓颉工具 (7K)
│   │   ├── harmony_toolset.cj        # 鸿蒙工具 (15K)
│   │   ├── search_toolset.cj         # 搜索工具
│   │   ├── shell_toolset.cj          # Shell执行
│   │   ├── plan_toolset.cj           # 计划管理
│   │   ├── tavily_tool.cj            # Web搜索
│   │   ├── cj_utils/                 # 仓颉工具库
│   │   │   ├── cangjie_doc_retriever.cj
│   │   │   ├── cangjie_doc_agent.cj
│   │   │   └── code_snippet_builder.cj
│   │   └── fs_utils/                 # 文件系统工具库
│   │       ├── grep.cj               # 文本搜索
│   │       ├── file_tree.cj          # 目录树
│   │       └── ...
│   │
│   ├── context/           # 上下文管理 (18个模块)
│   │   └── context_engine.cj         # 上下文引擎
│   │
│   ├── conversation/      # 会话管理 (5个文件)
│   │   ├── conversation_manager.cj   # 会话管理器 (545行)
│   │   ├── session_metadata.cj       # 会话元数据
│   │   ├── session_selector.cj       # 交互式选择器
│   │   ├── session_auto_namer.cj     # 自动命名
│   │   └── time_formatter.cj         # 时间格式化
│   │
│   ├── mcp/               # MCP 集成 (2个文件)
│   │   ├── mcp_config_manager.cj     # MCP配置
│   │   └── wrapper.cj                # MCP封装
│   │
│   ├── config/            # 配置管理 (3个文件)
│   │   ├── cli_config.cj             # 全局配置
│   │   ├── cli_setting.cj            # 用户设置
│   │   └── model_token_limits.cj     # Token限制
│   │
│   ├── commands/          # 命令系统
│   │   └── custom_command_manager.cj # 自定义命令
│   │
│   └── types/             # 类型定义
│       └── session_display.cj        # 会话显示信息
│
├── io/                    # IO 层 (7个文件)
│   ├── print_utils.cj    # 输出工具 (338行) ⭐核心
│   ├── input_utils.cj    # 输入工具 (252行) ⭐核心
│   ├── readline.cj       # 命令行读取 (663行)
│   ├── colors.cj         # 颜色系统 (197行)
│   ├── render_markdown.cj # Markdown渲染 (129行)
│   ├── wrap_box.cj       # 边框绘制 (329行)
│   └── codelin_cli_logo.cj # Logo显示
│
├── lsp/                   # LSP 客户端 (5个文件)
│   ├── lsp_client.cj     # LSP客户端
│   ├── json_rpc_client.cj # JSON-RPC
│   ├── semantic_info.cj  # 语义信息
│   ├── types.cj          # LSP类型
│   └── extract_fields.cj # 字段提取
│
├── utils/                 # 实用工具 (10个文件)
│   ├── git_utils.cj      # Git工具
│   ├── file_utils.cj     # 文件工具
│   ├── api_client.cj     # API客户端
│   ├── toml_parser.cj    # TOML解析
│   └── ...
│
├── telemetry/            # 遥测 (1个文件)
│   └── data_collector.cj # 数据收集
│
├── main.cj               # 程序入口 (81行)
├── parse_args.cj         # 参数解析
└── guideline.cj          # 使用指南
```

### 关键模块统计

| 模块 | 文件数 | 代码量 | 核心功能 |
|------|--------|--------|---------|
| **tools/** | 20+ | ~150K | 10+工具集，LSP/FS是核心 |
| **agents/** | 14 | ~100K | 2个主Agent + 7个子Agent |
| **io/** | 7 | ~2K | 终端UI全部实现 |
| **conversation/** | 5 | ~1K | Claude Code风格会话管理 |
| **lsp/** | 5 | ~10K | 语义级代码分析 |
| **context/** | 18 | ~5K | 上下文管理引擎 |

---

## 🔄 数据流架构

### 1. 用户输入流

```
用户输入
  ↓
[Readline] 读取一行
  ├─ 历史记录管理
  ├─ 自动补全 (命令/文件/@符号)
  ├─ 多行编辑
  └─ Ctrl+C 取消
  ↓
[InputUtils.getUserInput()]
  ├─ 构建提示符 (Session信息)
  ├─ 原始模式控制
  └─ ESC检测
  ↓
[ProcessInput]
  ├─ 命令检测 (/help, /new, /sessions...)
  ├─ Shell命令检测 (!command)
  ├─ 文件引用检测 (@file)
  └─ 自然语言请求
  ↓
[CliApp.processInput()]
  ├─ 命令分发
  ├─ Agent调用
  └─ 响应处理
```

### 2. Agent 执行流

```
用户请求
  ↓
[CliApp] 选择Agent
  ├─ language == "cangjie" → CangjieCodeAgent
  ├─ language == "general" → GeneralCodeAgent
  └─ language == "unknown" → 自动检测
  ↓
[Agent.asyncChat()] 异步执行
  ├─ 构建上下文 (ContextEngine)
  ├─ 添加用户规则 (CODELIN.md)
  ├─ 注入工具集
  └─ 调用LLM
  ↓
[工具调用循环]
  ├─ LLM返回工具调用
  ├─ 执行工具 (LSP/FS/Git...)
  ├─ 工具结果反馈给LLM
  └─ 继续直到完成
  ↓
[最终响应]
  ├─ Markdown格式化
  ├─ 代码高亮
  └─ 输出到终端
```

### 3. 工具执行流

```
Agent请求工具
  ↓
[Toolset.execute()]
  ↓
示例: LSPToolset.semanticSearch()
  ├─ 启动LSP Server
  ├─ 建立JSON-RPC连接
  ├─ 发送textDocument/semanticTokens
  ├─ 解析语义信息
  ├─ 执行语义级搜索
  └─ 返回结果
  ↓
示例: FSToolset.readFile()
  ├─ 路径解析
  ├─ 缓存检查
  ├─ 文件读取
  ├─ 编码检测
  └─ 返回内容
  ↓
结果返回给Agent
```

### 4. 会话管理流

```
启动 Codelin
  ↓
[ConversationManager.autoResume()]
  ├─ 读取 last-conversation.history
  ├─ 加载会话历史
  └─ 恢复上下文
  ↓
交互过程
  ├─ 每轮对话添加到 Conversation
  ├─ 更新 SessionMetadata
  │   ├─ messageCount++
  │   ├─ fileCount (自动统计)
  │   └─ lastModified (更新时间)
  ├─ 首次响应后自动命名
  │   └─ SessionAutoNamer.generate()
  └─ 自动压缩 (Compactor)
  ↓
退出时
  └─ ConversationManager.autoSave()
      └─ 保存到 .codelin/conversation-history/
```

---

## 🎨 关键设计模式

### 1. Agent 协作模式

**主Agent + 子Agent 架构**:
```
CangjieCodeAgent (主)
  ├─ 负责总体任务规划
  ├─ 调度子Agent完成子任务
  └─ 整合结果返回用户
  ↓
调用子Agent (按需)
  ├─ ExplorerAgent → 探索代码库结构
  ├─ PlannerAgent → 分解任务为步骤
  ├─ EditorAgent → 执行代码编辑
  ├─ ReviewerAgent → 代码质量检查
  ├─ TestGeneratorAgent → 生成单元测试
  └─ RefactoringAgent → 重构优化
```

**优势**:
- 职责分离，每个Agent专注特定任务
- 可并行执行多个子Agent
- 易于扩展新的子Agent

### 2. 工具集抽象

**Toolset 模式**:
```cangjie
// 基类 (来自Magic Framework)
class Toolset {
    abstract func getTools(): Array<Tool>
}

// 实现
class LSPToolset <: Toolset {
    public override func getTools(): Array<Tool> {
        [
            Tool(name: "semanticSearch", ...),
            Tool(name: "findReferences", ...),
            Tool(name: "getDefinition", ...),
        ]
    }
}
```

**优势**:
- 统一的工具接口
- 工具组合和复用
- 易于测试和模拟

### 3. 缓存策略

**多级缓存**:
```
L1: 内存缓存 (HashMap)
  ├─ FSToolset: 文件内容缓存
  ├─ LSPToolset: 语义信息缓存
  └─ GitToolset: Git信息缓存

L2: 磁盘缓存
  └─ .codelin/cache/

缓存失效策略:
  ├─ 文件修改时间检测
  ├─ Git commit 变化检测
  └─ 手动清理 (/clear)
```

### 4. 异步执行模式

**AsyncAgentResponse**:
```cangjie
let response = agent.asyncChat(request)

// 可选: 实时获取进度
while (!response.isDone) {
    if (let Some(chunk) <- response.nextChunk()) {
        print(chunk)  // 流式输出
    }
}

// 最终结果
let finalResult = response.content
```

### 5. 错误处理与降级

**模型降级机制**:
```
主模型 (claude-3-5-sonnet-20241022)
  ↓ 失败 (429/500/timeout)
快速模型 (claude-3-haiku-20240307)
  ↓ 失败
降级模型列表 [model1, model2, ...]
  ↓ 全部失败
提示用户重试
```

**工具降级**:
```
LSP语义搜索
  ↓ LSP Server 不可用
文本正则搜索 (grep)
  ↓
成功返回结果
```

---

## 🚀 性能优化设计

### 1. 并行处理

**文件批量读取**:
```cangjie
// ParallelFileReader (原型)
class ParallelFileReader {
    // 使用协程并行读取多个文件
    spawn { readFile(file1) }
    spawn { readFile(file2) }
    spawn { readFile(file3) }
    
    // 等待全部完成
    results.await()
}
```

### 2. 增量更新

**LSP 增量同步**:
```
textDocument/didChange (增量)
  ├─ 仅发送变更部分
  ├─ 减少网络传输
  └─ 加快响应速度
```

### 3. 懒加载

**工具集懒初始化**:
```cangjie
class LSPToolset {
    private var lspClient: Option<LspClient> = None
    
    func ensureClient(): LspClient {
        if (lspClient.isNone()) {
            lspClient = LspClient()  // 首次使用时才创建
        }
        return lspClient.getOrThrow()
    }
}
```

---

## 📊 技术栈总览

### 语言与框架
| 技术 | 版本 | 用途 |
|------|------|------|
| **仓颉语言** | 1.0.0 | 核心语言 |
| **Magic Framework** | - | AI Agent框架 |
| **LSP** | 3.17 | 语言服务器协议 |
| **JSON-RPC** | 2.0 | RPC通信 |

### 外部依赖
- **Anthropic Claude API** - LLM模型
- **Tavily API** - Web搜索
- **Git CLI** - 版本控制
- **各语言 LSP Server** - 语义分析

### 数据存储
```
.codelin/
├── codelin.log                    # 日志文件
├── codelinhis                     # 命令历史
├── todo.md                        # 任务计划
├── conversation-history/          # 会话历史
│   ├── last-conversation.history
│   ├── session-1.history
│   └── ...
├── session-metadata/              # 会话元数据
│   ├── last-conversation.json
│   └── ...
└── cache/                         # 缓存目录
```

---

## 💡 创新亮点

### 1. 仓颉语言深度优化
- 专门的 `CangjieCodeAgent`
- 编译错误自动修复 (`CangjieFixer`)
- 仓颉文档智能检索
- 仓颉代码质量检查

### 2. LSP 语义级理解
- 不仅是文本搜索，而是理解代码语义
- 精确的符号查找和重命名
- 类型层次分析
- 代码结构理解

### 3. Claude Code 风格体验
- 会话管理系统 (`/new`, `/sessions`, `/switch`)
- 交互式会话选择器
- 自动会话命名
- 上下文智能压缩

### 4. 多Agent 协作
- 7个专业子Agent分工协作
- 任务自动分解和并行执行
- 结果智能整合

### 5. 鸿蒙生态支持
- `HarmonyToolset` - 鸿蒙开发工具集
- ArkTS 语言支持
- DevEco Studio 集成

---

## 🎯 总结

Codelin 是一个**技术先进、架构清晰、功能完善**的智能编码助手：

### 技术优势
✅ **华为技术栈** - 仓颉语言 + Magic Framework + 鸿蒙支持  
✅ **五层架构** - 清晰的分层设计，易于扩展  
✅ **Agent 协作** - 主Agent + 7个子Agent，智能任务分解  
✅ **LSP 深度集成** - 语义级代码理解，远超文本搜索  
✅ **10+ 工具集** - 涵盖文件、Git、LSP、质量检查等  
✅ **Claude Code 体验** - 现代化CLI交互，会话管理完善  

### 代码质量
✅ **模块化设计** - 150+ 文件，职责清晰  
✅ **类型安全** - 仓颉静态类型保障  
✅ **文档完善** - 详细的注释和文档  
✅ **测试覆盖** - 单元测试和集成测试  

### 创新点
✅ **仓颉语言专家** - 业界首个仓颉AI助手  
✅ **语义理解** - LSP加持的深度代码分析  
✅ **鸿蒙支持** - HarmonyOS开发工具链  
✅ **多模型降级** - 保障服务可用性  

**Codelin = 华为技术栈 × AI Agent × 现代CLI × 语义理解**

---

**文档版本**: v1.0  
**最后更新**: 2025-11-18  
**维护者**: Codelin Team
