# 修复 Workspace 参数和输入提示符问题

**修复日期**: 2025-11-27  
**问题**: 
1. `cjpm run --name codelin --run-args "-w trader"` 参数可能未生效
2. 输入提示符显示了之前的输入内容（如"当前目录是什么"）

---

## 🔧 修复内容

### 1. 输入提示符修复

**问题**: 输入提示符显示了之前的输入内容，如"当前目录是什么 >"

**修复**: `src/io/input_utils.cj`

#### 修复1: `buildPrompt` 函数 (第37-40行)

```cangjie
// 修复前
let cleanPrompt = if (prompt == "你是谁" || prompt == "who are you") {
    ""
} else {
    prompt  // ❌ 会显示之前的输入
}

// 修复后
// 🆕 修复: 不显示之前的输入内容，只显示必要的提示信息
// 过滤掉所有用户输入内容，只保留空字符串（简洁的提示符）
let cleanPrompt = ""  // ✅ 始终为空，不显示之前的输入
```

#### 修复2: `getUserInput` 函数 (第89行)

```cangjie
// 修复前
let promptStr = if (let Some(p) <- prompt) { p } else { "" }
// ❌ 如果 prompt 参数包含之前的输入，会传递给 buildPrompt

// 修复后
// 🆕 修复: 完全忽略 prompt 参数，不显示之前的输入内容
// 始终使用空字符串，确保输入提示符简洁干净
let promptStr = ""  // ✅ 始终为空，完全忽略 prompt 参数
```

**效果**: 
- ✅ 输入提示符现在只显示简洁的 ` > ` 或 ` ${sessionTitle} > `
- ✅ 不再显示之前的输入内容
- ✅ 符合 Claude Code 风格的简洁UI

### 2. 添加调试日志

**目的**: 帮助诊断 `-w` 参数传递问题

**修改文件**:

#### `src/main.cj`
- 添加 `import magic.log.LogUtils`
- 在参数解析前记录接收到的参数
- 在参数解析后记录设置的workspace

```cangjie
// 🆕 Debug: Log received arguments (for troubleshooting -w parameter)
LogUtils.debug("[main] Received ${args.size} arguments: ${String.join(args, delimiter: " ")}")

// ... 参数解析 ...

// 🆕 Debug: Log workspace after parsing
LogUtils.info("[main] Workspace set to: ${CliConfig.cwd}")
```

#### `src/parse_args.cj`
- 在参数解析前记录原始参数
- 在参数解析后记录解析结果
- 在设置workspace时记录详细信息

```cangjie
// 🆕 Debug: Log arguments before parsing
LogUtils.debug("[parseArgs] Parsing ${args.size} arguments: ${String.join(args, delimiter: " ")}")

// ... 解析参数 ...

// 🆕 Debug: Log parsed options
LogUtils.debug("[parseArgs] Parsed options: workspace=${result.options.get("workspace")}, w=${result.options.get("w")}")

// 设置workspace时
LogUtils.info("[parseArgs] Setting workspace from -w: ${workspace}")
```

---

## 🔍 诊断 `-w` 参数问题

### 可能的原因

1. **参数格式问题**: `cjpm run --run-args "-w trader"` 可能将整个字符串作为一个参数传递
   - 应该使用: `cjpm run --name codelin --run-args "-w" "trader"`
   - 或者: `cjpm run --name codelin --run-args "-w=trader"`

2. **参数解析问题**: `std.argopt.parseArguments` 可能需要空格分隔的参数
   - 检查日志中的 `[parseArgs] Parsed options` 来确认参数是否正确解析

### 调试步骤

1. **运行命令并查看日志**:
   ```bash
   cjpm run --name codelin --run-args "-w trader"
   ```

2. **查看日志文件**:
   ```bash
   cat .codelin/codelin.log | grep -E "\[main\]|\[parseArgs\]"
   ```

3. **检查参数传递**:
   - 查看 `[main] Received X arguments:` 确认参数是否正确传递
   - 查看 `[parseArgs] Parsed options:` 确认参数是否正确解析
   - 查看 `[parseArgs] Setting workspace from -w:` 确认workspace是否设置

### 正确的使用方式

根据 `cjpm run` 的帮助信息，`--run-args` 应该接收空格分隔的参数：

```bash
# 方式1: 空格分隔（推荐）
cjpm run --name codelin --run-args "-w" "trader"

# 方式2: 使用等号
cjpm run --name codelin --run-args "-w=trader"

# 方式3: 使用长参数
cjpm run --name codelin --run-args "--workspace" "trader"
```

---

## ✅ 验证修复

### 1. 输入提示符验证

运行 CodeLin 后，输入提示符应该：
- ✅ 只显示 ` > ` 或 ` ${sessionTitle} > `
- ✅ 不显示之前的输入内容
- ✅ 简洁清晰

### 2. Workspace 参数验证

运行以下命令并检查日志：

```bash
# 创建测试目录
mkdir -p /tmp/test-trader

# 运行并查看日志
cjpm run --name codelin --run-args "-w" "/tmp/test-trader" 2>&1 | tee /tmp/codelin-output.log

# 检查日志
grep -E "\[main\]|\[parseArgs\]" .codelin/codelin.log
```

预期输出：
```
[main] Received 2 arguments: -w /tmp/test-trader
[parseArgs] Parsing 2 arguments: -w /tmp/test-trader
[parseArgs] Parsed options: workspace=None, w=Some(/tmp/test-trader)
[parseArgs] Setting workspace from -w: /tmp/test-trader
[main] Workspace set to: /tmp/test-trader
```

---

## 📝 修改文件总结

1. **`src/io/input_utils.cj`**
   - 修复输入提示符不显示之前输入内容

2. **`src/main.cj`**
   - 添加 `import magic.log.LogUtils`
   - 添加参数接收日志
   - 添加workspace设置日志

3. **`src/parse_args.cj`**
   - 添加参数解析前日志
   - 添加参数解析后日志
   - 添加workspace设置详细日志

---

## 🎯 下一步

如果 `-w` 参数仍然不生效，请：

1. **检查日志文件** `.codelin/codelin.log` 中的调试信息
2. **确认参数格式** 使用正确的空格分隔格式
3. **验证路径存在** 确保指定的workspace路径存在
4. **检查权限** 确保对workspace目录有读写权限

---

**修复完成时间**: 2025-11-27

