# MCP 工具动态增强修复报告

## 问题理解

用户指出：**MCP 工具应该是动态获取的，不应该硬编码特定服务器的逻辑**。

这个观点完全正确。MCP 的核心价值在于：
1. **动态注册**：工具在运行时从 MCP 服务器动态加载
2. **通用机制**：不应该针对特定服务器写死逻辑
3. **自动发现**：LLM 应该能够通过工具描述自动理解和使用

## 问题分析

### 1. 当前实现的问题

**MCPToolWrapper 原始实现**：
```cangjie
public prop description: String {
    get() { this.mcpTool.description }  // 直接返回原始描述
}
```

**问题**：
- ❌ 工具描述中没有 MCP 服务器标识
- ❌ LLM 无法识别这是 MCP 工具
- ❌ 当用户说"通过agentmem搜索"时，LLM 无法关联到对应的工具

### 2. 错误的修复方向

**之前尝试**：硬编码特定服务器的逻辑
```cangjie
match (this.serverName) {
    case "agentmem" => // 特定逻辑
    case "context7" => // 特定逻辑
    ...
}
```

**问题**：
- ❌ 违反了 MCP 的动态特性
- ❌ 每增加一个 MCP 服务器都需要修改代码
- ❌ 失去了 MCP 的通用性

## 解决方案

### 通用描述增强机制

在 `MCPToolWrapper` 中实现**通用的、自动的描述增强**：

```cangjie
public prop description: String {
    get() {
        let baseDesc = this.mcpTool.description
        
        // 1. 添加 MCP 服务器标识（通用）
        let mcpPrefix = "[MCP: ${this.serverName}] "
        
        // 2. 检查工具名称是否包含服务器名称（帮助 LLM 关联）
        let toolNameHint = if (this.mcpTool.name.contains(this.serverName)) {
            "This tool is from MCP server '${this.serverName}'. "
        } else {
            "This is an MCP tool from server '${this.serverName}'. "
        }
        
        // 3. 添加使用指导（通用）
        let usageHint = "Use this tool when the user explicitly mentions '${this.serverName}' " +
                       "or wants to use functionality provided by the ${this.serverName} MCP server. "
        
        // 4. 组合增强后的描述
        return "${mcpPrefix}${baseDesc} ${toolNameHint}${usageHint}"
    }
}
```

### 特点

1. **完全通用**：不针对任何特定服务器
2. **自动增强**：所有 MCP 工具自动获得增强描述
3. **动态适应**：新增 MCP 服务器无需修改代码
4. **LLM 友好**：描述中包含服务器名称和使用指导

### 示例效果

**原始描述**：
```
"在 AgentMem 系统中搜索相关记忆"
```

**增强后描述**：
```
"[MCP: agentmem] 在 AgentMem 系统中搜索相关记忆. This tool is from MCP server 'agentmem'. Use this tool when the user explicitly mentions 'agentmem' or wants to use functionality provided by the agentmem MCP server."
```

## Agent Prompt 改进

### 通用 MCP 工具说明

在 Agent prompt 中添加通用的 MCP 工具说明，而不是列举特定工具：

```cangjie
## MCP (Model Context Protocol) Tools

You have access to tools from MCP (Model Context Protocol) servers that are dynamically loaded at runtime.
These tools extend your capabilities beyond the built-in tools.

**How to identify MCP tools:**
- MCP tools have descriptions prefixed with `[MCP: serverName]`
- Tool names often contain the server name (e.g., `agentmem_search_memories` from `agentmem` server)
- When the user explicitly mentions a server name (e.g., "通过agentmem搜索", "use agentmem to search"),
  you MUST use the corresponding MCP tool from that server

**Important**: MCP tools are dynamically registered - check available tools and their descriptions to understand
what each MCP server provides. Use MCP tools when the user's request matches the server's functionality.
```

### 特点

1. **通用指导**：不列举特定工具
2. **动态发现**：指导 LLM 如何识别和使用 MCP 工具
3. **自动适应**：新增 MCP 服务器无需修改 prompt

## 实现细节

### 文件修改

1. **`src/core/mcp/wrapper.cj`**
   - 增强 `description` 属性，添加通用的 MCP 上下文
   - 不硬编码任何特定服务器逻辑

2. **`src/core/agents/cangjie_code_agent.cj`**
   - 更新 prompt，添加通用的 MCP 工具说明
   - 移除特定工具的列举

### 工作流程

```
1. MCP 服务器启动
   └─> 提供工具列表（包含原始描述）

2. MCPToolWrapper 包装工具
   └─> 自动增强描述（添加 [MCP: serverName] 前缀和使用指导）

3. 工具注册到 Agent
   └─> agent.toolManager.addTools(mcpTools)

4. LLM 获取工具列表
   └─> 看到增强后的描述，能够识别 MCP 工具

5. 用户说"通过agentmem搜索X"
   └─> LLM 识别到 agentmem 服务器
   └─> 选择 agentmem_search_memories 工具
   └─> 正确调用 MCP 工具
```

## 验证

### 测试场景

1. **用户输入**："通过agentmem搜索林"
2. **预期行为**：
   - LLM 识别到 "agentmem" 关键词
   - 选择 `agentmem_search_memories` 工具
   - 传递参数：`{"query": "林", ...}`
3. **验证方法**：
   - 查看日志中的工具调用
   - 确认使用了 MCP 工具而不是 grepSearch

## 优势

1. ✅ **完全动态**：不硬编码任何特定服务器
2. ✅ **自动适应**：新增 MCP 服务器自动获得增强
3. ✅ **LLM 友好**：描述清晰，易于理解
4. ✅ **维护简单**：不需要为每个服务器写代码
5. ✅ **符合 MCP 理念**：保持动态性和通用性

## 总结

**核心改进**：
- 在 `MCPToolWrapper` 中实现通用的描述增强机制
- 在 Agent prompt 中添加通用的 MCP 工具使用指导
- 完全移除硬编码的特定服务器逻辑

**结果**：
- MCP 工具能够被 LLM 正确识别和使用
- 保持 MCP 的动态特性和通用性
- 新增 MCP 服务器无需修改代码

