#!/bin/bash

# Build script for Codelin with static configuration
# This script builds the project using static toml files and restores original configuration

# Note: We don't use 'set -e' because we need to ensure files are restored on any error

# Color variables for better output
readonly RED='\033[0;31m'
readonly GREEN='\033[0;32m'
readonly YELLOW='\033[1;33m'
readonly BLUE='\033[0;34m'
readonly MAGENTA='\033[0;35m'
readonly CYAN='\033[0;36m'
readonly WHITE='\033[1;37m'
readonly NC='\033[0m' # No Color

# Formatting functions
info() {
    echo -e "${CYAN}ℹ️  $1${NC}"
}

success() {
    echo -e "${GREEN}✅ $1${NC}"
}

warning() {
    echo -e "${YELLOW}⚠️  $1${NC}"
}

error() {
    echo -e "${RED}❌ $1${NC}" >&2
}

step() {
    echo -e "${BLUE}🚀 $1${NC}"
}

complete() {
    echo -e "${GREEN}🎉 $1${NC}"
}

header() {
    echo -e "${MAGENTA}╔══════════════════════════════════════════════════════════════╗${NC}"
    echo -e "${MAGENTA}  $1${NC}"
    echo -e "${MAGENTA}╚══════════════════════════════════════════════════════════════╝${NC}"
}

# Load Cangjie runtime libs
if [ -z "${CANGJIE_HOME+x}" ]; then
    error "CANGJIE_HOME is not set!"
    exit 1
fi
. $CANGJIE_HOME/envsetup.sh

# Get the directory where this script is located
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

# Define paths
MAGIC_PATH="$MAGIC_PATH"  # Expected to be set in environment
TARGET_DIR="./target/release/bin"
BINARY_DIR="./binary"
ORIGINAL_MAGIC_CJPM="$MAGIC_PATH/cjpm.toml"
ORIGINAL_LOCAL_CJPM="./cjpm.toml"
BACKUP_MAGIC_CJPM="$MAGIC_PATH/cjpm.toml.backup"
BACKUP_LOCAL_CJPM="./cjpm.toml.backup"

# Function to restore original files
restore_files() {
    info "Restoring original cjpm.toml files..."

    # Restore MAGIC_PATH/cjpm.toml if backup exists
    if [ -f "$BACKUP_MAGIC_CJPM" ]; then
        mv "$BACKUP_MAGIC_CJPM" "$ORIGINAL_MAGIC_CJPM"
        success "Restored $ORIGINAL_MAGIC_CJPM"
    fi

    # Restore local cjpm.toml if backup exists
    if [ -f "$BACKUP_LOCAL_CJPM" ]; then
        mv "$BACKUP_LOCAL_CJPM" "$ORIGINAL_LOCAL_CJPM"
        success "Restored $ORIGINAL_LOCAL_CJPM"
    fi
}

# Function to handle errors
error_handler() {
    local line_number=$1
    error "Error occurred at line $line_number"
    restore_files
    exit 1
}

# Function to run commands with error handling
run_command() {
    if ! "$@"; then
        error_handler $LINENO
    fi
}

# Check if MAGIC_PATH is set
if [ -z "$MAGIC_PATH" ]; then
    error "MAGIC_PATH environment variable is not set"
    error_handler $LINENO
fi

# Check if required files exist
if [ ! -f "$SCRIPT_DIR/codelin.toml" ]; then
    error "$SCRIPT_DIR/codelin.toml not found"
    error_handler $LINENO
fi

if [ ! -f "$SCRIPT_DIR/cli-static.toml" ]; then
    error "$SCRIPT_DIR/cli-static.toml not found"
    error_handler $LINENO
fi

header "Starting Codelin Build Process"
echo ""
info "Script directory: $SCRIPT_DIR"
info "Magic path: $MAGIC_PATH"
echo ""

# Step 1: Backup original MAGIC_PATH/cjpm.toml and copy static version
step "Step 1: Setting up static configuration for $MAGIC_PATH/cjpm.toml"
if [ -f "$ORIGINAL_MAGIC_CJPM" ]; then
    cp "$ORIGINAL_MAGIC_CJPM" "$BACKUP_MAGIC_CJPM"
    success "Backed up $ORIGINAL_MAGIC_CJPM to $BACKUP_MAGIC_CJPM"
fi
cp "$SCRIPT_DIR/codelin.toml" "$ORIGINAL_MAGIC_CJPM"
success "Copied $SCRIPT_DIR/codelin.toml to $ORIGINAL_MAGIC_CJPM"
echo ""

# Step 2: Backup original local cjpm.toml and copy static version
step "Step 2: Setting up static configuration for local cjpm.toml"
if [ -f "$ORIGINAL_LOCAL_CJPM" ]; then
    cp "$ORIGINAL_LOCAL_CJPM" "$BACKUP_LOCAL_CJPM"
    success "Backed up $ORIGINAL_LOCAL_CJPM to $BACKUP_LOCAL_CJPM"
fi
cp "$SCRIPT_DIR/cli-static.toml" "$ORIGINAL_LOCAL_CJPM"
success "Copied $SCRIPT_DIR/cli-static.toml to $ORIGINAL_LOCAL_CJPM"
echo ""

# Step 3: Clean up ffi directory and build
step "Step 3.1: Cleaning up ffi directory..."
if [ -d "./ffi" ]; then
    info "Removing .a, .so, and .dylib files from ./ffi directory..."
    find ./ffi -name "*.a" -type f -delete
    find ./ffi -name "*.so" -type f -delete
    find ./ffi -name "*.dylib" -type f -delete
    success "Cleanup completed"
else
    warning "./ffi directory not found, skipping cleanup"
fi
echo ""

step "Step 3.2: Cleaning up existing project..."
run_command cjpm clean >/dev/null 2>&1
echo ""

step "Step 3.3: Building project with static configuration..."
BUILD_LOG="/tmp/codelin-build-$(date +%s).log"
info "构建日志将保存到: $BUILD_LOG"

# 执行构建，保存输出到日志
if cjpm build 2>&1 | tee "$BUILD_LOG"; then
    success "编译命令执行成功"
else
    BUILD_EXIT_CODE=$?
    error "编译失败，退出码: $BUILD_EXIT_CODE"
    error "构建日志: $BUILD_LOG"
    info "最后 50 行构建输出:"
    tail -50 "$BUILD_LOG" | sed 's/^/  /'
    restore_files
    exit 1
fi
echo ""

# Step 4: Copy binary to target location
step "Step 4: Copying binary..."
if [ ! -d "$BINARY_DIR" ]; then
    mkdir -p "$BINARY_DIR"
    success "Created directory: $BINARY_DIR"
fi

# 全面检查所有可能的二进制文件位置
info "检查编译输出位置..."

# 检查多个可能的位置
BINARY_SOURCE=""
BINARY_NAME=""
BINARY_FOUND=false

# 优先级 1: target/release/bin/codelin (项目名，release 版本)
if [ -f "$TARGET_DIR/codelin" ]; then
    BINARY_SOURCE="$TARGET_DIR/codelin"
    BINARY_NAME="codelin"
    BINARY_FOUND=true
    success "Found binary at $TARGET_DIR/codelin"
# 优先级 2: target/release/bin/cli (历史遗留名称，release 版本)
elif [ -f "$TARGET_DIR/cli" ]; then
    BINARY_SOURCE="$TARGET_DIR/cli"
    BINARY_NAME="cli (legacy)"
    BINARY_FOUND=true
    warning "Found legacy binary name 'cli' at $TARGET_DIR/cli, will rename to 'codelin'"
# 优先级 3: target/debug/bin/codelin (调试版本)
elif [ -f "./target/debug/bin/codelin" ]; then
    BINARY_SOURCE="./target/debug/bin/codelin"
    BINARY_NAME="codelin (debug)"
    BINARY_FOUND=true
    warning "Found debug binary at ./target/debug/bin/codelin (not release version)"
# 优先级 4: target/debug/bin/cli (调试版本，历史遗留)
elif [ -f "./target/debug/bin/cli" ]; then
    BINARY_SOURCE="./target/debug/bin/cli"
    BINARY_NAME="cli (debug, legacy)"
    BINARY_FOUND=true
    warning "Found debug binary at ./target/debug/bin/cli (not release version)"
fi

# 如果都没找到，提供详细的调试信息
if [ "$BINARY_FOUND" = false ]; then
    error "Binary not found in any expected location!"
    echo ""
    info "已检查的位置："
    info "  ❌ $TARGET_DIR/codelin"
    info "  ❌ $TARGET_DIR/cli"
    info "  ❌ ./target/debug/bin/codelin"
    info "  ❌ ./target/debug/bin/cli"
    echo ""
    info "调试信息："
    info "  1. 检查 target 目录结构:"
    if [ -d "./target" ]; then
        find ./target -type f -name "codelin" -o -name "cli" 2>/dev/null | head -10 | sed 's/^/    /'
        if [ $? -ne 0 ] || [ -z "$(find ./target -type f -name "codelin" -o -name "cli" 2>/dev/null)" ]; then
            info "    未找到任何二进制文件"
        fi
    else
        info "    target 目录不存在"
    fi
    echo ""
    info "  2. 检查构建日志: $BUILD_LOG"
    info "  3. 检查编译错误:"
    if [ -f "$BUILD_LOG" ]; then
        grep -i "error\|failed\|失败" "$BUILD_LOG" | tail -10 | sed 's/^/    /'
    fi
    echo ""
    error "请检查构建日志并修复编译错误后重试。"
    restore_files
    exit 1
fi

# 复制到 binary/codelin（统一命名）
cp "$BINARY_SOURCE" "$BINARY_DIR/codelin"
chmod +x "$BINARY_DIR/codelin"
success "Copied $BINARY_NAME to $BINARY_DIR/codelin"

# 为了向后兼容，也创建 codelin-cli 的符号链接（可选）
if [ -f "$BINARY_DIR/codelin-cli" ]; then
    rm -f "$BINARY_DIR/codelin-cli"
fi
ln -sf "codelin" "$BINARY_DIR/codelin-cli"
info "Created symlink: binary/codelin-cli -> codelin (for backward compatibility)"
echo ""

# Step 5: Restore original files
step "Step 5: Cleaning up and restoring original files..."
restore_files
echo ""

complete "Build completed successfully!"
success "Binary available at: $BINARY_DIR/codelin"