#!/bin/bash

OS="$(uname -s)"

# Get the absolute path of the script's directory (resolves symlinks)
SCRIPT_DIR=$(dirname "$(realpath "$0" 2>/dev/null || readlink -f "$0" 2>/dev/null || echo "$0")")

# Load environment variables from .env file
ENV_FILE="$SCRIPT_DIR/../.codelin/.env"
if [ -f "$ENV_FILE" ]; then
    while IFS= read -r line || [ -n "$line" ]; do
        # Skip comments and empty lines
        if [[ "$line" =~ ^[[:space:]]*# ]] || [[ -z "${line// }" ]]; then
            continue
        fi
        # Export variables
        if [[ "$line" =~ ^[[:space:]]*([A-Za-z_][A-Za-z0-9_]*)= ]]; then
            export "$line"
        fi
    done < "$ENV_FILE"
fi

# Load Cangjie runtime libs
if [ -z "${CANGJIE_HOME+x}" ]; then
    echo "Error: CANGJIE_HOME is not set!" >&2
    exit 1
fi
. $CANGJIE_HOME/envsetup.sh

# Set the Cangjie stdx libs according to the Magic project
if [ -z "${MAGIC_PATH+x}" ]; then
    echo "Error: MAGIC_PATH is not set!" >&2
    exit 1
fi
# Find the correct platform-specific stdx path (prioritize macOS/Linux over Windows)
STDX_LIB_PATH=$(find "${MAGIC_PATH:-}" -type d -path '*/darwin_aarch64_llvm/dynamic/stdx' -print -quit 2>/dev/null)
if [ -z "$STDX_LIB_PATH" ]; then
    STDX_LIB_PATH=$(find "${MAGIC_PATH:-}" -type d -path '*/linux_*/dynamic/stdx' -print -quit 2>/dev/null)
fi
if [ -z "$STDX_LIB_PATH" ]; then
    STDX_LIB_PATH=$(find "${MAGIC_PATH:-}" -type d -path '*/dynamic/stdx' -print -quit 2>/dev/null)
fi

# Set libs of this project and Magic according to "../target" relative to the script's location
CLI_LIB_PATH="$SCRIPT_DIR/../target/release/cli:$SCRIPT_DIR/../ffi"
MAGIC_LIB_PATH="$SCRIPT_DIR/../target/release/magic"

# Merge all lib paths
LIB_PATHS="$CLI_LIB_PATH:$MAGIC_LIB_PATH:$STDX_LIB_PATH"

# Path to the executable CLI binary
CLI_BIN="$SCRIPT_DIR/../target/release/bin/cli"
# Check if the executable exists before running
if [[ ! -f "$CLI_BIN" ]]; then
    echo "Error: Executable not found at $CLI_BIN" >&2
    exit 1
fi

# Add the library path to `LD_LIBRARY_PATH` (for Linux) and `DYLD_LIBRARY_PATH` (for macOS)
case "$OS" in
    "Linux")
        export LD_LIBRARY_PATH="$LIB_PATHS:$LD_LIBRARY_PATH"
        ;;
    "Darwin")
        export DYLD_LIBRARY_PATH="$LIB_PATHS:$DYLD_LIBRARY_PATH"  # macOS support
        # xattr -dr com.apple.quarantine ${script_dir}/* &> /dev/null || true
        codesign -s - -f --preserve-metadata=entitlements,requirements,flags,runtime $CLI_BIN &> /dev/null || true
        ;;
    *)
        echo "Unknown/Unsupported OS: $OS"
        ;;
esac

# Execute the CLI binary and forward all arguments (`$@`)
exec "$CLI_BIN" "$@"