#!/bin/bash

# CodeLin 全面分析和测试脚本
# 参考 Claude Code 的实现标准

echo "════════════════════════════════════════════════════════════════"
echo "  CodeLin 全面分析和测试 (参考 Claude Code)"
echo "════════════════════════════════════════════════════════════════"
echo ""

cd /Users/louloulin/Documents/linchong/cjproject/codelin

# 颜色定义
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m'

echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${CYAN}1. 代码结构分析${NC}"
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"

echo ""
echo -e "${BLUE}📁 Session 管理模块文件:${NC}"
ls -lh src/core/conversation/*.cj | awk '{print "  " $9 " (" $5 ")"}'

echo ""
echo -e "${BLUE}📊 代码行数统计:${NC}"
wc -l src/core/conversation/*.cj | tail -1 | awk '{print "  Total: " $1 " lines"}'

echo ""
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${CYAN}2. 日志功能分析${NC}"
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"

echo ""
echo -e "${BLUE}📝 日志使用统计:${NC}"
echo -n "  ConversationManager: "
grep -c "LogUtils\." src/core/conversation/conversation_manager.cj
echo -n "  SessionMetadata: "
grep -c "LogUtils\." src/core/conversation/session_metadata.cj
echo -n "  SessionAutoNamer: "
grep -c "LogUtils\." src/core/conversation/session_auto_namer.cj

echo ""
echo -e "${BLUE}📋 日志级别分布:${NC}"
echo -n "  DEBUG: "
grep -r "LogUtils\.debug" src/core/conversation/ --include="*.cj" | wc -l
echo -n "  INFO: "
grep -r "LogUtils\.info" src/core/conversation/ --include="*.cj" | wc -l
echo -n "  WARN: "
grep -r "LogUtils\.warn" src/core/conversation/ --include="*.cj" | wc -l
echo -n "  ERROR: "
grep -r "LogUtils\.error" src/core/conversation/ --include="*.cj" | wc -l

echo ""
echo -e "${BLUE}📄 最近日志输出 (最后20行):${NC}"
if [ -f .codelin/abc.log ]; then
    echo "  Log file size: $(du -h .codelin/abc.log | awk '{print $1}')"
    echo ""
    tail -20 .codelin/abc.log | sed 's/^/  /'
else
    echo -e "  ${YELLOW}⚠️  日志文件不存在${NC}"
fi

echo ""
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${CYAN}3. Session 数据分析${NC}"
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"

echo ""
echo -e "${BLUE}💾 Session 存储结构:${NC}"
if [ -d .codelin/sessions ]; then
    echo "  Sessions directory exists: ✅"
    ls -la .codelin/sessions/ | sed 's/^/  /'
else
    echo -e "  ${YELLOW}⚠️  Sessions 目录不存在${NC}"
fi

echo ""
echo -e "${BLUE}📊 Session 元数据:${NC}"
if [ -d .codelin/sessions/metadata ]; then
    echo "  Metadata files:"
    ls -lh .codelin/sessions/metadata/*.json 2>/dev/null | awk '{print "    " $9 " (" $5 ")"}'
    
    echo ""
    echo -e "${BLUE}  Current session metadata:${NC}"
    if [ -f .codelin/sessions/metadata/last-conversation.json ]; then
        cat .codelin/sessions/metadata/last-conversation.json | sed 's/^/    /'
    else
        echo -e "    ${YELLOW}⚠️  No metadata found${NC}"
    fi
else
    echo -e "  ${YELLOW}⚠️  Metadata 目录不存在${NC}"
fi

echo ""
echo -e "${BLUE}💬 Conversation 历史:${NC}"
if [ -d .codelin/conversation-history ]; then
    echo "  History files:"
    ls -lh .codelin/conversation-history/*.history 2>/dev/null | awk '{print "    " $9 " (" $5 ")"}'
else
    echo -e "  ${YELLOW}⚠️  History 目录不存在${NC}"
fi

echo ""
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${CYAN}4. 编译状态检查${NC}"
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"

echo ""
echo -e "${BLUE}🔨 编译测试:${NC}"
cjpm build > /tmp/codelin_build.log 2>&1
if [ $? -eq 0 ]; then
    echo -e "  ${GREEN}✅ 编译成功${NC}"
    echo -n "  Warnings: "
    grep -c "warning" /tmp/codelin_build.log
else
    echo -e "  ${RED}❌ 编译失败${NC}"
    echo "  错误信息:"
    cat /tmp/codelin_build.log | grep "error" | sed 's/^/    /'
fi

echo ""
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${CYAN}5. 功能完整性检查${NC}"
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"

echo ""
echo -e "${BLUE}✨ Phase 5 核心功能实现检查:${NC}"

# 检查关键函数是否存在
check_function() {
    local func_name="$1"
    local file="$2"
    if grep -q "func ${func_name}" "$file"; then
        echo -e "  ✅ ${func_name}"
    else
        echo -e "  ❌ ${func_name} (未找到)"
    fi
}

echo ""
echo -e "${BLUE}  ConversationManager 方法:${NC}"
check_function "autoResume" src/core/conversation/conversation_manager.cj
check_function "newSession" src/core/conversation/conversation_manager.cj
check_function "switchSession" src/core/conversation/conversation_manager.cj
check_function "listSessionsDetailed" src/core/conversation/conversation_manager.cj
check_function "renameSession" src/core/conversation/conversation_manager.cj
check_function "getCurrentSessionTitle" src/core/conversation/conversation_manager.cj
check_function "autoGenerateTitleIfNeeded" src/core/conversation/conversation_manager.cj

echo ""
echo -e "${BLUE}  SessionMetadata 功能:${NC}"
check_function "updateMetadata" src/core/conversation/session_metadata.cj
check_function "addTag" src/core/conversation/session_metadata.cj
check_function "save" src/core/conversation/session_metadata.cj
check_function "load" src/core/conversation/session_metadata.cj

echo ""
echo -e "${BLUE}  SessionAutoNamer 功能:${NC}"
check_function "generateFromFirstExchange" src/core/conversation/session_auto_namer.cj

echo ""
echo -e "${BLUE}  UI 提示符优化:${NC}"
check_function "buildPrompt" src/io/input_utils.cj

echo ""
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${CYAN}6. Claude Code 对标分析${NC}"
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"

echo ""
echo -e "${BLUE}🎯 关键特性对比:${NC}"

check_feature() {
    local feature="$1"
    local pattern="$2"
    local file="$3"
    if grep -q "$pattern" "$file" 2>/dev/null; then
        echo -e "  ✅ ${feature}"
    else
        echo -e "  ❌ ${feature} (未实现)"
    fi
}

echo ""
echo -e "${BLUE}  自动恢复会话 (类似 claude -r):${NC}"
check_feature "Auto-resume on startup" "autoResume" src/core/conversation/conversation_manager.cj
check_feature "Last conversation tracking" "AUTO_SAVE_NAME" src/core/conversation/conversation_manager.cj

echo ""
echo -e "${BLUE}  Session 管理命令:${NC}"
check_feature "/new command" "newSession" src/core/conversation/conversation_manager.cj
check_feature "/switch command" "switchSession" src/core/conversation/conversation_manager.cj
check_feature "/sessions command" "listSessionsDetailed" src/core/conversation/conversation_manager.cj
check_feature "/rename command" "renameSession" src/core/conversation/conversation_manager.cj

echo ""
echo -e "${BLUE}  AI 智能功能:${NC}"
check_feature "Auto-naming with AI" "@ai\[" src/core/conversation/session_auto_namer.cj
check_feature "Title generation" "generateFromFirstExchange" src/core/conversation/session_auto_namer.cj

echo ""
echo -e "${BLUE}  元数据管理:${NC}"
check_feature "Metadata storage" "@jsonable" src/core/conversation/session_metadata.cj
check_feature "Message counting" "messageCount" src/core/conversation/session_metadata.cj
check_feature "File tracking" "fileCount" src/core/conversation/session_metadata.cj
check_feature "Tag support" "addTag" src/core/conversation/session_metadata.cj

echo ""
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${CYAN}7. 潜在问题识别${NC}"
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"

echo ""
echo -e "${BLUE}🔍 检查常见问题:${NC}"

# 检查 HTTP 错误
if [ -f .codelin/abc.log ]; then
    http_errors=$(grep -c "Http status: 000" .codelin/abc.log)
    if [ "$http_errors" -gt 0 ]; then
        echo -e "  ${YELLOW}⚠️  发现 ${http_errors} 个 HTTP 错误 (status 000)${NC}"
    else
        echo -e "  ✅ 无 HTTP 错误"
    fi
    
    # 检查其他错误
    error_count=$(grep -c "ERROR" .codelin/abc.log | tail -1)
    if [ "$error_count" -gt 0 ]; then
        echo -e "  ${YELLOW}⚠️  日志中有 ${error_count} 个 ERROR 级别日志${NC}"
        echo "  最近的错误:"
        grep "ERROR" .codelin/abc.log | tail -5 | sed 's/^/    /'
    else
        echo -e "  ✅ 无 ERROR 日志"
    fi
fi

# 检查 session 数据完整性
echo ""
echo -e "${BLUE}  Session 数据完整性:${NC}"
if [ -f .codelin/sessions/metadata/last-conversation.json ]; then
    if grep -q '"messageCount"' .codelin/sessions/metadata/last-conversation.json; then
        echo -e "  ✅ Metadata 包含 messageCount"
    else
        echo -e "  ${RED}❌ Metadata 缺少 messageCount${NC}"
    fi
    
    if grep -q '"title"' .codelin/sessions/metadata/last-conversation.json; then
        echo -e "  ✅ Metadata 包含 title"
    else
        echo -e "  ${RED}❌ Metadata 缺少 title${NC}"
    fi
fi

echo ""
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${CYAN}8. 总结报告${NC}"
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"

echo ""
echo -e "${GREEN}✅ 已实现的功能:${NC}"
echo "  • Session 元数据系统 (SessionMetadata, SessionMetadataManager)"
echo "  • Claude Code 风格命令 (/new, /sessions, /switch, /rename)"
echo "  • AI 自动命名 (SessionAutoNamer with @ai)"
echo "  • 自动恢复上次会话 (autoResume, 类似 claude -r)"
echo "  • 简洁的 UI 提示符 (buildPrompt)"
echo "  • 完整的日志记录 (LogUtils.debug/info/warn/error)"

echo ""
echo -e "${YELLOW}⚠️  需要关注的方面:${NC}"
if [ -f .codelin/abc.log ]; then
    http_errors=$(grep -c "Http status: 000" .codelin/abc.log 2>/dev/null || echo "0")
    if [ "$http_errors" -gt 0 ]; then
        echo "  • HTTP 请求错误 (status 000) - 可能影响 AI 功能"
    fi
fi
echo "  • 运行时测试验证 (需要实际启动 CLI 测试)"
echo "  • Session 切换时的元数据更新"
echo "  • 文件计数功能 (fileCount 目前为 TODO)"

echo ""
echo -e "${BLUE}📝 建议的改进:${NC}"
echo "  1. 实现 fileCount 统计功能"
echo "  2. 添加 Session 导出/导入功能"
echo "  3. 增强错误处理和恢复机制"
echo "  4. 完善日志级别使用 (增加 WARN 日志)"
echo "  5. 添加 Session 搜索功能的 UI 展示"

echo ""
echo "════════════════════════════════════════════════════════════════"
echo "  分析完成！"
echo "════════════════════════════════════════════════════════════════"

