#!/bin/bash

# Comprehensive test for P0 + P1 implementation
# Tests all core features: MCP integration, prompt simplification, dynamic injection, context pruner, layers architecture, mode system, tool optimization

# No set -e here to allow all tests to run even if one fails

echo "=========================================="
echo "Comprehensive P0 + P1 Implementation Test"
echo "=========================================="
echo ""

# Colors
GREEN='\033[0;32m'
RED='\033[0;31m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

PASSED=0
FAILED=0

test_check() {
    if [ $? -eq 0 ]; then
        echo -e "${GREEN}✓${NC} $1"
        ((PASSED++))
    else
        echo -e "${RED}✗${NC} $1"
        ((FAILED++))
    fi
}

# P0-1: MCP Prompt Tests
echo -e "${BLUE}=== P0-1: MCP Prompt Emergency Fix ===${NC}"
echo ""

# Test MCP prompt module exists
if [ -f "src/core/agents/mcp_prompt.cj" ]; then
    test_check "MCP prompt module exists"
else
    echo -e "${RED}✗${NC} MCP prompt module exists"
    ((FAILED++))
fi

# Test MCP prompt integrated in General Agent
if grep -q "PLACEHOLDER.MCP_TOOLS_PROMPT" src/core/agents/general_code_agent.cj; then
    test_check "MCP prompt integrated in General Agent"
else
    echo -e "${RED}✗${NC} MCP prompt integrated in General Agent"
    ((FAILED++))
fi

# Test MCP prompt integrated in Cangjie Agent
if grep -q "PLACEHOLDER.MCP_TOOLS_PROMPT" src/core/agents/cangjie_code_agent.cj; then
    test_check "MCP prompt integrated in Cangjie Agent"
else
    echo -e "${RED}✗${NC} MCP prompt integrated in Cangjie Agent"
    ((FAILED++))
fi

# Ensure PromptFormatter injects MCP prompt
if grep -q "MCPPrompt.MCP_TOOLS_PROMPT" src/core/agents/prompt_formatter.cj; then
    test_check "PromptFormatter wires MCP prompt"
else
    echo -e "${RED}✗${NC} PromptFormatter wires MCP prompt"
    ((FAILED++))
fi

# Test ToolListGenerator exists
if [ -f "src/core/agents/tool_list_generator.cj" ]; then
    test_check "ToolListGenerator exists"
else
    echo -e "${RED}✗${NC} ToolListGenerator exists"
    ((FAILED++))
fi

# Test AgentListGenerator exists
if [ -f "src/core/agents/agent_list_generator.cj" ]; then
    test_check "AgentListGenerator exists"
else
    echo -e "${RED}✗${NC} AgentListGenerator exists"
    ((FAILED++))
fi

# Test dynamic injection in General Agent
if (grep -q "ToolListGenerator.generateToolList" src/core/agents/general_code_agent.cj || \
    grep -q "PLACEHOLDER.TOOLS_PROMPT" src/core/agents/general_code_agent.cj) && \
   (grep -q "AgentListGenerator.generateAgentList" src/core/agents/general_code_agent.cj || \
    grep -q "PLACEHOLDER.AGENTS_PROMPT" src/core/agents/general_code_agent.cj); then
    test_check "Dynamic injection in General Agent"
else
    echo -e "${RED}✗${NC} Dynamic injection in General Agent"
    ((FAILED++))
fi

# Test dynamic injection in Cangjie Agent
if (grep -q "ToolListGenerator.generateToolList" src/core/agents/cangjie_code_agent.cj || \
    grep -q "PLACEHOLDER.TOOLS_PROMPT" src/core/agents/cangjie_code_agent.cj) && \
   (grep -q "AgentListGenerator.generateAgentList" src/core/agents/cangjie_code_agent.cj || \
    grep -q "PLACEHOLDER.AGENTS_PROMPT" src/core/agents/cangjie_code_agent.cj); then
    test_check "Dynamic injection in Cangjie Agent"
else
    echo -e "${RED}✗${NC} Dynamic injection in Cangjie Agent"
    ((FAILED++))
fi

echo ""

# P0-2: Prompt Simplification Tests
echo -e "${BLUE}=== P0-2: Prompt Simplification + Dynamic Injection ===${NC}"
echo ""

GENERAL_LINES=$(wc -l < src/core/agents/general_code_agent.cj)
CANGJIE_LINES=$(wc -l < src/core/agents/cangjie_code_agent.cj)
ORIGINAL_GENERAL_LINES=646
ORIGINAL_CANGJIE_LINES=462
ORIGINAL_TOTAL_LINES=1108

# Test General Agent prompt reduced
if [ "$GENERAL_LINES" -lt 500 ]; then
    test_check "General Agent prompt reduced (${GENERAL_LINES} lines, was ${ORIGINAL_GENERAL_LINES})"
else
    echo -e "${RED}✗${NC} General Agent prompt reduced (${GENERAL_LINES} lines, was ${ORIGINAL_GENERAL_LINES})"
    ((FAILED++))
fi

# Test Cangjie Agent prompt reduced
if [ "$CANGJIE_LINES" -lt 520 ]; then
    test_check "Cangjie Agent prompt reduced (${CANGJIE_LINES} lines, was ${ORIGINAL_CANGJIE_LINES})"
else
    echo -e "${RED}✗${NC} Cangjie Agent prompt reduced (${CANGJIE_LINES} lines, was ${ORIGINAL_CANGJIE_LINES})"
    ((FAILED++))
fi

# Test total reduction
TOTAL_LINES=$((GENERAL_LINES + CANGJIE_LINES))
if [ "$TOTAL_LINES" -lt 1000 ]; then
    test_check "Total prompt reduction (${TOTAL_LINES} lines, was ${ORIGINAL_TOTAL_LINES})"
else
    echo -e "${RED}✗${NC} Total prompt reduction (${TOTAL_LINES} lines, was ${ORIGINAL_TOTAL_LINES})"
    ((FAILED++))
fi

echo ""

# P0-3: Context Pruner Tests
echo -e "${BLUE}=== P0-3: Context Pruner Implementation ===${NC}"
echo ""

# Test context compactor exists
if [ -f "src/core/agents/context_compactor.cj" ]; then
    test_check "Context compactor exists"
else
    echo -e "${RED}✗${NC} Context compactor exists"
    ((FAILED++))
fi

# Test 4-pass strategy in prompt
if grep -q "Pass 1:" src/core/agents/context_compactor.cj && \
   grep -q "Pass 2:" src/core/agents/context_compactor.cj && \
   grep -q "Pass 3:" src/core/agents/context_compactor.cj && \
   grep -q "Pass 4:" src/core/agents/context_compactor.cj; then
    test_check "4-pass compression strategy in prompt"
else
    echo -e "${RED}✗${NC} 4-pass compression strategy in prompt"
    ((FAILED++))
fi

echo ""

# P1-1: Layers Architecture Tests
echo -e "${BLUE}=== P1-1: Layers Architecture ===${NC}"
echo ""

# Test Layers Architecture in General Agent
if grep -q "🏗️ Layers Architecture" src/core/agents/general_code_agent.cj && \
   grep -q "Layer 1: Context Gathering" src/core/agents/general_code_agent.cj && \
   grep -q "Layer 5: Validation" src/core/agents/general_code_agent.cj && \
   grep -q "AUTO-EXECUTED" src/core/agents/general_code_agent.cj && \
   grep -q "/validate" src/core/agents/general_code_agent.cj; then
    test_check "Layers Architecture in General Agent (with Auto Layer 1/5 support)"
else
    echo -e "${RED}✗${NC} Layers Architecture in General Agent (with Auto Layer 1/5 support)"
    ((FAILED++))
fi

# Test Layers Architecture in Cangjie Agent
if grep -q "🏗️ Layers Architecture" src/core/agents/cangjie_code_agent.cj && \
   grep -q "Layer 1: Context Gathering" src/core/agents/cangjie_code_agent.cj && \
   grep -q "AUTO-EXECUTED" src/core/agents/cangjie_code_agent.cj && \
   grep -q "/validate" src/core/agents/cangjie_code_agent.cj; then
    test_check "Layers Architecture in Cangjie Agent (with Auto Layer 1/5 support)"
else
    echo -e "${RED}✗${NC} Layers Architecture in Cangjie Agent (with Auto Layer 1/5 support)"
    ((FAILED++))
fi

echo ""

# P1-2: Mode System Tests
echo -e "${BLUE}=== P1-2: Mode System ===${NC}"
echo ""

# Test Mode System in General Agent
if grep -q "🎭 Agent Execution Modes" src/core/agents/general_code_agent.cj && \
   grep -q "🚀 Fast Mode" src/core/agents/general_code_agent.cj && \
   grep -q "⚙️ Default Mode" src/core/agents/general_code_agent.cj && \
   grep -q "💎 Max Mode" src/core/agents/general_code_agent.cj; then
    test_check "Mode System in General Agent"
else
    echo -e "${RED}✗${NC} Mode System in General Agent"
    ((FAILED++))
fi

# Test Mode System in Cangjie Agent
if grep -q "🎭 Agent Execution Modes" src/core/agents/cangjie_code_agent.cj && \
   grep -q "🚀 Fast Mode" src/core/agents/cangjie_code_agent.cj; then
    test_check "Mode System in Cangjie Agent"
else
    echo -e "${RED}✗${NC} Mode System in Cangjie Agent"
    ((FAILED++))
fi

echo ""

# P1-3: Tool System Optimization Tests
echo -e "${BLUE}=== P1-3: Tool System Optimization ===${NC}"
echo ""

# Test tool categorization
if grep -q "categorizeTools" src/core/agents/tool_list_generator.cj && \
   grep -q "File Operations (Most Common)" src/core/agents/tool_list_generator.cj && \
   grep -q "Code Analysis" src/core/agents/tool_list_generator.cj; then
    test_check "Tool categorization implemented"
else
    echo -e "${RED}✗${NC} Tool categorization implemented"
    ((FAILED++))
fi

# Test concise description format
if grep -q "formatToolDescription(tool: Tool, concise" src/core/agents/tool_list_generator.cj && \
   grep -q "firstSentence" src/core/agents/tool_list_generator.cj; then
    test_check "Concise description format implemented"
else
    echo -e "${RED}✗${NC} Concise description format implemented"
    ((FAILED++))
fi

echo ""

# P2-1: SubAgent Specialization Tests
echo -e "${BLUE}=== P2-1: SubAgent Specialization ===${NC}"
echo ""

# Test FileSearchAgent prompt exists
if grep -q "class FileSearchAgent" src/core/agents/subagents/explorer_agent.cj && \
   grep -q "# File Search Results" src/core/agents/subagents/explorer_agent.cj; then
    test_check "FileSearchAgent prompt added"
else
    echo -e "${RED}✗${NC} FileSearchAgent prompt added"
    ((FAILED++))
fi

# Test CodeSearchAgent prompt exists
if grep -q "class CodeSearchAgent" src/core/agents/subagents/explorer_agent.cj && \
   grep -q "# Code Search Report" src/core/agents/subagents/explorer_agent.cj; then
    test_check "CodeSearchAgent prompt added"
else
    echo -e "${RED}✗${NC} CodeSearchAgent prompt added"
    ((FAILED++))
fi

# Test DirectoryAnalyzerAgent prompt exists
if grep -q "class DirectoryAnalyzerAgent" src/core/agents/subagents/explorer_agent.cj && \
   grep -q "# Directory Analysis" src/core/agents/subagents/explorer_agent.cj; then
    test_check "DirectoryAnalyzerAgent prompt added"
else
    echo -e "${RED}✗${NC} DirectoryAnalyzerAgent prompt added"
    ((FAILED++))
fi

# Test ResearcherWebAgent prompt exists
if grep -q "class ResearcherWebAgent" src/core/agents/subagents/explorer_agent.cj && \
   grep -q "# Web Research Findings" src/core/agents/subagents/explorer_agent.cj; then
    test_check "ResearcherWebAgent prompt added"
else
    echo -e "${RED}✗${NC} ResearcherWebAgent prompt added"
    ((FAILED++))
fi

# Test ResearcherDocsAgent prompt exists
if grep -q "class ResearcherDocsAgent" src/core/agents/subagents/explorer_agent.cj && \
   grep -q "# Documentation Findings" src/core/agents/subagents/explorer_agent.cj; then
    test_check "ResearcherDocsAgent prompt added"
else
    echo -e "${RED}✗${NC} ResearcherDocsAgent prompt added"
    ((FAILED++))
fi

# Test CLI integration adds new SubAgents
if grep -q "FileSearchAgent()" src/app/cli_app.cj && \
   grep -q "CodeSearchAgent()" src/app/cli_app.cj && \
   grep -q "DirectoryAnalyzerAgent()" src/app/cli_app.cj && \
   grep -q "ResearcherWebAgent()" src/app/cli_app.cj && \
   grep -q "ResearcherDocsAgent()" src/app/cli_app.cj; then
    test_check "New SubAgents registered in CliApp"
else
    echo -e "${RED}✗${NC} New SubAgents registered in CliApp"
    ((FAILED++))
fi

# Test allowed agent list updated
if grep -q "FileSearchAgent" src/app/cli_app.cj && \
   grep -q "DirectoryAnalyzerAgent" src/app/cli_app.cj && \
   grep -q "ResearcherDocsAgent" src/app/cli_app.cj; then
    test_check "Allowed SubAgent list updated"
else
    echo -e "${RED}✗${NC} Allowed SubAgent list updated"
    ((FAILED++))
fi

echo ""

# P2-2: Parallel Execution Optimization
echo -e "${BLUE}=== P2-2: Parallel Execution Optimization ===${NC}"
echo ""

# Test auto context layer helper exists
if grep -q "Layer1-AutoContext" src/app/cli_app.cj && \
   grep -q "buildAutoContextSection" src/app/cli_app.cj; then
    test_check "Auto Layer 1 context builder implemented"
else
    echo -e "${RED}✗${NC} Auto Layer 1 context builder implemented"
    ((FAILED++))
fi

# Test process_input injects auto layer
if grep -q "buildAutoContextSection" src/app/process_input.cj; then
    test_check "Auto Layer 1 injected before agent execution"
else
    echo -e "${RED}✗${NC} Auto Layer 1 injected before agent execution"
    ((FAILED++))
fi

# Test auto validation helper exists
if grep -q "runAutoValidationSection" src/app/cli_app.cj && \
   grep -q "buildParallelValidationJobs" src/app/cli_app.cj; then
    test_check "Auto Layer 5 validation helper implemented"
else
    echo -e "${RED}✗${NC} Auto Layer 5 validation helper implemented"
    ((FAILED++))
fi

# Test /validate command wiring
if grep -q "/validate" src/app/process_input.cj; then
    test_check "/validate command dispatches to auto validation"
else
    echo -e "${RED}✗${NC} /validate command dispatches to auto validation"
    ((FAILED++))
fi

# Test auto layer metrics hook
if grep -q "printAutoLayerStats" src/app/cli_app.cj && \
   grep -q "/autolayer-stats" src/app/process_input.cj; then
    test_check "Auto Layer metrics command wired"
else
    echo -e "${RED}✗${NC} Auto Layer metrics command wired"
    ((FAILED++))
fi

# Test auto layer stats exposed in prompt
if grep -q "autoLayerStatsInline" src/core/agents/prompt_formatter.cj; then
    test_check "Auto Layer stats injected into System Info section"
else
    echo -e "${RED}✗${NC} Auto Layer stats injected into System Info section"
    ((FAILED++))
fi

echo ""

# Summary
echo -e "${YELLOW}=== Summary ===${NC}"
echo ""
echo "Tests Passed: ${PASSED}"
echo "Tests Failed: ${FAILED}"
echo ""
echo "Prompt Statistics:"
echo "  General Agent:      ${GENERAL_LINES} lines (was ${ORIGINAL_GENERAL_LINES}, reduced $((ORIGINAL_GENERAL_LINES - GENERAL_LINES)) lines, $(( (ORIGINAL_GENERAL_LINES - GENERAL_LINES) * 100 / ORIGINAL_GENERAL_LINES ))%)"
echo "  Cangjie Agent:      ${CANGJIE_LINES} lines (was ${ORIGINAL_CANGJIE_LINES}, reduced $((ORIGINAL_CANGJIE_LINES - CANGJIE_LINES)) lines, $(( (ORIGINAL_CANGJIE_LINES - CANGJIE_LINES) * 100 / ORIGINAL_CANGJIE_LINES ))%)"
echo "  Total: $((GENERAL_LINES + CANGJIE_LINES)) lines (was ${ORIGINAL_TOTAL_LINES}, reduced $((ORIGINAL_TOTAL_LINES - (GENERAL_LINES + CANGJIE_LINES))) lines, $(( (ORIGINAL_TOTAL_LINES - (GENERAL_LINES + CANGJIE_LINES)) * 100 / ORIGINAL_TOTAL_LINES ))%)"
echo ""

if [ $FAILED -eq 0 ]; then
    echo -e "${GREEN}=========================================="
    echo "All tests passed! ✓"
    echo "==========================================${NC}"
    exit 0
else
    echo -e "${RED}=========================================="
    echo "Some tests failed. Please review the output.${NC}"
    echo -e "${RED}==========================================${NC}"
    exit 1
fi

