package mill.javalib.errorprone

import mill.{T, Task}
import mill.api.Discover
import mill.javalib.JavaModule
import mill.testkit.{TestRootModule, UnitTester}
import os.Path
import utest.*
import mill.util.TokenReaders.*
object ErrorProneTests extends TestSuite {

  object noErrorProne extends TestRootModule with JavaModule {
    lazy val millDiscover = Discover[this.type]
  }
  object errorProne extends TestRootModule with JavaModule with ErrorProneModule {
    lazy val millDiscover = Discover[this.type]
  }
  object errorProneCustom extends TestRootModule with JavaModule with ErrorProneModule {
    override def errorProneOptions: T[Seq[String]] = Task {
      Seq("-XepAllErrorsAsWarnings")
    }
    lazy val millDiscover = Discover[this.type]
  }
  // Test module with ErrorProne 2.36.0+ which requires --should-stop=ifError=FLOW
  object errorProne236 extends TestRootModule with JavaModule with ErrorProneModule {
    override def errorProneVersion: T[String] = Task { "2.36.0" }
    override def errorProneOptions: T[Seq[String]] = Task {
      Seq("-XepAllErrorsAsWarnings")
    }
    lazy val millDiscover = Discover[this.type]
  }

  val testModuleSourcesPath: Path = os.Path(sys.env("MILL_TEST_RESOURCE_DIR")) / "errorprone"

  def tests = Tests {
    test("reference") {
      test("compile") {
        UnitTester(noErrorProne, testModuleSourcesPath).scoped { eval =>

          val res = eval(noErrorProne.compile)
          assert(res.isRight)
        }
      }
    }
    test("errorprone") {
      test("compileFail") {
        UnitTester(errorProne, testModuleSourcesPath).scoped { eval =>
          val res = eval(errorProne.compile)
          assert(res.isLeft)
        }
      }
      test("compileWarn") {
        UnitTester(errorProneCustom, testModuleSourcesPath).scoped { eval =>
          val Right(opts) = eval(errorProneCustom.mandatoryJavacOptions).runtimeChecked
          assert(opts.value.exists(_.contains("-XepAllErrorsAsWarnings")))
          val res = eval(errorProneCustom.compile)
          assert(res.isRight)
        }
      }
      test("shouldStopOption236") {
        // ErrorProne 2.36.0+ requires --should-stop=ifError=FLOW
        // See https://github.com/com-lihaoyi/mill/issues/4926
        UnitTester(errorProne236, testModuleSourcesPath).scoped { eval =>
          val Right(opts) = eval(errorProne236.mandatoryJavacOptions).runtimeChecked
          assert(opts.value.contains("--should-stop=ifError=FLOW"))
          val res = eval(errorProne236.compile)
          assert(res.isRight)
        }
      }
    }
  }
}
