package mill.tabcomplete

import mill.Task
import mill.api.{Cross, Discover, Module}
import mill.testkit.UnitTester
import mill.testkit.TestRootModule
import utest.*

import java.io.{ByteArrayOutputStream, PrintStream}
import scala.annotation.unused
import scala.collection.immutable.HashSet

object TabCompleteTests extends TestSuite {
  object mainModule extends TestRootModule {
    lazy val millDiscover = Discover[this.type]
    def task1(@unused argA: String = "", @unused argB2: Int = 0) = Task.Command { 123 }
    object foo extends Module
    object bar extends Module {
      def task2(
          @mainargs.arg(doc = "arg a 3 docs") @unused argA3: String,
          @mainargs.arg(doc = "arg b 4 docs") @unused argB4: Int
      ) = Task.Command { 456 }
      def taskPositional(
          @mainargs.arg(positional = true) @unused argA3: String,
          @mainargs.arg(positional = true) @unused argB4: Int
      ) = Task.Command { 456 }

    }
    object qux extends Cross[QuxModule](12, 34, 56)
    trait QuxModule extends Cross.Module[Int] {
      def task3 = Task { 789 }
    }
    object versioned extends Cross[VersionedModule]("2.12.21", "2.13.18", "3.5.0")
    trait VersionedModule extends Cross.Module[String] {
      def compile = Task { crossValue }
    }
  }
  override def tests: Tests = Tests {

    val outStream = new ByteArrayOutputStream()
    val errStream = new ByteArrayOutputStream()

    def evalComplete(s: String*) = {
      UnitTester(
        mainModule,
        null,
        outStream = new PrintStream(outStream),
        errStream = new PrintStream(errStream)
      ).scoped { tester =>
        os.write(tester.evaluator.workspace / "file1.txt", "")
        os.write(tester.evaluator.workspace / "file2.txt", "")
        os.write(tester.evaluator.workspace / "folder/file3.scala", "", createFolders = true)
        tester.evaluator.evaluate(Seq("mill.tabcomplete.TabCompleteModule/complete") ++ s).get
      }
      outStream.toString.linesIterator.toSet
    }

    test("tasks") {
      test("empty-bash") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", ""),
          HashSet(
            "qux",
            "versioned",
            "file2.txt",
            "out",
            "folder",
            "bar",
            "file1.txt",
            "task1",
            "foo"
          )
        )
      }
      test("empty-zsh") {
        assertGoldenLiteral(
          evalComplete("1", "./mill"),
          HashSet(
            "qux",
            "versioned",
            "file2.txt",
            "out",
            "folder",
            "bar",
            "file1.txt",
            "task1",
            "foo"
          )
        )
      }
      test("task") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "t"),
          Set("task1")
        )
      }
      test("firstTask") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "t", "bar.task2"),
          Set("task1")
        )
      }

      test("secondNonTask") {
        assertGoldenLiteral(
          evalComplete("2", "./mill", "bar.task2", "f"),
          Set("file2.txt", "file1.txt", "folder")
        )
      }
      test("secondNonTaskEmpty") {
        assertGoldenLiteral(
          evalComplete("2", "./mill", "bar.task2", ""),
          Set("file2.txt", "file1.txt", "out", "folder")
        )
      }
      test("nestedNonTask") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "folder/"),
          Set("folder/file3.scala")
        )
      }
      test("nestedNonTask2") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "folder/f"),
          Set("folder/file3.scala")
        )
      }
      test("nestedScriptTask") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "folder/file3.scala:compile"),
          HashSet(
            "folder/file3.scala:compileMvnDeps                     Same as `mvnDeps`, but only present at compile time. Useful for e.g. macro-related dependencies like `scala-reflect` that doesn't need to be present at runtime",
            "folder/file3.scala:compileGeneratedSources            Path to sources generated as part of the `compile` step, eg.  by Java annotation processors which often generate source code alongside classfiles during compilation. Typically these do not need to be compiled again, and are only used by IDEs",
            "folder/file3.scala:compile                            Compiles the current module to generate compiled classfiles/bytecode. When you override this, you probably also want/need to override [[bspCompileClassesPath]], as that needs to point to the same compilation output path. Keep in sync with [[bspCompileClassesPath]]",
            "folder/file3.scala:compileResources                   Scripts default to having no compile-time resource folders. The folders where the compile time resource files for this module live. If your resources files do not necessarily need to be seen by the compiler, you should use [[resources]] instead.",
            "folder/file3.scala:compiledClassesAndSemanticDbFiles  Task containing the SemanticDB files used by VSCode and other IDEs to provide code insights. This is marked `persistent` so that when compilation fails, we keep the old semanticdb files around so IDEs can continue to analyze the code based on the metadata generated by the last successful compilation Keep the returned path to the compiled classes directory in sync with [[bspCompiledClassesAndSemanticDbFiles]].",
            "folder/file3.scala:compileClasspath                   [[compileClasspathTask]] for regular compilations. Keep return value in sync with [[bspCompileClasspath]].",
            "folder/file3.scala:compile0"
          )
        )
      }
      test("nestedScriptTaskAlternateSpelling") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "./folder/file3.scala:compile"),
          HashSet(
            "./folder/file3.scala:compileClasspath                   [[compileClasspathTask]] for regular compilations. Keep return value in sync with [[bspCompileClasspath]].",
            "./folder/file3.scala:compileResources                   Scripts default to having no compile-time resource folders. The folders where the compile time resource files for this module live. If your resources files do not necessarily need to be seen by the compiler, you should use [[resources]] instead.",
            "./folder/file3.scala:compileMvnDeps                     Same as `mvnDeps`, but only present at compile time. Useful for e.g. macro-related dependencies like `scala-reflect` that doesn't need to be present at runtime",
            "./folder/file3.scala:compiledClassesAndSemanticDbFiles  Task containing the SemanticDB files used by VSCode and other IDEs to provide code insights. This is marked `persistent` so that when compilation fails, we keep the old semanticdb files around so IDEs can continue to analyze the code based on the metadata generated by the last successful compilation Keep the returned path to the compiled classes directory in sync with [[bspCompiledClassesAndSemanticDbFiles]].",
            "./folder/file3.scala:compile                            Compiles the current module to generate compiled classfiles/bytecode. When you override this, you probably also want/need to override [[bspCompileClassesPath]], as that needs to point to the same compilation output path. Keep in sync with [[bspCompileClassesPath]]",
            "./folder/file3.scala:compile0",
            "./folder/file3.scala:compileGeneratedSources            Path to sources generated as part of the `compile` step, eg.  by Java annotation processors which often generate source code alongside classfiles during compilation. Typically these do not need to be compiled again, and are only used by IDEs"
          )
        )
      }
      test("nestedNonTask2") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "folder/doesntExist"),
          Set()
        )
      }
      test("externalModule") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "mill.tabcomplete.TabCompleteModule/"),
          Set(
            "mill.tabcomplete.TabCompleteModule/complete  The main entrypoint for Mill's Bash and Zsh tab-completion logic",
            "mill.tabcomplete.TabCompleteModule/install   Installs the Mill tab completion script globally and hooks it into `~/.zshrc` and `~/.bash_profile`. Can be passed an optional `--dest <path>` to instead write it to a manually-specified destination path"
          )
        )
      }
      test("externalModulePartial") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "mill.tabcomplete.TabCompleteModule/c"),
          Set(
            "mill.tabcomplete.TabCompleteModule/complete",
            "mill.tabcomplete.TabCompleteModule/complete: The main entrypoint for Mill's Bash and Zsh tab-completion logic"
          )
        )
      }
      test("externalModuleAlias") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "mill.tabcomplete/"),
          Set(
            "mill.tabcomplete/complete  The main entrypoint for Mill's Bash and Zsh tab-completion logic",
            "mill.tabcomplete/install   Installs the Mill tab completion script globally and hooks it into `~/.zshrc` and `~/.bash_profile`. Can be passed an optional `--dest <path>` to instead write it to a manually-specified destination path"
          )
        )
      }
      test("externalModuleAliasPartial") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "mill.tabcomplete/i"),
          Set(
            "mill.tabcomplete/install",
            "mill.tabcomplete/install: Installs the Mill tab completion script globally and hooks it into `~/.zshrc` and `~/.bash_profile`. Can be passed an optional `--dest <path>` to instead write it to a manually-specified destination path"
          )
        )
      }

      test("module") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "fo"),
          Set("foo", "folder")
        )
      }

      test("exactModule") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "bar"),
          Set("bar", "bar.task2", "bar.taskPositional")
        )
      }

      test("nested") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "bar."),
          Set("bar.task2", "bar.taskPositional")
        )
      }

      test("cross") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "qux["),
          Set("qux[12]", "qux[34]", "qux[56]")
        )
      }

      test("cross2") {
        // Without bracket, use new dot syntax
        assertGoldenLiteral(
          evalComplete("1", "./mill", "qux"),
          Set("qux", "qux.12", "qux.34", "qux.56")
        )
      }

      test("crossPartial") {
        // With bracket, keep bracket syntax
        assertGoldenLiteral(
          evalComplete("1", "./mill", "qux[1"),
          Set("qux[12]")
        )
      }

      test("crossNested") {
        // With bracket, keep bracket syntax
        assertGoldenLiteral(
          evalComplete("1", "./mill", "qux[12]"),
          Set("qux[12].task3")
        )
      }

      test("crossNestedDotSyntax") {
        // Without bracket, use new dot syntax
        assertGoldenLiteral(
          evalComplete("1", "./mill", "qux.12"),
          Set("qux.12", "qux.12.task3")
        )
      }

      test("crossNestedSlashed") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "qux\\[12\\]"),
          Set("qux[12].task3")
        )
      }
      test("crossNestedSingleQuoted") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "'qux[12]"),
          Set("qux[12].task3")
        )
      }
      test("crossNestedDoubleQuoted") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "\"qux[12]"),
          Set("qux[12].task3")
        )
      }

      test("crossComplete") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "qux[12].task3"),
          Set("qux[12].task3")
        )
      }

      test("crossVersioned") {
        // Version numbers with dots are converted to underscores in dot syntax
        assertGoldenLiteral(
          evalComplete("1", "./mill", "versioned"),
          Set("versioned", "versioned.2_12_21", "versioned.2_13_18", "versioned.3_5_0")
        )
      }

      test("crossVersionedNested") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "versioned.2_12_21"),
          Set("versioned.2_12_21", "versioned.2_12_21.compile")
        )
      }

      test("crossVersionedBracket") {
        // With bracket, keep original syntax with dots
        assertGoldenLiteral(
          evalComplete("1", "./mill", "versioned[2"),
          Set("versioned[2.12.21]", "versioned[2.13.18]")
        )
      }
    }
    test("flags") {
      test("short") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "-"),
          Set(
            "-i  Alias for now `--no-daemon`. No longer needed for interactive commands since Mill 1.1.0",
            "-b  Ring the bell once if the run completes successfully, twice if it fails.",
            "-w  Watch and re-run the given tasks when when their inputs change.",
            "-k  Continue build, even after build failures.",
            "-D  <k=v> Define (or overwrite) a system property.",
            "-d  Show debug output on STDOUT",
            "-v  Show mill version information and exit.",
            "-j  <str> The number of parallel threads. It can be an integer e.g. `5` meaning 5 threads, an expression e.g. `0.5C` meaning half as many threads as available cores, or `C-2` meaning 2 threads less than the number of cores. `1` disables parallelism and `0` (the default) uses 1 thread per core."
          )
        )
      }
      test("emptyAfterFlag") {
        assertGoldenLiteral(
          evalComplete("2", "./mill", "-v"),
          HashSet(
            "qux",
            "versioned",
            "file2.txt",
            "out",
            "folder",
            "bar",
            "file1.txt",
            "task1",
            "foo"
          )
        )

      }
      test("filterAfterFlag") {
        assertGoldenLiteral(
          evalComplete("2", "./mill", "-v", "f"),
          Set("foo", "file2.txt", "file1.txt", "folder")
        )
      }
      test("filterAfterFlagAfterTask") {
        assertGoldenLiteral(
          evalComplete("3", "./mill", "-v", "task1", "f"),
          Set("file2.txt", "file1.txt", "folder")
        )
      }

      test("long") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "--"),
          Set(
            "--use-file-locks          Use traditional file-based locking instead of PID-based locking for the Mill daemon. This removes the chance of race conditions when claiming the lock after a crash, but may have issues on some filesystems that do not support lock (e.g. docker mounts on mac)",
            "--debug                   Show debug output on STDOUT",
            "--bell                    Ring the bell once if the run completes successfully, twice if it fails.",
            "--no-build-lock           Evaluate tasks / commands without acquiring an exclusive lock on the Mill output directory",
            "--bsp-install             Create mill-bsp.json with Mill details under .bsp/",
            "--allow-positional        Allows command args to be passed positionally without `--arg` by default",
            "--watch                   Watch and re-run the given tasks when when their inputs change.",
            "--bsp                     Enable BSP server mode. Typically used by a BSP client when starting the Mill BSP server.",
            "--help-advanced           Print a internal or advanced command flags not intended for common usage",
            "--tab-complete            Runs Mill in tab-completion mode",
            "--interactive             Alias for now `--no-daemon`. No longer needed for interactive commands since Mill 1.1.0",
            "--import                  <str> Additional ivy dependencies to load into mill, e.g. plugins.",
            "--meta-level              <int> Select a meta-level to run the given tasks. Level 0 is the main project in `build.mill`, level 1 the first meta-build in `mill-build/build.mill`, etc. If negative, -1 means the deepest meta-build (boostrap build), -2 the second deepest meta-build, etc.",
            "--offline                 Try to work offline. This tells modules that support it to work offline and avoid any access to the internet. This is on a best effort basis. There are currently no guarantees that modules don't attempt to fetch remote sources.",
            "--keep-going              Continue build, even after build failures.",
            "--define                  <k=v> Define (or overwrite) a system property.",
            "--no-filesystem-checker   Globally disables the checks that prevent you from reading and writing to disallowed files or folders during evaluation. Useful as an escape hatch in case you desperately need to do something unusual and you are willing to take the risk",
            "--notify-watch            <bool> Use filesystem based file watching instead of polling based one (defaults to true).",
            "--bsp-watch               <bool> Automatically reload the build when its sources change when running the BSP server (defaults to true).",
            "--help                    Print this help message and exit.",
            "--jobs                    <str> The number of parallel threads. It can be an integer e.g. `5` meaning 5 threads, an expression e.g. `0.5C` meaning half as many threads as available cores, or `C-2` meaning 2 threads less than the number of cores. `1` disables parallelism and `0` (the default) uses 1 thread per core.",
            "--ticker                  <bool> Enable or disable the ticker log, which provides information on running tasks and where each log line came from",
            "--color                   <bool> Toggle colored output; by default enabled only if the console is interactive or FORCE_COLOR environment variable is set, and NO_COLOR is not set",
            "--no-daemon               Run without a long-lived background daemon.",
            "--no-wait-for-build-lock  Do not wait for an exclusive lock on the Mill output directory to evaluate tasks / commands.",
            "--version                 Show mill version information and exit.",
            "--task                    <str> The name or a query of the tasks(s) you want to build."
          )
        )
      }
      test("longflagsfiltered") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "--h"),
          Set(
            "--help           Print this help message and exit.",
            "--help-advanced  Print a internal or advanced command flags not intended for common usage"
          )
        )
      }
      test("longFlagBrokenEarlier") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "--jo", "1", "task1"),
          Set(
            "--jobs",
            "--jobs: <str> The number of parallel threads. It can be an integer e.g. `5` meaning 5 threads, an expression e.g. `0.5C` meaning half as many threads as available cores, or `C-2` meaning 2 threads less than the number of cores. `1` disables parallelism and `0` (the default) uses 1 thread per core."
          )
        )
      }
      test("longFlagCompleteEarlier") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "--jobs", "1", "task1"),
          Set(
            "--jobs",
            "--jobs: <str> The number of parallel threads. It can be an integer e.g. `5` meaning 5 threads, an expression e.g. `0.5C` meaning half as many threads as available cores, or `C-2` meaning 2 threads less than the number of cores. `1` disables parallelism and `0` (the default) uses 1 thread per core."
          )
        )
      }
      test("longFlagIncomplete") {
        assertGoldenLiteral(
          evalComplete("1", "./mill", "--jobs"),
          Set(
            "--jobs",
            "--jobs: <str> The number of parallel threads. It can be an integer e.g. `5` meaning 5 threads, an expression e.g. `0.5C` meaning half as many threads as available cores, or `C-2` meaning 2 threads less than the number of cores. `1` disables parallelism and `0` (the default) uses 1 thread per core."
          )
        )
      }
    }
    test("commandflags") {
      test("required") {
        test {
          assertGoldenLiteral(
            evalComplete("2", "./mill", "task1", "--a"),
            Set("--arg-a    <str> ", "--arg-b-2  <int> ")
          )
        }
        test {
          assertGoldenLiteral(
            evalComplete("2", "./mill", "task1", "--arg-b"),
            Set("--arg-b-2", "--arg-b-2: <int> ")
          )
        }
      }
      test("positional") {
        test {
          assertGoldenLiteral(
            evalComplete("2", "./mill", "bar.taskPositional", "--"),
            Set()
          )
        }
        test {
          assertGoldenLiteral(
            evalComplete("2", "./mill", "bar.taskPositional", ""),
            Set("file2.txt", "file1.txt", "out", "folder")
          )
        }
      }
      test("optional") {
        test {
          assertGoldenLiteral(
            evalComplete("2", "./mill", "bar.task2", "--a"),
            Set("--arg-a-3  <str> arg a 3 docs", "--arg-b-4  <int> arg b 4 docs")
          )
        }
        test {
          assertGoldenLiteral(
            evalComplete("2", "./mill", "bar.task2", "--arg-b"),
            Set("--arg-b-4", "--arg-b-4: <int> arg b 4 docs")
          )
        }

        test {
          assertGoldenLiteral(
            evalComplete("3", "./mill", "bar.task2", "--arg-a", "--"),
            Set()
          )
        }
        test {
          assertGoldenLiteral(
            evalComplete("3", "./mill", "bar.task2", "--arg-a", ""),
            Set("file2.txt", "file1.txt", "out", "folder")
          )
        }
        test {
          assertGoldenLiteral(
            evalComplete("5", "./mill", "bar.task2", "--arg-a", "", "--arg-b", ""),
            Set("file2.txt", "file1.txt", "out", "folder")
          )
        }
        test {
          assertGoldenLiteral(
            evalComplete("3", "./mill", "bar.task2", "--arg-a", "", "--arg-b", ""),
            Set("file2.txt", "file1.txt", "out", "folder")
          )
        }
      }
    }
  }
}
