/*
 * Decompiled with CFR 0.152.
 */
package com.hbm.dev;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.Reader;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.world.item.Item;
import net.minecraft.world.level.block.Block;
import net.minecraftforge.fml.loading.FMLPaths;
import net.minecraftforge.registries.ForgeRegistries;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public final class AssetConsistencyChecker {
    private static final Logger LOGGER = LogManager.getLogger((String)"HBM-AssetChecker");
    private static final String PROPERTY = "hbm.assetCheck";

    private AssetConsistencyChecker() {
    }

    public static void runIfRequested() {
        if (!Boolean.parseBoolean(System.getProperty(PROPERTY, "false"))) {
            return;
        }
        LOGGER.info("HBM asset checker enabled via -D{}=true", (Object)PROPERTY);
        try {
            AssetConsistencyChecker.runCheck();
        }
        catch (Exception ex) {
            LOGGER.error("HBM asset checker failed", (Throwable)ex);
        }
    }

    private static void runCheck() throws IOException {
        List<Path> assetRoots = AssetConsistencyChecker.discoverAssetRoots();
        if (assetRoots.isEmpty()) {
            LOGGER.warn("No asset roots discovered; skipped asset consistency check.");
            return;
        }
        Report report = new Report();
        AssetConsistencyChecker.checkItems(assetRoots, report);
        AssetConsistencyChecker.checkBlocks(assetRoots, report);
        report.dump();
        report.writeToFile(FMLPaths.GAMEDIR.get().resolve("asset-report.txt"));
    }

    private static List<Path> discoverAssetRoots() {
        Path runtimeAssets;
        Path generatedAssets;
        Path gameDir = FMLPaths.GAMEDIR.get();
        ArrayList<Path> roots = new ArrayList<Path>();
        Path mainAssets = gameDir.resolve("src/main/resources/assets/hbm");
        if (Files.isDirectory(mainAssets, new LinkOption[0])) {
            roots.add(mainAssets);
        }
        if (Files.isDirectory(generatedAssets = gameDir.resolve("src/generated/resources/assets/hbm"), new LinkOption[0])) {
            roots.add(generatedAssets);
        }
        if (Files.isDirectory(runtimeAssets = gameDir.resolve("assets/hbm"), new LinkOption[0])) {
            roots.add(runtimeAssets);
        }
        LOGGER.info("HBM asset checker roots: {}", roots);
        return roots;
    }

    private static void checkItems(List<Path> roots, Report report) {
        for (Item item : ForgeRegistries.ITEMS) {
            ResourceLocation id = ForgeRegistries.ITEMS.getKey((Object)item);
            if (id == null || !"hbm".equals(id.m_135827_())) continue;
            String name = id.m_135815_();
            Path modelPath = AssetConsistencyChecker.findAsset(roots, Path.of("models", "item", name + ".json"));
            if (modelPath == null) {
                report.missingItemModels.add(name);
                continue;
            }
            JsonObject modelJson = AssetConsistencyChecker.readJson(modelPath, report);
            if (modelJson == null) continue;
            JsonObject texturesObj = modelJson.getAsJsonObject("textures");
            if (texturesObj == null || !texturesObj.has("layer0")) {
                report.itemModelMissingTexture.add(name + " -> layer0");
                continue;
            }
            String texture = texturesObj.get("layer0").getAsString();
            Path texturePath = AssetConsistencyChecker.resolveTexturePath(texture, "item");
            if (texturePath != null && AssetConsistencyChecker.findAsset(roots, texturePath) != null) continue;
            report.missingItemTextures.add(name + " -> " + texture);
        }
    }

    private static void checkBlocks(List<Path> roots, Report report) {
        for (Block block : ForgeRegistries.BLOCKS) {
            ResourceLocation id = ForgeRegistries.BLOCKS.getKey((Object)block);
            if (id == null || !"hbm".equals(id.m_135827_())) continue;
            String name = id.m_135815_();
            Path blockstatePath = AssetConsistencyChecker.findAsset(roots, Path.of("blockstates", name + ".json"));
            if (blockstatePath == null) {
                report.missingBlockstates.add(name);
                continue;
            }
            JsonObject blockstateJson = AssetConsistencyChecker.readJson(blockstatePath, report);
            if (blockstateJson == null) continue;
            Set<String> models = AssetConsistencyChecker.collectModelReferences((JsonElement)blockstateJson);
            if (models.isEmpty()) {
                report.blockstateMissingModels.add(name);
                continue;
            }
            for (String model : models) {
                JsonObject texturesObj;
                Path modelPath = AssetConsistencyChecker.resolveModelPath(model);
                if (modelPath == null) {
                    report.invalidBlockModelRefs.add(name + " -> " + model);
                    continue;
                }
                Path absoluteModel = AssetConsistencyChecker.findAsset(roots, modelPath);
                if (absoluteModel == null) {
                    report.missingBlockModels.add(model);
                    continue;
                }
                JsonObject modelJson = AssetConsistencyChecker.readJson(absoluteModel, report);
                if (modelJson == null || (texturesObj = modelJson.getAsJsonObject("textures")) == null) continue;
                for (Map.Entry entry : texturesObj.entrySet()) {
                    Path texturePath;
                    String texture;
                    if (!((JsonElement)entry.getValue()).isJsonPrimitive() || !(texture = ((JsonElement)entry.getValue()).getAsString()).startsWith("hbm:") || (texturePath = AssetConsistencyChecker.resolveTexturePath(texture, "block")) != null && AssetConsistencyChecker.findAsset(roots, texturePath) != null) continue;
                    report.missingBlockTextures.add(model + " -> " + texture);
                }
            }
        }
    }

    private static Set<String> collectModelReferences(JsonElement element) {
        HashSet<String> models;
        block5: {
            block4: {
                models = new HashSet<String>();
                if (element == null) {
                    return models;
                }
                if (!element.isJsonObject()) break block4;
                JsonObject obj = element.getAsJsonObject();
                if (obj.has("model") && obj.get("model").isJsonPrimitive()) {
                    models.add(obj.get("model").getAsString());
                }
                for (Map.Entry entry : obj.entrySet()) {
                    models.addAll(AssetConsistencyChecker.collectModelReferences((JsonElement)entry.getValue()));
                }
                break block5;
            }
            if (!element.isJsonArray()) break block5;
            for (JsonElement child : element.getAsJsonArray()) {
                models.addAll(AssetConsistencyChecker.collectModelReferences(child));
            }
        }
        return models;
    }

    private static JsonObject readJson(Path path, Report report) {
        JsonObject jsonObject;
        block8: {
            BufferedReader reader = Files.newBufferedReader(path);
            try {
                jsonObject = JsonParser.parseReader((Reader)reader).getAsJsonObject();
                if (reader == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (reader != null) {
                        try {
                            reader.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (Exception ex) {
                    report.failedToReadFiles.add(path.toString() + " -> " + ex.getMessage());
                    return null;
                }
            }
            reader.close();
        }
        return jsonObject;
    }

    private static Path resolveModelPath(String namespacePath) {
        if (!namespacePath.startsWith("hbm:")) {
            return null;
        }
        String value = namespacePath.substring("hbm".length() + 1);
        if (!value.startsWith("block/")) {
            return null;
        }
        return Path.of("models", value + ".json");
    }

    private static Path resolveTexturePath(String namespacePath, String expectedFolder) {
        if (!namespacePath.startsWith("hbm:")) {
            return null;
        }
        String value = namespacePath.substring("hbm".length() + 1);
        if (!value.startsWith(expectedFolder + "/")) {
            return null;
        }
        return Path.of("textures", value + ".png");
    }

    private static Path findAsset(List<Path> roots, Path relative) {
        for (Path root : roots) {
            Path candidate = root.resolve(relative);
            if (!Files.exists(candidate, new LinkOption[0])) continue;
            return candidate;
        }
        return null;
    }

    private static final class Report {
        private final List<String> missingItemModels = new ArrayList<String>();
        private final List<String> itemModelMissingTexture = new ArrayList<String>();
        private final List<String> missingItemTextures = new ArrayList<String>();
        private final List<String> missingBlockstates = new ArrayList<String>();
        private final List<String> blockstateMissingModels = new ArrayList<String>();
        private final List<String> invalidBlockModelRefs = new ArrayList<String>();
        private final List<String> missingBlockModels = new ArrayList<String>();
        private final List<String> missingBlockTextures = new ArrayList<String>();
        private final List<String> failedToReadFiles = new ArrayList<String>();

        private Report() {
        }

        private void dump() {
            if (this.isClean()) {
                LOGGER.info("HBM asset checker: no inconsistencies found.");
                return;
            }
            this.logList("Missing item models", this.missingItemModels);
            this.logList("Item models missing textures", this.itemModelMissingTexture);
            this.logList("Missing item textures", this.missingItemTextures);
            this.logList("Missing blockstates", this.missingBlockstates);
            this.logList("Blockstates without model references", this.blockstateMissingModels);
            this.logList("Blockstates referencing invalid models", this.invalidBlockModelRefs);
            this.logList("Missing block models", this.missingBlockModels);
            this.logList("Missing block textures", this.missingBlockTextures);
            this.logList("Failed to read JSON files", this.failedToReadFiles);
        }

        private void logList(String title, List<String> entries) {
            if (entries.isEmpty()) {
                return;
            }
            LOGGER.warn("{} ({}):", (Object)title, (Object)entries.size());
            entries.stream().sorted().forEach(entry -> LOGGER.warn(" -> {}", entry));
        }

        private boolean isClean() {
            return this.missingItemModels.isEmpty() && this.itemModelMissingTexture.isEmpty() && this.missingItemTextures.isEmpty() && this.missingBlockstates.isEmpty() && this.blockstateMissingModels.isEmpty() && this.invalidBlockModelRefs.isEmpty() && this.missingBlockModels.isEmpty() && this.missingBlockTextures.isEmpty() && this.failedToReadFiles.isEmpty();
        }

        private void writeToFile(Path output) throws IOException {
            ArrayList<String> lines = new ArrayList<String>();
            lines.add("HBM Asset Consistency Report");
            lines.add("----------------------------");
            lines.add("");
            this.appendSection(lines, "Missing item models", this.missingItemModels);
            this.appendSection(lines, "Item models missing textures", this.itemModelMissingTexture);
            this.appendSection(lines, "Missing item textures", this.missingItemTextures);
            this.appendSection(lines, "Missing blockstates", this.missingBlockstates);
            this.appendSection(lines, "Blockstates without model references", this.blockstateMissingModels);
            this.appendSection(lines, "Blockstates referencing invalid models", this.invalidBlockModelRefs);
            this.appendSection(lines, "Missing block models", this.missingBlockModels);
            this.appendSection(lines, "Missing block textures", this.missingBlockTextures);
            this.appendSection(lines, "Failed to read JSON files", this.failedToReadFiles);
            Files.write(output, lines, new OpenOption[0]);
            LOGGER.info("HBM asset checker wrote {}", (Object)output.toAbsolutePath());
        }

        private void appendSection(List<String> lines, String title, List<String> entries) {
            lines.add(title + " (" + entries.size() + ")");
            if (entries.isEmpty()) {
                lines.add("  OK");
            } else {
                lines.addAll(entries.stream().sorted().map(s -> "  - " + s).collect(Collectors.toList()));
            }
            lines.add("");
        }
    }
}

