@tool
class_name FilterableItemList extends VBoxContainer

signal item_activated(text: String, metadata: Variant)
signal item_selected(text: String, metadata: Variant)

@export var filter_text: String:
	get:
		return _filter_text
	set(new_filter_text):
		_filter_text = new_filter_text
		_refresh_filter_text()

@onready var _filter: LineEdit = $Filter
@onready var _item_list: ItemList = $ItemList

var _item_filtered: Array[int]
var _filter_text: String
var _item_source: Object
var _item_source_icon: bool
var _item_source_metadata: bool
var _item_source_tooltip: bool


func _ready() -> void:
	_filter.right_icon = EditorInterface.get_editor_theme().get_icon("Search", "EditorIcons")
	_refresh_filter_text()


func _refresh_filter_text() -> void:
	if is_node_ready():
		_filter.placeholder_text = _filter_text


func get_selected_item_text() -> String:
	var index := _get_selected_item_index()
	if index < 0:
		return ""
	return _item_list.get_item_text(index)


func get_selected_item_metadata() -> Variant:
	var index := _get_selected_item_index()
	if index < 0:
		return null
	return _item_list.get_item_metadata(index)


func is_anything_selected() -> bool:
	return _item_list.is_anything_selected()


func select_item_by_text(text: String) -> bool:
	return refresh_items({"text": text})


func select_item_by_metadata(metadata: Variant) -> bool:
	return refresh_items({"metadata": metadata})


func set_item_source(item_source: Object) -> bool:
	if item_source is ProxyItemSource:
		if !_check_method(item_source.source, "get_item_count", item_source.method_get_item_count):
			return false
		if !_check_method(item_source.source, "get_item_text", item_source.method_get_item_text):
			return false
		_item_source = item_source
		_item_source_icon = _has_method(item_source.source, item_source.method_get_item_icon)
		_item_source_metadata = _has_method(item_source.source, item_source.method_get_item_metadata)
		_item_source_tooltip = _has_method(item_source.source, item_source.method_get_item_tooltip)
		refresh_items()
		return true

	if !_check_method(item_source, "get_item_count", "get_item_count"):
		return false
	if !_check_method(item_source, "get_item_text", "get_item_text"):
		return false
	_item_source = item_source
	_item_source_icon = _has_method(item_source, "get_item_icon")
	_item_source_metadata = _has_method(item_source, "get_item_metadata")
	_item_source_tooltip = _has_method(item_source, "get_item_tooltip")
	refresh_items()
	return true


func refresh_items(select: Dictionary = {}) -> bool:
	if _item_source == null:
		_item_list.clear()
		return false

	# set or preserve selected item
	var selected_source_i = -1
	if select.size() > 0:
		var key = select.keys()[0]
		var value = select.get(key)
		match key:
			"text":
				for i in _item_source.get_item_count():
					var text: String = _item_source.get_item_text(i)
					if text == value:
						selected_source_i = i
			"metadata":
				for i in _item_source.get_item_count():
					var metadata: Variant = _item_source.get_item_metadata(i)
					if metadata == value:
						selected_source_i = i
	else:
		var index = _get_selected_item_index()
		if index >= 0:
			selected_source_i = _item_filtered[index]

	# apply filter
	var selected_i = -1
	_item_filtered = []
	var filter = _filter.text
	var item_count_new: int = _item_source.get_item_count()
	if filter == "":
		for i in item_count_new:
			_item_filtered.append(i)
		# if unfiltered, item index is source index
		selected_i = selected_source_i
	else:
		for i in item_count_new:
			var text: String = _item_source.get_item_text(i)
			var selected: bool = i == selected_source_i
			if selected || text.contains(filter):
				if selected:
					selected_i = _item_filtered.size()
				_item_filtered.append(i)

	# materialize filtered items
	item_count_new = _item_filtered.size()
	var item_count_old: int = _item_list.get_item_count()
	for i in item_count_new:
		var index = _item_filtered[i]
		var text: String = _item_source.get_item_text(index)
		if i >= item_count_old:
			_item_list.add_item(text)
		else:
			_item_list.set_item_text(i, text)
		_item_list.set_item_icon(i, _item_source.get_item_icon(index) if _item_source_icon else null)
		var metadata: Variant = _item_source.get_item_metadata(index) if _item_source_metadata else null
		_item_list.set_item_metadata(i, metadata)
		var tooltip: String = _item_source.get_item_tooltip(index) if _item_source_tooltip else ""
		_item_list.set_item_tooltip(i, tooltip)
		_item_list.set_item_tooltip_enabled(i, tooltip != "")
		# restore selected item
		if selected_i == i:
			_item_list.select(i)

	while item_count_new < _item_list.get_item_count():
		_item_list.remove_item(_item_list.get_item_count()-1)

	return selected_i >= 0


func _get_selected_item_index() -> int:
	var selected_items := _item_list.get_selected_items()
	if selected_items.size() == 0:
		return -1
	return selected_items[0]


func _on_focus_entered() -> void:
	_filter.grab_focus.call_deferred()


func _on_filter_text_changed(_new_text: String) -> void:
	refresh_items()


func _on_item_list_item_activated(index: int) -> void:
	var text := _item_list.get_item_text(_item_filtered[index])
	var metadata := _item_list.get_item_metadata(_item_filtered[index])
	item_activated.emit(text, metadata)


func _on_item_list_item_selected(index: int) -> void:
	var text := _item_list.get_item_text(_item_filtered[index])
	var metadata := _item_list.get_item_metadata(_item_filtered[index])
	item_selected.emit(text, metadata)


static func _check_method(item_source: Object, as_method: String, method: String) -> bool:
	if method == "":
		push_error("FilterableItemList.set_item_source: invalid item source %s: missing %s method name" % [item_source, as_method])
		return false

	if !item_source.has_method(method):
		if as_method == method:
			push_error("FilterableItemList.set_item_source: invalid item source %s: missing %s method" % [item_source, as_method])
		else:
			push_error("FilterableItemList.set_item_source: invalid item source %s: missing %s method: %s" % [item_source, as_method, method])
		return false

	return true


static func _has_method(item_source: Object, method: String) -> bool:
	if method == "":
		return false

	return item_source.has_method(method)
